<?php
class Streaming extends CFormModel {

    const FILE = '/etc/raperca/streaming.json';
    const XML_FILE = '/etc/raperca/streaming.xml';

    const DISABLED = 'Disabled';
    const IPTV = 'IPTV';
    const RTSP = 'RTSP';
    const RTMP = 'RTMP';
    const CUSTOM = 'Custom';

    const HIGH = 'Best';
    const MEDIUM = 'Default';
    const LOW = 'Good';

    const MP2 = 'MP2'; // MPEG-1/2 Audio Layer II
    const AC3 = 'AC3';
    const AAC = 'AAC';

    const AUDIO_VIDEO_END_POINT = '/audio+video';
    const VIDEO_END_POINT = '/video';

    public function attributeNames() {
        $names = parent::attributeNames();
        $names[] = 'RTP';
        $names[] = 'multicast';
        return $names;
    }

    public function attributeLabels() {
        return array(
            'protocol' => 'Protocol',
            'resolution' => 'Resolution',
            'frameRate' => 'Frame Rate',
            'quality' => 'Quality',
            'codec' => 'Audio Codec',
            'IP' => 'Destination IP',
            'port' => 'Port',
            'RTP' => 'Use RTP Headers',
            'multicast' => 'Enable Multicast',
            'group' => 'Multicast Group',
            'serverURL' => 'Server URL',
            'URL' => 'Destination URL',
        );
    }

    public function rules() {
        return array(
            array(
                'protocol', 'in', 'allowEmpty' => false, 'range' => 
                $this->_protocols , 'strict' => true),

            array(
                'resolution', 'match', 'on' => array(self::IPTV, self::RTSP, 
                self::RTMP), 'allowEmpty' => false, 'pattern' => 
                '/^[1-9]{1}\d*x[1-9]{1}\d*$/'),
            array(
                'frameRate', 'numerical', 'on' => array(self::IPTV, 
                self::RTSP, self::RTMP), 'allowEmpty' => false, 'integerOnly' 
                => true, 'min' => 1),
            array(
                'quality', 'in', 'on' => array(self::IPTV, self::RTSP, 
                self::RTMP), 'allowEmpty' => false, 'range' => 
                $this->_qualities, 'strict' => true),

            array(
                'codec', 'in', 'on' => self::IPTV, 'allowEmpty' => false, 
                'range' => $this->_codecs, 'strict' => true),
            array(
                'IP', 'checkIP', 'on' => self::IPTV, 'allowEmpty' => false),
            array(
                'port', 'numerical', 'on' => self::IPTV, 
                'allowEmpty' => false, 
                'integerOnly' => true, 
                'integerPattern' => '/^\d*[02468]$/', 
                'min' => 1, 'max' => 65535, 
                'message' => '{attribute} must be an even integer.'),
            array(
                'RTP', 'boolean', 'on' => self::IPTV, 'allowEmpty' => false),

            array(
                'multicast', 'boolean', 'on' => self::RTSP, 'allowEmpty' => 
                false),
            array('group', 'validateMulticast', 'on' => self::RTSP),
    
            array(
                'URL', 'SimpleURLValidator', 'on' => self::RTMP, 'allowEmpty' 
                => false, 'validSchemes' => array('rtmp', 'rtmps')),
    	);
    }

    public function validateMulticast($attribute, $params) {
        if ($this->multicast && !self::isMulticast($this->$attribute)) {
            $this->addError($attribute, 'Invalid multicast address.');
        }
    }
    
    public function load() {
        $basicInfo = new BasicInfo();
        $this->_serverURL = 
            'rtsp://' . $basicInfo->getHostname() . 
            self::AUDIO_VIDEO_END_POINT;
        if (!file_exists(self::FILE)) {
            return;
        }
        $values = json_decode(file_get_contents(self::FILE), true);
        foreach ($values as $name => $value) {
            $this->$name = $value;
        }
        if ($this->protocol == self::CUSTOM) {
            if (!($XML = file_get_contents(self::XML_FILE))) {
                return;
            };
            $this->_XML = $XML;
        }
    }

    public function save() {
        if ($this->protocol == self::DISABLED) {
            if (file_exists(self::FILE)) {
                unlink(self::FILE);
            }
            if (file_exists(self::XML_FILE)) {
                unlink(self::XML_FILE);
            }
        } else {
            if ($this->protocol == self::CUSTOM) {
                $values = array ('protocol' => self::CUSTOM);
                $XML = $this->getXML();;
            } else {
                $values = $this->getAttributes();
                $XML = $this->createXML();
            }
            Tools::save_file(self::FILE, json_encode($values));
            Tools::save_file(self::XML_FILE, $XML);
        }
    }

    public function getAttributes($names = null) {
        if (!is_array($names)) {
            $this->setScenario($this->protocol);
            $names = $this->getSafeAttributeNames();
        }
        return parent::getAttributes($names);
    }

    public function setAttributes($values, $safeOnly = true) {
        parent::setAttributes($values, $safeOnly);
        if (!$this->validate()) {
            return false;
        }          
        $this->setScenario($this->protocol);
        parent::setAttributes($values, $safeOnly);
        return $this->validate();
    }

    public function getRTP() {
        return $this->_RTP;
    }

    public function setRTP($RTP) {
        $this->_RTP = intval($RTP);
    }

    public function getMulticast() {
        return $this->_multicast;
    }

    public function setMulticast($multicast) {
        $this->_multicast = intval($multicast);
    }

    public function getBitRate() {
        return $this->_bitRate;
    }

    public function getServerURL() {
       return $this->_serverURL;
    }

    public function getXML() {
        return $this->_XML;
    }

    public $protocol = self::DISABLED;
    public $resolution = '';
    public $frameRate = '';
    public $quality = self::MEDIUM;
    public $codec = self::MP2;
    public $IP = '';
    public $port = '';
    public $group = '239.255.0.1';
    public $URL = '';

    protected function createXML() {
        $this->prepareVideoAttributes();
        $doc = new DOMDocument('1.0', 'UTF-8');
        $streaming = $doc->appendChild($doc->createElement('streaming'));
        if ($this->protocol == self::IPTV) {
            $iptv = $streaming->appendChild($doc->createElement('iptv'));
            $iptv->setAttribute('name', 'iptv');
            $this->setVideoAttributes($iptv);
            $iptv->setAttribute('audioCodec', $this->codec);
            $iptv->setAttribute('rtpHeaders', $this->RTP ? 'on' : 'off');
            $destination = $iptv->appendChild(
                $doc->createElement('destination'));
            $destination->setAttribute('ip', $this->IP);
            $destination->setAttribute('port', $this->port);
            if (self::isMulticast($this->IP)) {
                $destination->setAttribute('ttl', '4');
            }
        } elseif ($this->protocol == self::RTSP) {
            $rtsp = $streaming->appendChild($doc->createElement('rtsp'));
            $rtsp->setAttribute('portNumber', '554');
            $this->addRTSPEndPoint($doc, $rtsp, true);
            $this->addRTSPEndPoint($doc, $rtsp, false);
        } elseif ($this->protocol == self::RTMP) {
            $rtmp = $streaming->appendChild($doc->createElement('rtmp'));
            $rtmp->setAttribute('url', $this->URL);
            $rtmp->setAttribute(
                'flashVersion', 'FME/3.0%20(compatible;%20FMSc%201.0)');
            $this->setVideoAttributes($rtmp);
            $rtmp->setAttribute('audioCodec', 'AAC');
        }
        return $doc->saveXML();
    }

    protected function prepareVideoAttributes() {
        list($width, $height) = explode('x', $this->resolution);
        $this->size = max($width, $height);
        $bitRate = $width * $height * log(
            $this->frameRate > 2 ? $this->frameRate : 2);
        if ($this->quality == self::HIGH) {
            $bitRate *= 1.3;
        } elseif ($this->quality == self::LOW) {
            $bitRate *= 0.7;
        }
        $this->_bitRate = floor($bitRate); 
    }

    protected function setVideoAttributes($element) {
        $element->setAttribute('videoCodec', 'H264');
        $element->setAttribute('videoSize', $this->size);
        $element->setAttribute('videoFPS', $this->frameRate);
        $element->setAttribute('bitRate', $this->getBitRate()); 
    }

    protected static function isMulticast($IP) {
        $IP = inet_pton($IP);
        if (strlen($IP) == 4) {
            return $IP >= inet_pton('224.0.0.0') && $IP <= inet_pton(
                '239.255.255.255') ? true : false;
        } elseif (strlen($IP) == 16) {
            return $IP >= inet_pton('ff00::') ? true: false;
        } else {
            return false;
        }
    }

    protected function addRTSPEndPoint($doc, $rtsp, $audio) {
        $endPoint = $rtsp->appendChild($doc->createElement('endPoint'));
        $endPoint->setAttribute(
            'path', $audio ? self::AUDIO_VIDEO_END_POINT : 
            self::VIDEO_END_POINT);
        $this->setVideoAttributes($endPoint);
        $endPoint->setAttribute('audioCodec', $audio ? 'AAC' : 'none');
        $range = $endPoint->appendChild($doc->createElement('range'));
        $range->setAttribute('startIP', '0.0.0.0');
        $range->setAttribute('endIP', '0.0.0.0');
        $range->setAttribute('startPort', '5000');
        $range->setAttribute('endPort', '6000');
        $range = $endPoint->appendChild($doc->createElement('range'));
        $range->setAttribute('startIP', '::');
        $range->setAttribute('endIP', '::');
        $range->setAttribute('startPort', '5000');
        $range->setAttribute('endPort', '6000');
        if ($this->multicast) {
            $range = $endPoint->appendChild($doc->createElement('range'));
            $range->setAttribute('startIP', $this->group);
            $range->setAttribute('endIP', $this->group);
            $range->setAttribute('startPort', '5000');
            $range->setAttribute('endPort', '6000');
            $range->setAttribute('ttl', '4');            
        }
    }

    protected $_RTP = 0;
    protected $_multicast = 0;

    protected $_bitRate = 0;
    protected $_serverURL = '';
    protected $_XML = '';

    protected $size = 0;

    protected $_protocols = array(
        self::DISABLED, self::IPTV, self::RTSP, self::RTMP, self::CUSTOM);
    protected $_qualities = array(self::HIGH, self::MEDIUM, self::LOW);
    protected $_codecs = array(self::MP2, self::AC3, self::AAC);
}
