/**!
 * jSignage.UI
 * https://www.spinetix.com/
 * Copyright SpinetiX S.A.
 * Released under the GPL Version 2 license.
 *
 * $Date: 2022-03-04 15:25:57 +0000 (Fri, 04 Mar 2022) $
 * $Revision: 38094 $
 * 
 * @requires jSignage
 */

(function () {
    // Private declarations
    var version = new String("1.1.2");
    version.major = 1;
    version.minor = 1;
    version.revision = 2;

    var theme = {
        bgColorDark: '#0E2A38',
        bgColorLight: '#123447',
        highlightColor: '#A7DAF4',
        fontFamily: 'Noto Sans',
        textColor: 'white'
    };

    const icons = {
        sunny: { width: 178, height: 136, d: 'M88.059,35.343c-18.121,0-32.861,14.741-32.861,32.861c0,18.121,14.74,32.861,32.861,32.861 c18.12,0,32.86-14.74,32.86-32.861C120.919,50.084,106.179,35.343,88.059,35.343z M88.059,92.4 c-13.342,0-24.194-10.854-24.194-24.195c0-13.34,10.853-24.195,24.194-24.195c13.341,0,24.194,10.855,24.194,24.195 C112.253,81.546,101.399,92.4,88.059,92.4z M88.059,29.204c2.394,0,4.333-1.94,4.333-4.333V12.593 c0-2.395-1.939-4.334-4.333-4.334s-4.333,1.939-4.333,4.334v12.277C83.726,27.263,85.665,29.204,88.059,29.204z M88.059,107.566 c-2.394,0-4.333,1.939-4.333,4.334v12.277c0,2.394,1.939,4.334,4.333,4.334s4.333-1.94,4.333-4.334V111.9 C92.392,109.505,90.452,107.566,88.059,107.566z M54.224,40.677c0.846,0.846,1.955,1.271,3.064,1.271 c1.108,0,2.217-0.425,3.063-1.271c1.692-1.691,1.692-4.434,0-6.127l-8.682-8.682c-1.69-1.692-4.437-1.692-6.127,0 c-1.692,1.692-1.692,4.436,0,6.128L54.224,40.677z M121.89,96.089c-1.691-1.693-4.436-1.693-6.127,0 c-1.693,1.691-1.693,4.436,0,6.128l8.682,8.683c0.846,0.846,1.955,1.27,3.064,1.27s2.219-0.424,3.064-1.27 c1.691-1.692,1.691-4.436,0-6.129L121.89,96.089z M48.875,68.385c0-2.394-1.941-4.334-4.334-4.334H32.264 c-2.394,0-4.333,1.94-4.333,4.334c0,2.393,1.939,4.333,4.333,4.333h12.277C46.934,72.718,48.875,70.778,48.875,68.385z M143.849,64.051h-12.279c-2.393,0-4.332,1.94-4.332,4.334c0,2.393,1.939,4.333,4.332,4.333h12.279 c2.391,0,4.332-1.94,4.332-4.333C148.181,65.992,146.239,64.051,143.849,64.051z M54.224,96.092l-8.684,8.679 c-1.693,1.693-1.693,4.438-0.001,6.129c0.846,0.848,1.955,1.271,3.064,1.271c1.108,0,2.218-0.423,3.064-1.269l8.684-8.682 c1.692-1.691,1.692-4.436,0-6.128C58.658,94.4,55.916,94.399,54.224,96.092z M118.825,41.95c1.107,0,2.217-0.423,3.062-1.269 L130.571,32c1.693-1.692,1.693-4.436,0-6.128c-1.691-1.691-4.436-1.693-6.129,0l-8.682,8.68 c-1.693,1.693-1.693,4.436-0.002,6.129C116.606,41.527,117.716,41.95,118.825,41.95z' },
        partlySunny: { width: 178, height: 136, d: 'M25.756,60.291c0-2.236-1.812-4.049-4.047-4.049H10.242c-2.235,0-4.047,1.812-4.047,4.049 c0,2.234,1.812,4.047,4.047,4.047h11.468C23.945,64.338,25.756,62.525,25.756,60.291z M62.356,23.695 c2.234,0,4.047-1.812,4.047-4.049V8.18c0-2.236-1.812-4.048-4.047-4.048c-2.236,0-4.049,1.812-4.049,4.048v11.467 C58.307,21.883,60.12,23.695,62.356,23.695z M30.752,34.412c0.79,0.789,1.827,1.186,2.862,1.186s2.071-0.396,2.862-1.186 c1.58-1.58,1.58-4.145,0-5.724l-8.109-8.108c-1.579-1.581-4.143-1.581-5.723,0c-1.581,1.58-1.581,4.143,0,5.723L30.752,34.412z M30.752,86.168l-8.11,8.107c-1.581,1.582-1.581,4.145-0.001,5.725c0.791,0.791,1.826,1.186,2.863,1.186 c1.035,0,2.071-0.395,2.861-1.184l8.111-8.109c1.58-1.58,1.58-4.143,0-5.723C34.897,84.588,32.334,84.588,30.752,86.168z M91.093,35.6c1.035,0,2.07-0.395,2.861-1.185l8.109-8.108c1.58-1.58,1.58-4.144,0.002-5.724 c-1.582-1.581-4.143-1.581-5.725-0.001l-8.11,8.107c-1.581,1.581-1.581,4.145-0.001,5.725C89.02,35.205,90.056,35.6,91.093,35.6z M52.092,80.231c-7.311-3.748-12.336-11.346-12.336-20.11c0-12.459,10.139-22.598,22.6-22.598c6.236,0,11.893,2.541,15.984,6.641 c2.217-1.625,4.59-3.05,7.087-4.252c-5.63-6.418-13.883-10.483-23.071-10.483c-16.925,0-30.693,13.769-30.693,30.692 c0,11.008,5.828,20.676,14.557,26.094C47.929,83.991,49.903,81.985,52.092,80.231z M143.247,75.299 c-1.125,0-2.25,0.068-3.367,0.201c-2.375-17.237-17.203-30.557-35.084-30.557c-19.527,0-35.415,15.889-35.415,35.417 c0,0.682,0.021,1.367,0.062,2.055c-11.896,2.059-20.975,12.453-20.975,24.928c0,13.949,11.349,25.297,25.297,25.297h69.481 c15.809,0,28.67-12.861,28.67-28.67S159.056,75.299,143.247,75.299z M143.247,124.543H73.766 c-9.484,0-17.201-7.717-17.201-17.201c0-9.451,7.66-17.146,17.1-17.201l5.184,0.107l-0.904-4.883 c-0.31-1.672-0.467-3.357-0.467-5.006c0-15.066,12.256-27.321,27.319-27.321c15.031,0,27.266,12.199,27.322,27.217l-0.002,5.58 l5.213-1.566c1.932-0.58,3.924-0.875,5.918-0.875c11.346,0,20.576,9.23,20.576,20.574 C163.823,115.315,154.593,124.543,143.247,124.543z' },
        cloudy: { width: 178, height: 136, d: 'M64.435,22.514c7.18,0,13.699,2.803,18.58,7.347c2.617-1.298,5.377-2.34,8.25-3.112 c-6.5-7.543-16.115-12.329-26.83-12.329c-19.529,0-35.414,15.888-35.414,35.415c0,0.682,0.021,1.367,0.061,2.056 C17.185,53.947,8.109,64.342,8.109,76.817c0,13.949,11.346,25.297,25.297,25.297h4.078c-0.242-1.652-0.373-3.34-0.373-5.061 c0-1.023,0.057-2.033,0.145-3.033h-3.85c-9.486,0-17.203-7.719-17.203-17.203c0-9.451,7.66-17.147,17.1-17.202l5.184,0.105 l-0.904-4.88c-0.311-1.673-0.467-3.356-0.467-5.007C37.115,34.769,49.371,22.514,64.435,22.514z M141.333,65.011 c-1.125,0-2.25,0.067-3.367,0.2c-2.375-17.237-17.203-30.556-35.084-30.556c-19.527,0-35.414,15.887-35.414,35.416 c0,0.684,0.02,1.367,0.061,2.057c-11.895,2.057-20.973,12.451-20.973,24.926c0,13.949,11.348,25.299,25.295,25.299h69.482 c15.809,0,28.67-12.863,28.67-28.672S157.142,65.011,141.333,65.011z M141.333,114.256H71.851 c-9.482,0-17.199-7.717-17.199-17.203c0-9.451,7.66-17.146,17.1-17.201l5.184,0.105l-0.904-4.879 c-0.312-1.674-0.469-3.357-0.469-5.008c0-15.065,12.258-27.321,27.32-27.321c14.994,0,27.207,12.141,27.318,27.108v5.689 l5.213-1.566c1.932-0.58,3.924-0.875,5.92-0.875c11.344,0,20.574,9.23,20.574,20.574 C161.907,105.026,152.677,114.256,141.333,114.256z' },
        rainy: { width: 178, height: 136, d: 'M122.109,41.761c-1.125,0-2.25,0.068-3.367,0.199c-2.377-17.236-17.205-30.555-35.084-30.555 c-19.527,0-35.414,15.887-35.414,35.416c0,0.682,0.02,1.367,0.061,2.055C36.408,50.933,27.332,61.33,27.332,73.804 c0,12.932,9.76,23.613,22.301,25.102l4.832-7.9h-1.838c-9.484,0-17.201-7.715-17.201-17.201c0-9.447,7.658-17.143,17.092-17.201 l5.174,0.01l-0.887-4.785c-0.309-1.672-0.467-3.357-0.467-5.006c0-15.066,12.256-27.322,27.32-27.322 c15.031,0,27.268,12.201,27.32,27.221v5.576l5.211-1.564c1.932-0.58,3.924-0.875,5.92-0.875c11.345,0,20.575,9.23,20.575,20.574 s-9.23,20.574-20.575,20.574h-1.848l-4.953,8.096h6.801c15.81,0,28.671-12.861,28.671-28.67S137.919,41.761,122.109,41.761z M89.938,61.064c-1.91-1.162-4.4-0.559-5.561,1.352l-34,55.805c-1.162,1.908-0.557,4.4,1.352,5.562 c0.658,0.4,1.385,0.592,2.102,0.592c1.365,0,2.699-0.689,3.461-1.943l33.998-55.805C92.453,64.718,91.846,62.226,89.938,61.064z M107.488,82.658c-1.914-1.164-4.4-0.559-5.564,1.352L80.479,119.21c-1.162,1.91-0.559,4.398,1.35,5.561 c0.658,0.4,1.385,0.594,2.104,0.594c1.363,0,2.697-0.693,3.459-1.943l21.447-35.203C110,86.312,109.395,83.822,107.488,82.658z' },
        windy: { width: 178, height: 136, d: 'M151.952,39.281c-2.235,0-4.048,1.812-4.048,4.049s1.812,4.047,4.048,4.047 c4.017,0,7.284,3.266,7.284,7.281s-3.268,7.283-7.284,7.283h-46.534c-2.235,0-4.048,1.812-4.048,4.047 c0,2.236,1.812,4.047,4.048,4.047h46.956c0.146,0,0.284-0.025,0.425-0.043c8.085-0.443,14.532-7.141,14.532-15.334 C167.331,46.179,160.432,39.281,151.952,39.281z M18.875,76.259c0-9.449,7.661-17.146,17.098-17.201 c0.105,0.006,0.227,0.01,0.3,0.01c1.204,0,2.345-0.535,3.113-1.459s1.088-2.143,0.868-3.324 c-0.311-1.674-0.467-3.357-0.467-5.008c0-15.064,12.254-27.32,27.321-27.32c15.044,0,27.285,12.221,27.318,27.258 c-0.001,0.033-0.001,0.066-0.001,0.096c0,1.281,0.604,2.484,1.632,3.248c1.027,0.766,2.357,0.998,3.581,0.629 c1.932-0.58,3.923-0.875,5.921-0.875c2.182,0,4.638,0.312,6.9,0.859h14.417c-0.025-0.049-0.039-0.102-0.066-0.152 c-3.938-6.959-15.622-8.801-21.251-8.801c-1.128,0-2.252,0.066-3.369,0.199c-2.376-17.238-17.203-30.557-35.082-30.557 c-19.531,0-35.417,15.889-35.417,35.416c0,0.682,0.021,1.369,0.062,2.057C19.857,53.39,10.78,63.785,10.78,76.259 c0,13.949,11.348,25.297,25.296,25.297h74.879c0.482-2.857,1.448-5.586,2.841-8.094h-77.72 C26.592,93.462,18.875,85.746,18.875,76.259z M131.756,76.539c-0.04-0.004-0.077-0.014-0.118-0.014H67.511 c-2.235,0-4.047,1.812-4.047,4.049c0,2.234,1.812,4.047,4.047,4.047h64.121c0.047,0,0.091,0.01,0.137,0.008 c5.604-0.164,11.029,2.076,14.872,6.16c2.725,2.893,4.156,6.672,4.037,10.643c-0.122,3.971-1.779,7.656-4.672,10.379 c-4.452,4.189-11.487,3.977-15.676-0.475c-1.477-1.568-2.255-3.617-2.188-5.77c0.062-2.152,0.964-4.15,2.532-5.627 c2.267-2.131,5.845-2.02,7.976,0.244c1.53,1.627,4.091,1.707,5.719,0.172c1.628-1.531,1.707-4.092,0.175-5.721 c-5.192-5.516-13.9-5.779-19.416-0.59c-3.143,2.957-4.947,6.963-5.076,11.275c-0.133,4.314,1.425,8.422,4.384,11.562 c3.768,4.006,8.865,6.027,13.973,6.027c4.719,0,9.446-1.725,13.146-5.205c9.222-8.682,9.661-23.244,0.982-32.465 C147.158,79.527,139.598,76.371,131.756,76.539z' },
        snowy: { width: 178, height: 136, d: 'M120.581,37.759c-1.104,0-2.205,0.065-3.299,0.195c-2.348-16.961-16.939-30.063-34.536-30.063 c-19.225,0-34.865,15.64-34.865,34.864c0,0.666,0.02,1.336,0.061,2.008c-11.707,2.032-20.637,12.264-20.637,24.541 c0,13.734,11.174,24.908,24.908,24.908v-8c-9.322,0-16.908-7.584-16.908-16.908c0-9.287,7.527-16.849,16.801-16.907l5.123,0.083 l-0.887-4.8c-0.307-1.649-0.461-3.307-0.461-4.925c0-14.813,12.051-26.864,26.865-26.864c14.762,0,26.78,11.967,26.864,26.709 l-0.002,5.568l5.152-1.548c1.9-0.571,3.857-0.861,5.82-0.861c11.152,0,20.225,9.073,20.225,20.227 c0,6.312-2.918,12.187-7.871,16.009c1.637,1.938,2.658,4.396,2.801,7.09l0.332,0.504c7.977-5.244,12.738-14.068,12.738-23.603 C148.806,50.421,136.144,37.759,120.581,37.759z M123.769,85.719h-0.004l-14.092,0.01l7.037-12.209c1.104-1.914,0.445-4.359-1.469-5.462 c-1.914-1.102-4.359-0.445-5.463,1.468l-7.037,12.209l-7.055-12.197c-1.105-1.912-3.553-2.566-5.465-1.46 c-1.913,1.107-2.567,3.554-1.461,5.466l7.057,12.197l-14.092,0.012c-2.211,0.002-4,1.793-3.998,4.002s1.793,3.998,4,3.998h0.004 l14.09-0.012l-7.036,12.209c-1.104,1.914-0.447,4.359,1.467,5.463c0.631,0.363,1.316,0.535,1.994,0.535 c1.385,0,2.729-0.719,3.471-2.004l7.035-12.209l7.057,12.199c0.74,1.281,2.084,1.998,3.467,1.998 c0.678,0,1.367-0.174,1.998-0.539c1.912-1.105,2.566-3.553,1.459-5.465l-7.055-12.199l14.092-0.01 c2.209-0.002,4-1.795,3.998-4.004C127.767,87.508,125.976,85.719,123.769,85.719z M57.125,128.879c-1.023,0-2.047-0.391-2.828-1.172c-1.562-1.562-1.562-4.096,0-5.656l15.91-15.91 c1.562-1.562,4.094-1.562,5.656,0c1.562,1.561,1.562,4.094,0,5.656l-15.91,15.91C59.172,128.489,58.149,128.879,57.125,128.879z M73.036,128.879c-1.023,0-2.047-0.391-2.828-1.172l-15.91-15.91c-1.562-1.562-1.562-4.096,0-5.656 c1.562-1.562,4.094-1.562,5.656,0l15.91,15.91c1.562,1.561,1.562,4.094,0,5.656C75.083,128.489,74.059,128.879,73.036,128.879z' }
    };

    /**
     * @extends jSignage
     */
    jSignage.UI = {
        version: version,
        fingerTouchPath: 'M74.3,205.7V73.6c0-10.1,8.2-18.3,18.3-18.3c10.1,0,18.3,8.2,18.3,18.3v88.7 M141.3,168.4v-15.9c0-8.4-6.8-15.2-15.2-15.2c-8.4,0-15.2,6.8-15.2,15.2v9.8 M141.3,168.4v-15.9c0-8.4,6.8-15.2,15.2-15.2c8.4,0,15.2,6.8,15.2,15.2v22 M171.8,174.5v-22c0-8.4,6.8-15.2,15.2-15.2c8.4,0,15.2,6.8,15.2,15.2v111.7c0,14.7-15.2,33.5-30.5,33.5 c-17.8,0-56.1,4.6-75.4-7.6C77.1,277.9,57,237.5,40.5,221c-21.8-21.8-37.6-29.5-34-37.6c5.8-13.3,32-11.7,51.3,3.6 c10.2,8,16.5,18.8,16.5,18.8 M58.3,104.3c-21.9-21.9-21.9-57.5,0-79.5 M38.4,124.2C5.5,91.3,5.5,37.9,38.4,5 M130.6,24.9c21.9,21.9,21.9,57.5,0,79.5 M150.5,5c32.9,32.9,32.9,86.3,0,119.2',
        smileys: [
            { id: 'sad', fill: '#de5b49', width: 131, height: 131, d: 'M65.25,0a65.25,65.25,0,1,0,65.25,65.25A65.32,65.32,0,0,0,65.25,0Zm0,8.51A56.74,56.74,0,1,1,8.51,65.25,56.68,56.68,0,0,1,65.25,8.51ZM42.56,36.88a9.93,9.93,0,1,0,9.93,9.93A9.93,9.93,0,0,0,42.56,36.88Zm45.39,0a9.93,9.93,0,1,0,9.93,9.93A9.93,9.93,0,0,0,88,36.88ZM65.25,72.35A41.86,41.86,0,0,0,30.5,91.23,4.27,4.27,0,0,0,37.58,96l0,0a32.87,32.87,0,0,1,55.32,0A4.27,4.27,0,0,0,100,91.26l0,0A41.86,41.86,0,0,0,65.25,72.35Z' },
            { id: 'neutral', fill: '#f0ca4d', width: 131, height: 131, d: 'M65.25,0a65.25,65.25,0,1,0,65.25,65.25A65.32,65.32,0,0,0,65.25,0Zm0,8.51A56.74,56.74,0,1,1,8.51,65.25,56.68,56.68,0,0,1,65.25,8.51ZM42.56,36.88a9.93,9.93,0,1,0,9.93,9.93A9.93,9.93,0,0,0,42.56,36.88Zm45.39,0a9.93,9.93,0,1,0,9.93,9.93A9.93,9.93,0,0,0,88,36.88Zm5.67,30.5zM92.81,91.47a4,4,0,0,0,3.94-3.94,3.93,3.93,0,0,0-3.94-3.94H37.7a4,4,0,0,0-3.94,3.94,3.93,3.93,0,0,0,3.94,3.94Z' },
            { id: 'happy', fill: '#46b39d', width: 131, height: 131, d: 'M65.25,0a65.25,65.25,0,1,0,65.25,65.25A65.32,65.32,0,0,0,65.25,0Zm0,8.51A56.74,56.74,0,1,1,8.51,65.25,56.68,56.68,0,0,1,65.25,8.51ZM42.56,36.88a9.93,9.93,0,1,0,9.93,9.93A9.93,9.93,0,0,0,42.56,36.88Zm45.39,0a9.93,9.93,0,1,0,9.93,9.93A9.93,9.93,0,0,0,88,36.88Zm-22.7,61A41.86,41.86,0,0,0,100,79a4.27,4.27,0,0,0-7.08-4.77l0,0a32.87,32.87,0,0,1-55.32,0A4.27,4.27,0,0,0,30.48,79l0,0A41.86,41.86,0,0,0,65.25,97.88Z' }
        ],

        setTheme: function (args) {
            if (!window.__jSignage__global.UI) {
                window.__jSignage__global.UI = {
                    theme: theme
                };
            }
            window.__jSignage__global.UI.theme = this.mergeStyle(window.__jSignage__global.UI.theme, args);
        },

        theme: function () {
            return window.__jSignage__global.UI && window.__jSignage__global.UI.theme || theme;
        },

        /**
         * Builds a  text area style object using text attributes 
         * from A, otherwise B, otherwise default values.
         * @param {Object} A 
         * @param {Object} B 
         * @returns {Object}
         */
        getTextStyle: function (A, B) {
            return {
                fill: A.fill || B.fill || this.theme().textColor,
                fontSize: A.fontSize || B.fontSize || 'max',
                fontFamily: A.fontFamily || B.fontFamily || this.theme().fontFamily,
                fontStyle: A.fontStyle || B.fontStyle || 'normal',
                fontVariant: A.fontVariant || B.fontVariant || 'normal',
                fontWeight: A.fontWeight || B.fontWeight || 'bold',
                textAlign: A.textAlign || B.textAlign || 'center',
                displayAlign: A.displayAlign || B.displayAlign || 'center',
                lineIncrement: A.lineIncrement || B.lineIncrement || 'auto',
                frame: A.frame || B.frame || null
            }
        },

        /**
         * Copies attributes from B into A.
         * @param {Object} A 
         * @param {Object} B 
         * @returns {Object}
         */
        mergeStyle: function (A, B) {
            for (var x in B) {
                if (!(x in A))
                    A[x] = B[x];
                else if (jSignage.isArray(A[x]))
                    A[x] = A[x].concat(B[x]);
                else if (typeof (A[x]) == 'object' && A[x] && typeof (B[x]) == 'object' && B[x])
                    jSignage.UI.mergeStyle(A[x], B[x]);
                else
                    A[x] = B[x];
            }
            return A;
        },

        /**
         *  Calculates a point on a circle, given a center point and a radius. 
         *  The angle can be provided or randomly selected.
         *  @param {number} cx
         *  @param {number} cy
         *  @param {number} radius
         *  @param {number=} angle Optional angle value in radians.
         *  @return {Object} An object defining the selected point.
         */
        getPointOnCircle: function (cx, cy, radius, angle) {
            if (angle === undefined) {
                angle = 2 * Math.PI * Math.random();
            }
            return {
                angle: angle,
                x: cx + Math.cos(angle) * radius,
                y: cy + Math.sin(angle) * radius
            };
        },

        /**
         * Generates an icon.
         * @param {Object} args 
         * @param {Object} args.layer Object describing a layer or a path.
         * @param {number} args.width Layer's width.
         * @param {number} args.height Layer's height.
         * @param {number} args.top Layer's top offset.
         * @param {number} args.left Layer's left offset.
         * @param {number} args.opacity Layer's opacity, between 0 and 1.
         * @param {number} args.scaleFactor Icon' scale factor, between 0 and 1.
         * @param {string} args.fill Fill color
         * @param {string} args.stroke Stroke color.
         * @param {string} args.style Stroke style. 
         * @returns {Object} jSignage layer
         */
        icon: function (args) {
            const width = args.width || 256,
                height = args.height || 256,
                top = args.top || 0,
                left = args.left || 0,
                scaleFactor = args.scaleFactor || 0.8,
                opacity = args.opacity || 1,
                style = args.style || 'round';
            var f, cx, cy,
                layer = jSignage.extend(true, {}, args.layer || {}),
                icon = jSignage.g({ width: width, height: height, top: top, left: left, opacity: opacity });

            if ('ctor' in layer) {
                icon.add(jSignage.uncan(layer));
            } else if ('d' in layer) {
                layer.width = ('width' in layer && jSignage.isNumeric(layer.width) ? parseFloat(layer.width) : width);
                layer.height = ('height' in layer && jSignage.isNumeric(layer.height) ? parseFloat(layer.height) : height);
                if (!('fill' in layer)) {
                    layer.fill = args.fill || jSignage.UI.theme().highlightColor;
                }
                if (!('stroke' in layer) && ('strokeWidth' in layer)) {
                    layer.stroke = args.stroke || jSignage.UI.theme().highlightColor;
                }
                if (!('strokeLineCap' in layer)) {
                    layer.strokeLineCap = (style == 'round' ? 'round' : 'square');
                }
                if (!('strokeLineJoin' in layer)) {
                    layer.strokeLineJoin = (style == 'round' ? 'round' : 'mitter');
                }
                f = Math.min(width / layer.width, height / layer.height) * scaleFactor;
                cx = (width - f * layer.width) / 2;
                cy = (height - f * layer.height) / 2;
                layer = jSignage.path(layer).addTo(icon);
                layer.attr('transform', 'translate(' + cx + ',' + cy + ') scale(' + f + ')');
            }
            return icon;
        },

        /**
         * Makes a button out of an icon.
         * @param {Object} args Object that can have any of the properties detailed under {@link icon}, plus
         * @param {Function} args.onButtonClick Handler function for the click event.
         * @returns {Object} jSignage layer
         */
        button: function (args) {
            if (arguments.length > 1) {
                // backwards compatibility
                args = {
                    layer: arguments[0],
                    width: arguments[1],
                    height: arguments[2],
                    opacity: arguments[3],
                    style: arguments[4],
                    onButtonClick: arguments[5]
                }
            }
            var layer = args.layer || {},
                button = jSignage.UI.icon(args);
            if ('onButtonClick' in args && typeof args.onButtonClick === 'function') {
                if ('d' in layer) {
                    const overlay = jSignage.rect({
                        width: args.width || 256,
                        height: args.height || 256,
                        fill: 'none', stroke: 'none', pointerEvents: 'fill'
                    }).click(args.onButtonClick);
                    button.add(overlay);
                } else
                    button.click(args.onButtonClick);
            }
            return button;
        },

        menu: function (args) {
            if (!args || !args.data || !args.data.length) {
                return;
            }
            if (!args.direction) {
                args.direction = 'topToBottom';
            }
            if (!args.position) {
                args.position = 'left-center';
            }
            if (!args.width && (args.position !== 'top' && args.position !== 'bottom')) {
                args.width = '22%';
            }
            if (!args.right && args.position.startsWith('right')) {
                args.right = 0;
            }
            if (!args.bottom && args.position === 'bottom') {
                args.bottom = 0;
            }
            return new jSignage.UI.menuClass('stripCarouselClass', args);
        },

        tabs: function (args) {
            if (!args || !args.data || !args.data.length) {
                return;
            }
            if (!args.position) {
                args.position = 'top';
            }
            if (!args.height && (args.position === 'top' || args.position === 'bottom')) {
                args.height = '10%';
            }
            if (!args.right && args.position.startsWith('right')) {
                args.right = 0;
            }
            if (!args.bottom && args.position === 'bottom') {
                args.bottom = 0;
            }
            return new jSignage.UI.tabsClass('stripCarouselClass', args);
        },

        cursor: function (args, parent) {
            var r = args && args.size || 10,
                color = args && args.color || '#FF0000',
                type = args && args.type || 'rect',
                cursor = animation = null;

            if (type === 'circle') {
                cursor = jSignage.circle({ r: r, fill: color, pointerEvents: 'none' });
            } else {
                cursor = jSignage.rect({
                    x: -r, y: -r, width: 2 * r, height: 2 * r,
                    fill: color, pointerEvents: 'none'
                });
            }
            if (!parent) {
                parent = jSignage('svg');
            }
            parent.add(cursor).mousemove(function (evt) {
                var c = jSignage.getLocalCoord(this, evt.clientX, evt.clientY);
                if (animation) {
                    jSignage.removeAnimation(animation);
                }
                animation = jSignage.svgAnimation(cursor[0], 'animateTransform', {
                    attributeName: 'transform',
                    type: 'translate',
                    values: c.x + ',' + c.y + ';' + c.x + ',' + c.y,
                    begin: 0, dur: 'indefinite'
                });
            });
            return cursor;
        },

        feedback: function (args) {
            if (!args || !args.data || !args.data.length) {
                args.data = jSignage.UI.smileys;
            }
            return new jSignage.UI.feedbackClass('stripCarouselClass', args);
        },

        fingerTouch: function (args) {
            return new jSignage.UI.fingerTouchClass('fingerTouch', args);
        },

        /**
         * Generates a weather icon
         * @param {Object} args Object that can have any of the properties detailed under {@link icon}, plus
         * @param {string} args.icon The icon's name.
         * @returns {Object} jSignage layer
         */
        weatherIcon: function (args) {
            if (!args || !args.icon) return null;
            const icon = args.icon;
            if (icon == 'cloudy1') {
                args.layer = icons.partlySunny;
            } else if (icon.startsWith('cloud') || icon == 'mist') {
                args.layer = icons.cloudy;
            } else if (icon.startsWith('sun')) {
                args.layer = icons.sunny;
            } else if (icon.startsWith('snow')) {
                args.layer = icons.snowy;
            } else if (icon.startsWith('wind') || icon == 'tornado') {
                args.layer = icons.windy;
            } else if (icon == 'storm' || icon == 'thunderstorm' || icon.startsWith('rain')) {
                args.layer = icons.rainy;
            } else {
                args.layer = icons.partlySunny;
            }
            if (!args.scaleFactor) args.scaleFactor = 1;

            return jSignage.UI.icon(args);
        },

        // UI classes

        menuClass: jSignage.subclass(
            jSignage.stripCarouselClass,
            function (args) {
                this.position = args.position;
                this.onRight = args.position.startsWith('right');
                this.numVisible = args.numVisible || 3;
                if (this.numVisible > args.data.length) {
                    this.numVisible = args.data.length;
                    this.draggable = false;
                }
                this.labels = new Array(args.data.length);
                this.isVisible = true;
                this.autoHideLabels = args.autoHideLabels || false;
                this.autoHideTimer = null;
                this.autoHideOffset = null;
                this.delay = jSignage.durInSeconds(args.delay, 5);
                this.animateDur = jSignage.durInSeconds(args.animateDur, 0.5);
                this.renderToSVG = jSignage.isFunction(args.renderToSVG) ? args.renderToSVG : this._defaultRenderToSVG(args);
                this.onItemClick = jSignage.isFunction(args.onItemClick) ? args.onItemClick : false;
            }, {
            // more methods
            postLayout: function (g, width, height) {
                var self = this, maxItems, trm;
                this.setWH(width, height);
                this.pixSlide = this.height * 3 / 16;
                maxItems = Math.floor(this.width / this.pixSlide);
                if (maxItems < this.numVisible) {
                    this.numVisible = maxItems;
                    this.draggable = true;
                }
                this.maxPixPosition = this.pixSlide * (this.data.length - this.numVisible);
                this.margin = this.pixSlide * this.spacing;
                this.slideWidth = this.pixSlide - this.margin;
                this.slideHeight = this.height;
                if (this.vertical) {
                    trm = g.getAttribute('transform');
                    height = this.pixSlide * this.numVisible;
                    g.setAttribute('height', height);
                    if (this.position.endsWith('bottom')) {
                        g.setAttribute('transform', trm + ' translate(0,' + (this.width - height) + ')');
                    } else if (this.position.endsWith('center')) {
                        g.setAttribute('transform', trm + ' translate(0,' + (this.width - height) / 2 + ')');
                    }
                } else {
                    // TO DO : add support for hortizontal menus.
                    //width = this.pixSlide * this.numVisible;
                }

                jSignage.carouselClass.prototype.postLayout.call(this, g, width, height);
                this.beginEvent(function () {
                    if (self.autoHideLabels) {
                        self.hideLabels();
                    }
                });
            },

            _hideAnim: function (target) {
                var dur = this.animateDur,
                    toXY = (this.slideHeight - this.autoHideOffset) * (this.onRight ? 1 : -1) + ',0';
                return jSignage.svgAnimation(target, 'animateMotion', {
                    from: '0,0', to: toXY,
                    begin: 'indefinite',
                    dur: dur, fill: 'freeze'
                });
            },

            _showAnim: function (target) {
                var dur = this.animateDur,
                    fromXY = (this.slideHeight - this.autoHideOffset) * (this.onRight ? 1 : -1) + ',0';
                return jSignage.svgAnimation(target, 'animateMotion', {
                    from: fromXY, to: '0,0',
                    begin: 'indefinite',
                    dur: dur, fill: 'freeze'
                });
            },

            _defaultRenderToSVG: function (args) {
                var self = this,
                    defaultTextStyle = jSignage.UI.getTextStyle(args, { fontSize: 44 });

                return function (index, item) {
                    var g, pageNumber = '', layers = [], attr, label,
                        width, height, cw = 0, offset = 0, cellTextStyle,
                        color = (args.banded && index % 2) ? jSignage.UI.theme().bgColorLight : jSignage.UI.theme().bgColorDark;

                    if (self.vertical) {
                        width = self.slideHeight;
                        height = self.slideWidth;
                    } else {
                        width = self.slideWidth;
                        height = self.slideHeight;
                    }
                    pad = Math.floor(height * 0.1);

                    cellTextStyle = jSignage.extend(true, {}, defaultTextStyle,
                        { frame: { backColor: color, padding: [pad / 2, pad, pad * 2.5, pad] } },
                        args.cellStyle ? args.cellStyle : {}
                    );
                    if (args.showPageNumber) {
                        cw = pad * 9;
                        pageNumber = jSignage.applyDateNumberFormat(index + 1, jSignage.getLocale(), args.data.length > 9 ? '00' : '#');
                    } else {
                        cw = pad * 3;
                    }
                    cellTextStyle.width = cw;
                    if (self.onRight) {
                        cellTextStyle.right = 0;
                    }
                    layers.push(jSignage.fitTextArea(cellTextStyle).text(pageNumber));

                    offset = cw - 1;
                    cw = pad * 5;
                    if (self.onRight) {
                        attr = { right: offset, width: cw };
                    } else {
                        attr = { left: offset, width: cw };
                    }
                    attr = jSignage.extend(true, {}, jSignage.UI.getTextStyle({
                        fontSize: 'max',
                        fill: jSignage.UI.theme().highlightColor,
                        displayAlign: 'center',
                        textAlign: self.onRight ? 'end' : 'start',
                        frame: { backColor: color, padding: [0, 0, pad / 2, 0] }
                    }, args), attr);
                    layers.push(jSignage.textArea(attr).text('∣'));
                    offset += cw - 1;

                    self.autoHideOffset = offset;

                    delete cellTextStyle.width;
                    if (self.onRight) {
                        cellTextStyle.right = offset;
                    } else {
                        cellTextStyle.left = offset;
                    }
                    label = jSignage.fitTextArea(cellTextStyle).text(item || '');
                    layers.push(label);
                    if (self.autoHideLabels) {
                        var toXY = (self.slideHeight - self.autoHideOffset) * (self.onRight ? 1 : -1) + ',0';
                        var animHide = function () {
                            return jSignage.beginAnimation(
                                jSignage.svgAnimation(label[0], 'animateMotion', {
                                    from: '0,0', to: toXY,
                                    dur: 0.5, fill: 'freeze'
                                })
                            );
                        };
                        self.labels[index] = {
                            layer: label,
                            hide: self._hideAnim(label[0]),
                            show: self._showAnim(label[0])
                        };
                    }

                    g = jSignage.g().add(layers.reverse());
                    return g;
                }
            },

            clickAt: function (x, y) {
                if (!this.onItemClick) { return; }
                x += this.pixPosition;
                var choice = Math.floor(x / this.pixSlide);
                if (x >= this.pixSlide * choice + this.margin &&
                    x < this.pixSlide * choice + this.margin + this.slideWidth) {
                    this.onItemClick.call(this, choice);
                }
            },

            mouseDown: function (ev, x, y) {
                this.showLabels();
            },

            mouseOut: function (ev, x, y) {
                if (this.autoHideLabels && this.isVisible) {
                    this.hideLabels();
                }
            },

            hideLabels: function () {
                if (!this.isVisible) return;
                var self = this;

                if (this.autoHideTimer) {
                    jSignage.clearTimeout(this.autoHideTimer);
                    this.autoHideTimer = null;
                }
                this.autoHideTimer = jSignage.setTimeout(function () {
                    if (self.dragging) {
                        self.hideLabels();
                        return;
                    }
                    for (var i = 0; i < self.labels.length; i++) {
                        if (!self.labels[i]) continue;
                        jSignage.beginAnimation(self.labels[i].hide);
                        //jSignage.removeAnimation(self.labels[i].show);
                    }
                    self.isVisible = false;
                    self.overlay.setAttribute('width', self.autoHideOffset);
                    if (self.onRight) {
                        self.overlay.setAttribute('transform', 'translate(' + (self.slideHeight - self.autoHideOffset) + ',0)');
                    }
                }, this.delay * 1000);
            },

            showLabels: function (preventHide) {
                if (!this.isVisible) {
                    for (var i = 0; i < this.labels.length; i++) {
                        if (!this.labels[i]) continue;
                        jSignage.beginAnimation(this.labels[i].show);
                        //jSignage.removeAnimation(this.labels[i].hide);
                    }
                    this.isVisible = true;
                    this.overlay.setAttribute('width', this.height);
                    if (this.onRight) {
                        this.overlay.setAttribute('transform', '');
                    }
                }
                if (this.autoHideLabels && preventHide !== true) {
                    this.hideLabels();
                }
            }
        }
        ),

        tabsClass: jSignage.subclass(
            jSignage.stripCarouselClass,
            function (args) {
                this.position = args.position;
                this.numVisible = args.numVisible || 3;
                if (this.numVisible > args.data.length) {
                    this.numVisible = args.data.length;
                    this.draggable = false;
                }
                this.renderToSVG = jSignage.isFunction(args.renderToSVG) ? args.renderToSVG : this._defaultRenderToSVG(args);
                this.onItemClick = jSignage.isFunction(args.onItemClick) ? args.onItemClick : false;
            }, {
            // more methods
            _defaultRenderToSVG: function (args) {
                var defaultTextStyle = jSignage.extend(true, {},
                    jSignage.UI.getTextStyle(args, { fontSize: 44 }),
                    { frame: { backColor: jSignage.UI.theme().bgColorDark, padding: [10, 10, 20, 10] } },
                    args.cellStyle ? args.cellStyle : {}
                );
                return function (index, item) {
                    var cellTextStyle = (args.banded && index % 2)
                        ? jSignage.extend(true, {}, defaultTextStyle, { frame: { backColor: jSignage.UI.theme().bgColorLight } })
                        : defaultTextStyle;

                    return jSignage.fitTextArea(cellTextStyle).text(item || '');
                }
            },

            clickAt: function (x, y) {
                if (!this.onItemClick) { return; }
                x += this.pixPosition;
                var choice = Math.floor(x / this.pixSlide);
                if (x >= this.pixSlide * choice + this.margin &&
                    x < this.pixSlide * choice + this.margin + this.slideWidth) {
                    this.onItemClick.call(this, choice);
                }
            }
        }
        ),

        feedbackClass: jSignage.subclass(
            jSignage.stripCarouselClass,
            function (args) {
                this.data = args.data;
                this.numVisible = args.numVisible || args.data.length;
                this.draggable = false;
                this.autoTimer = null;
                this.animate = args.animate || false;
                this.animateDur = jSignage.durInSeconds(args.animateDur, 2);
                this.displayResults = args.displayResults || false;
                this.stopClicks = false;
                this.displayResultsDur = jSignage.durInSeconds(args.displayResultsDur, 5);
                this.renderToSVG = jSignage.isFunction(args.renderToSVG) ? args.renderToSVG : this._defaultRenderToSVG(args);
                this.onItemClick = jSignage.isFunction(args.onItemClick) ? args.onItemClick : false;
                this.key = 'jSignage:UI:feedbackResults:' + (args.keyName || '');
                this.votes = '[]';
                this.filter = args.filter || false;
                this.limit = args.limit || 10000;
                this.storage = window.createSharedVariable ? createSharedVariable(this.key) : null;
                if (this.storage) {
                    var self = this;
                    this.storage.addUpdateListener(function (sv) {
                        self.votes = sv.value;
                    }, args.sync || true);
                }
                this.backColor = args.backColor || jSignage.UI.theme().bgColorLight;
                this.colors = jSignage.isArray(args.colors) ? args.colors : [];
                for (var i = this.colors.length; i < this.data.length; i++) {
                    this.colors[i] = (jSignage.isPlainObject(this.data[i]) && 'fill' in this.data[i])
                        ? this.data[i].fill
                        : jSignage.UI.theme().bgColorDark;
                }
                this.cellStyle = args.cellStyle || {};
                this.defaultTextStyle = jSignage.extend(true, {},
                    jSignage.UI.getTextStyle(args, { fontSize: 72 }),
                    { frame: { frameSize: 0, padding: ['10%', '10%', '20%', '10%'] } },
                    this.cellStyle
                );
                this.scaleFactor = jSignage.getPercent(args.scaleFactor, 0.8);
            }, {
            postLayout: function (g, width, height) {
                if (this.horizontal && width / height < 1) {
                    this.direction = this.reverse ? 'bottomToTop' : 'topToBottom';
                    this.vertical = true;
                    this.horizontal = false;
                    this.scaleFactor = Math.sqrt(this.scaleFactor);
                }
                jSignage.stripCarouselClass.prototype.postLayout.call(this, g, width, height);
            },

            _defaultRenderToSVG: function (args) {
                var self = this;
                return function (index, item) {
                    var w, h;
                    if (self.vertical) {
                        w = self.slideHeight;
                        h = self.slideWidth;
                    } else {
                        w = self.slideWidth;
                        h = self.slideHeight;
                    }
                    return jSignage.g(self.cellStyle).add(self._renderCell(index, item, w, h, self.scaleFactor));
                }
            },

            _renderCell: function (index, item, width, height, scaleFactor) {
                var w, h, f, cx, cy, layer = null,
                    color = this.colors[index],
                    cell = jSignage.g(jSignage.extend(true, {}, this.cellStyle, { width: width, height: height }));

                if (!jSignage.isPlainObject(item)) {
                    var defaultTextStyle = jSignage.extend(true, {},
                        this.defaultTextStyle,
                        { fill: color, frame: { frameColor: color } },
                        this.cellStyle
                    );
                    cell.add(jSignage.fitTextArea(defaultTextStyle).text(item || ''));
                } else if ('ctor' in item) {
                    cell.add(jSignage.uncan(item));
                } else if ('d' in item) {
                    if (!item.width) {
                        item.width = width;
                    }
                    if (!item.height) {
                        item.height = height;
                    }
                    f = Math.min(width / item.width, height / item.height) * scaleFactor;
                    cx = (width - f * item.width) / 2;
                    cy = (height - f * item.height) / 2;
                    item.fill = color;
                    layer = jSignage.path(item);
                    layer.attr('transform', 'translate(' + cx + ',' + cy + ') scale(' + f + ')');
                    cell.add(layer);
                }
                return cell;
            },

            _calculate: function (votes) {
                var i, d = 0, count = 0,
                    ratings = [];

                for (i = 0; i < this.data.length; i++) {
                    ratings[i] = 0;
                }
                if (this.filter) {
                    if (this.filter === 'hour') {
                        d = Date.now() - 3600000;
                    } else {
                        d = new Date();
                        d = Date.UTC(
                            d.getUTCFullYear(),
                            d.getUTCMonth(),
                            this.filter == 'day' ? d.getUTCDate() : 1
                        );
                    }
                }
                // count votes
                for (i = votes.length - 1; i >= 0; i--) {
                    var item = votes[i];
                    if (this.filter && item.d < d) {
                        jSignage.info('Filter reached : ' + count + ' from ' + votes.length + ' votes are considered.');
                        break;
                    }
                    ratings[item.v]++;
                    count++;
                }
                // calculate percentages
                if (count) {
                    for (i = 0; i < ratings.length; i++) {
                        ratings[i] = {
                            count: ratings[i],
                            rate: ratings[i] / count
                        }
                    }
                }
                return ratings.slice(0, this.data.length);
            },

            clickAt: function (x, y) {
                if (this.stopClicks) { return false; }
                x += this.pixPosition;
                var choice = Math.floor(x / this.pixSlide);
                if (x >= this.pixSlide * choice + this.margin &&
                    x < this.pixSlide * choice + this.margin + this.slideWidth) {
                    if (this.onItemClick) {
                        this.onItemClick.call(this, choice);
                    }
                    this.vote(choice);
                }
            },

            vote: function (value) {
                var a = JSON.parse(this.votes || '[]');
                a.push({ d: Date.now(), v: value });
                if (a.length > this.limit) {
                    jSignage.info('There are now ' + a.length + ' votes; keeping only the last ' + this.limit + ' votes.');
                    a = a.slice(-this.limit);
                }
                if (this.storage) {
                    this.storage.set(JSON.stringify(a));
                } else {
                    localStorage.setItem(this.key, JSON.stringify(a));
                }
                if (this.displayResults) {
                    this.showResults(this._calculate(a));
                }
            },

            resetVotes: function () {
                if (this.storage) {
                    this.storage.set('[]');
                } else {
                    localStorage.setItem(this.key, '[]');
                }
            },

            showResults: function (results) {
                var self = this, overlay = null, cellWidth, rowHeight, barWidth, barHeight, fontSize;

                if (this.vertical) {
                    cellWidth = 0.25 * this.height;
                    rowHeight = this.width * (1 / results.length - 0.05);
                    barWidth = 2 * cellWidth;
                } else {
                    cellWidth = 0.2 * this.width;
                    rowHeight = this.height * (1 / results.length - 0.05);
                    barWidth = 3 * cellWidth;
                }
                fontSize = Math.round(Math.min(rowHeight / 2.4, cellWidth / 3.5));
                barHeight = 0.3 * rowHeight;

                overlay = jSignage.table({
                    id: 'feedbackResults', begin: 'now', dur: this.displayResultsDur,
                    frame: { backColor: this.backColor },
                    primaryOrder: this.reverse ? 'bottomToTop' : 'topToBottom',
                    rows: results.length,
                    rowPadding: '5%',
                    renderToSVG: function (index) {
                        var layer = bar = null,
                            color = self.colors[index],
                            text = jSignage.applyDateNumberFormat(results[index].rate, jSignage.getLocale(), '%');

                        bar = jSignage.g({ left: cellWidth, width: barWidth, height: rowHeight }).add(
                            jSignage.rect({
                                fill: color,
                                height: barHeight,
                                width: barWidth * results[index].rate,
                                y: 0.35 * rowHeight
                            })
                        ).wipeIn({ dur: self.animateDur, type: 'bar', subType: 'leftToRight' });

                        layer = jSignage.g(self.cellStyle).add([
                            self._renderCell(index, self.data[index], cellWidth, rowHeight, 0.8),
                            bar,
                            jSignage.fitTextArea(jSignage.extend(true, {}, self.defaultTextStyle, {
                                right: 0, width: cellWidth, height: rowHeight,
                                textAlign: 'end', fontSize: fontSize, fill: color
                            })).text(text)
                        ]);

                        return layer;
                    }
                });

                this.stopClicks = true;
                overlay.end(this.displayResultsDur).removeAfter(0, function () {
                    self.stopClicks = false;
                }).addTo(jSignage(this));
            }
        }
        ),

        fingerTouchClass: jSignage.subclass(
            jSignage.widgetClass,
            function (args) {
                this.autoTimer = null;
                this.animate = args.animate || false;
                this.animateDur = jSignage.durInSeconds(args.animateDur, 5);
                this.delay = jSignage.durInSeconds(args.delay, 5);
                this.animateMin = args.animateMin || 0;
                this.animateMax = args.animateMax || 300;
                this.colors = jSignage.isArray(args.colors) ? args.colors : [jSignage.UI.theme().bgColorDark, jSignage.UI.theme().highlightColor];
                this.pathAttr = {
                    d: args.path || jSignage.UI.fingerTouchPath,
                    fill: args.fill || 'none',
                    stroke: args.color || this.colors[0],
                    strokeWidth: args.strokeWidth || 7,
                    width: args.pathWidth || 300,
                    height: args.pathHeight || 300
                };
            }, {
            postLayout: function (g, width, height, x, y, bbw, bbh, parent) {
                jSignage.widgetClass.prototype.postLayout.call(this, g, width, height);
                var self = this,
                    colors = this.colors,
                    maxX = (width - this.pathAttr.width),
                    maxY = (height - this.pathAttr.height),
                    cx = maxX / 2,
                    cy = maxY / 2,
                    path = jSignage.path(this.pathAttr).attr('transform', 'translate(' + cx + ',' + cy + ')');

                g.appendChild(path[0]);

                if (this.animate) {
                    path.animateOpacity('stroke', {
                        values: [0, 1, 1, 0],
                        keyTimes: [0, 0.25, 0.75, 1],
                        begin: 0, dur: this.animateDur,
                        repeatCount: 'indefinite'
                    });
                    if (colors.length > 1) {
                        path.animateColor('stroke', {
                            values: [colors[0], colors[0], colors[1], colors[1], colors[0]],
                            keyTimes: [0, 0.45, 0.5, 0.95, 1],
                            begin: 0, dur: 2 * this.animateDur,
                            repeatCount: 'indefinite'
                        });
                    }
                }

                this.beginEvent(function () {
                    if (self.autoTimer) {
                        jSignage.clearInterval(self.autoTimer);
                        self.autoTimer = null;
                    }
                    var move = function () {
                        if (jSignage.isInRenderingTree(g)) {
                            var radius = Math.random() * (self.animateMax - self.animateMin) + self.animateMin,
                                point = jSignage.UI.getPointOnCircle(cx, cy, radius),
                                tx = point.x < 0 ? 0 : (point.x > maxX ? maxX : point.x),
                                ty = point.y < 0 ? 0 : (point.y > maxY ? maxY : point.y);
                            path.attr('transform', 'translate(' + tx + ',' + ty + ')');
                        } else {
                            jSignage.clearInterval(self.autoTimer);
                            self.autoTimer = null;
                        }
                    };
                    if ( !navigator.spxDocumentTime || document.documentElement.getCurrentTime() != navigator.spxDocumentTime ) {
                        self.timer = createTimer( ( navigator.spxDocumentTime - document.documentElement.getCurrentTime() + self.animateDur ) * 1000, self.animateDur * 1000 );
                        self.timer.addEventListener( 'SVGTimer', move, false );
                        self.timer.start();
                    } else {
                        self.autoTimer = jSignage.setInterval( move, self.animateDur * 1000 );
                    }
                });

                this.endEvent(function () {
                    jSignage.setTimeout(function () {
                        self.show();
                    }, self.delay * 1000);
                });
            },

            mouseDown: function (ev, x, y) {
                this.hide();
                jSignage.setTimeout(function () {
                    fireSharedEvent('mousedown', 'x=' + ev.screenX + ',y=' + ev.screenY + ',button=0');
                }, 1);
            }
        }
        )
    };

})();
