<?php
class IPConfig extends CFormModel
{
    const NET_ETHERNET = "ethernet";
    const NET_WIFI = "wifi";

    protected $loaded = false;
    var $netiface = self::NET_ETHERNET;
    protected $currentNetiface = self::NET_ETHERNET;
    var $dhcp = 1;
    var $address = "";
    var $netmask = "";
    var $gateway = "";

    var $ipv6Configuration = "none"; // none, static, dhcp
    var $ipv6ConfigurationOption = array(
        "none" => "None",
        /* "dhcp" => "DHCP", */
        "static" => "Static",
    );
    var $ipv6Address = ""; //
    var $ipv6Netmask = ""; // 0-128
    var $ipv6Gateway = ""; //

    var $dnsConfiguration = "automatic"; // manual, automatic
    var $nameserver1 = "";
    var $nameserver2 = "";
    var $nameserver3 = "";
    var $domain = "";

    var $fnameDNS = "/etc/network/resolv.conf.static";

    var $fnameSLAAC = "/etc/default/network";
    var $disableSLAAC = 0;

    protected $startTag = "# START-spxsysconfig DO NOT REMOVE THE START AND END TAGS";
    protected $endTag = "# END-spxsysconfig";
    protected $fname = "/etc/network/interfaces";

    var $configType = array("DHCP" => 1, "Static" => 0);

    protected $ipv6 = array();

    public function getNetifaceSelect()
    {
        $netifaceSelect = array(
            IPConfig::NET_ETHERNET => "Ethernet",
        );
        if (Yii::app()->branding->hasRight('WIFI')) {
            $netifaceSelect[IPConfig::NET_WIFI] = "Wi-Fi";
        }
        return $netifaceSelect;
    }

    /**
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        return array(
            array('netiface', 'safe'),
            array(
                'address, netmask',
                'requiredIf',
                'var' => 'dhcp',
                'value' => 0,
            ),
            array('address, gateway', 'checkIPv4', 'allowEmpty' => true),
            array('netmask', 'checkNetmask', 'allowEmpty' => true),

            array(
                'ipv6Configuration',
                'in',
                'range' => array("none", "static" /*"dhcp"*/),
            ),
            array(
                'ipv6Address, ipv6Netmask',
                'requiredIf',
                'var' => 'ipv6Configuration',
                'value' => 'static',
            ),
            array(
                'ipv6Address, ipv6Gateway',
                'checkIPv6',
                'allowEmpty' => true,
            ),
            array(
                'ipv6Netmask',
                'numerical',
                'allowEmpty' => true,
                'integerOnly' => true,
                'min' => 0,
                'max' => 128,
            ),

            array(
                'dnsConfiguration',
                'in',
                'range' => array("manual", "automatic"),
            ),
            array(
                'nameserver1, nameserver2, nameserver3',
                'checkIP',
                'allowEmpty' => true,
            ),
            array('domain', 'match', 'pattern' => '/(?=^.{1,253}\.?$)^(?:[-_a-zA-Z0-9]{1,63}\.)*[-_a-zA-Z0-9]{1,63}\.?$/'),

            array('dhcp', 'safe'),
            array('disableSLAAC', 'safe'),
        );
    }

    public function attributeLabels()
    {
        return array(
            'netiface' => 'Network',

            'dhcp' => 'Configuration IPv4',

            'address' => 'Address',
            'netmask' => 'Netmask',
            'gateway' => 'Gateway',

            'ipv6Configuration' => 'Configuration IPv6',
            'ipv6Address' => 'Address',
            'ipv6Netmask' => 'Prefix length',
            'ipv6Gateway' => 'Gateway',

            'dnsConfiguration' => 'DNS Configuration',
            'nameserver1' => 'DNS Server 1',
            'nameserver2' => 'DNS Server 2',
            'nameserver3' => 'DNS Server 3',
            'domain' => 'DNS suffix',

            'disableSLAAC' => 'Disable IPv6 auto-configured addresses',
        );
    }
    public function requiredIf($attribute, $params)
    {
        if (
            !isset($params['var']) ||
            !isset($params['value']) ||
            $this->{$params['var']} == $params['value']
        ) {
            unset($params['var']);
            unset($params['value']);
            $ev = CValidator::createValidator(
                'required',
                $this,
                $attribute,
                $params
            );
            $ev->validate($this);
        }
    }
    function mask($adr, $msk)
    {
        $a = ip2long($adr);
        $m = ip2long($msk);
        return long2ip($a & $m);
    }
    function broad($adr, $msk)
    {
        $a = ip2long($adr);
        $m = ip2long($msk);

        return long2ip(($a & $m) + ~$m);
    }
    public function resolveLinkLocalAddress($name, $urlEncode = false) {
        if (($address = inet_pton($name)) && strlen($address) == 16 && 
            $address >= inet_pton('fe80::') && 
            $address <= inet_pton('febf::')
        ) {
            $name .= ($urlEncode ? urlencode('%') : '%') . 
                ($this->netiface == self::NET_WIFI ? 'wlan0' : 'eth0');
        }
        return $name;
    }
    protected function formatDNSServer($name)
    {
        return $this->resolveLinkLocalAddress($name);
    }
    protected function cleanDNSServer($name)
    {
        $val = explode("%", $name);
        return $val[0];
    }
    public function beforeValidate()
    {
        // insure that the DHCP is 0 or 1, and not "0" or "1"
        $this->dhcp = intval($this->dhcp);
        return parent::beforeValidate();
    }
    public function readIPConf($ifname, $type, $config)
    {
        $result = array();

        $str1 = "iface $ifname $type ";
        $iface = strpos($config, $str1);

        if ($iface === false) {
            $result['mode'] = "none";
            return $result;
        }
        $iface += strlen($str1);
        $conf = substr($config, $iface);
        $vals = explode("\n", $conf);

        if (trim($vals[0]) == "dhcp") {
            $result['mode'] = "dhcp";
        } elseif (trim($vals[0]) == "static") {
            $result['mode'] = "static";
            foreach ($vals as $name) {
                if (($n = strpos($name, "address ")) !== false) {
                    $result['address'] = trim(
                        substr($name, strlen("address") + $n)
                    );
                } elseif (($n = strpos($name, "netmask ")) !== false) {
                    $result['netmask'] = trim(
                        substr($name, strlen("netmask") + $n)
                    );
                } elseif (($n = strpos($name, "gateway ")) !== false) {
                    $result['gateway'] = trim(
                        substr($name, strlen("gateway") + $n)
                    );
                } elseif (strpos($name, "iface ") !== false) {
                    break;
                }
            }
        } elseif (trim($vals[0]) == "ppp") {
            $result['mode'] = "ppp";
            // nothing else to read, we always use the same ppp config
        } else {
            $this->addError(
                'dhcp',
                "Warning: Unknown inet state: '" . $vals[0] . "'"
            );
        }
        return $result;
    }
    public function loadIP()
    {
        if ($this->loaded) {
            return true;
        }

        if (!file_exists($this->fname)) {
            $this->addError(
                'file',
                'Failed loading IP config. ' . 'Please re-instal the firmware.'
            );
            return false;
        }

        $interface = file_get_contents($this->fname);
        if ($interface == "") {
            $this->addError(
                'file',
                'Configuration file is incomplete. ' .
                    'Please re-instal the firmware.'
            );
            return false;
        }

        $start = strpos($interface, $this->startTag);
        $end = strpos($interface, $this->endTag);
        if ($start === false || $end === false) {
            return false;
        }

        // should look for first auto interface to be
        // consistent with ifwatchdog
        $str1 = "auto ";
        $iface = strpos($interface, $str1, $start);
        $iface += strlen($str1);
        $line_end = strpos($interface, "\n", $iface);
        $ifname = substr($interface, $iface, $line_end - $iface);

        if ($ifname == "wlan0") {
            $this->netiface = self::NET_WIFI;
        } else {
            $this->netiface = self::NET_ETHERNET;
        }
        $this->currentNetiface = $this->netiface;

        $ipv4 = $this->readIPConf(
            $ifname,
            "inet",
            substr($interface, $start, $end - $start)
        );
        $this->dhcp = $ipv4['mode'] === "dhcp" ? 1 : 0;
        if (isset($ipv4['address'])) {
            $this->address = $ipv4['address'];
        }
        if (isset($ipv4['netmask'])) {
            $this->netmask = $ipv4['netmask'];
        }
        if (isset($ipv4['gateway'])) {
            $this->gateway = $ipv4['gateway'];
        }

        $ipv6 = $this->readIPConf(
            $ifname,
            "inet6",
            substr($interface, $start, $end - $start)
        );
        $this->ipv6Configuration = $ipv6['mode'];
        if ($this->ipv6Configuration === "static") {
            if (isset($ipv6['address'])) {
                $this->ipv6Address = $ipv6['address'];
            }
            if (isset($ipv6['netmask'])) {
                $this->ipv6Netmask = $ipv6['netmask'];
            }
            if (isset($ipv6['gateway'])) {
                $this->ipv6Gateway = $ipv6['gateway'];
            }
        }

        $this->loaded = true;
        return true;
    }

    public function saveIP()
    {
        if ($this->netiface == self::NET_WIFI) {
            $iface = "wlan0";
        } else {
            $iface = "eth0";
        }
        $conf = "auto $iface\niface $iface inet ";

        if (!$this->dhcp) {
            $conf .= "static\n";
            $conf .= "\taddress $this->address\n";
            $conf .=
                "\tnetwork " .
                $this->mask($this->address, $this->netmask) .
                "\n";
            $conf .= "\tnetmask $this->netmask\n";
            if ($this->gateway != "") {
                $conf .= "\tgateway $this->gateway\n";
            }
            $conf .=
                "\tbroadcast " .
                $this->broad($this->address, $this->netmask) .
                "\n";
            if (file_exists('/etc/network/eth-inet-dad')) {
                // not needed in firmware 4.0
                $conf .=
                    "\tpre-up [ ! -x /etc/network/eth-inet-dad ] || /etc/network/eth-inet-dad\n";
            }
        } else {
            $conf .= "dhcp\n";
        }

        if ($this->getIpv6('ipv6Configuration') === "static") {
            $conf .= "iface $iface inet6 static\n";
            $conf .= "\taddress {$this->getIpv6('ipv6Address')}\n";
            $conf .= "\tnetmask {$this->getIpv6('ipv6Netmask')}\n";
            if ($this->getIpv6('ipv6Gateway') !== "") {
                $conf .= "\tgateway {$this->getIpv6('ipv6Gateway')}\n";
                $conf .= "\tmetric 768\n";
            }
        } elseif ($this->getIpv6('ipv6Configuration') === "dhcp") {
            $conf .= "iface $iface inet6 dhcp\n";
        }

        if (!file_exists($this->fname)) {
            $this->addError(
                'address',
                'Failed opening network configuration, please try again.'
            );
            return false;
        }

        $interface = file_get_contents($this->fname);
        if ($interface == "") {
            $this->addError(
                'address',
                'Network configuration is empty, please try again.'
            );
            return false;
        }

        $start = strpos($interface, $this->startTag);
        $end = strpos($interface, $this->endTag);

        if ($start === false) {
            $start = strlen($interface);
        }
        if ($end === false) {
            $end = $start;
            $conf .= $this->endTag . "\n";
        }
        if (
            !Tools::save_file(
                $this->fname,
                substr_replace(
                    $interface,
                    $this->startTag . "\n" . $conf,
                    $start,
                    $end - $start
                )
            )
        ) {
            $this->addError(
                'file',
                'Failed saving network configuration, please try again.'
            );
            return false;
        }

        return true;
    }

    function loadDNS()
    {
        if (!file_exists($this->fnameDNS)) {
            $this->addError(
                'file',
                'Failed loading DNS config. ' . 'Please re-instal the firmware.'
            );
            return false;
        }
        $dns = file_get_contents($this->fnameDNS);
        if (trim($dns) === "") {
            $this->dnsConfiguration = "automatic";
            return;
        } else {
            $this->dnsConfiguration = "manual";
        }

        $tmp = $dns;
        $i = 1;
        $pos = strpos($tmp, "nameserver");
        while (!($pos === false)) {
            $pos += strlen("nameserver");
            $name = "nameserver" . $i;
            $i++;
            $this->$name = $this->cleanDNSServer(
                trim(substr($tmp, $pos, strpos($tmp, "\n", $pos) - $pos))
            );
            $tmp = substr($tmp, strpos($tmp, "\n", $pos));
            $pos = strpos($tmp, "nameserver");
        }

        $pos = strpos($dns, "domain ");
        if (!($pos === false)) {
            $pos += strlen("domain ");
            $this->domain = trim(
                substr($dns, $pos, strpos($dns, "\n", $pos) - $pos)
            );
        }
    }
    function saveDNS()
    {
        if ($this->dnsConfiguration == "manual") {
            $conf = "# manual configuration \n";
            if ($this->nameserver1 != "") {
                $conf .=
                    "nameserver " .
                    $this->formatDNSServer($this->nameserver1) .
                    "\n";
            }
            if ($this->nameserver2 != "") {
                $conf .=
                    "nameserver " .
                    $this->formatDNSServer($this->nameserver2) .
                    "\n";
            }
            if ($this->nameserver3 != "") {
                $conf .=
                    "nameserver " .
                    $this->formatDNSServer($this->nameserver3) .
                    "\n";
            }

            if ($this->domain != "") {
                $conf .= "domain $this->domain\n";
            }
        } else {
            $conf = "\n";
        }
        return Tools::save_file($this->fnameDNS, $conf);
    }
    function getSLAAC()
    {
        $disableSLAAC = false;
        if (file_exists($this->fnameSLAAC)) {
            $config = file_get_contents($this->fnameSLAAC);
            if (preg_match("/IPV6_AUTOCONF\s*=\s*(.*)/", $config, $matches)) {
                if ($matches[1] == 'no') {
                    $disableSLAAC = true;
                }
            }
        }
        return $disableSLAAC;
    }
    function loadSLAAC()
    {
        $this->disableSLAAC = $this->getSLAAC();
    }
    function saveSLAAC()
    {
        if ($this->disableSLAAC == $this->getSLAAC()) {
            return;
        }
        if (file_exists($this->fnameSLAAC)) {
            $config = file_get_contents($this->fnameSLAAC);
        } else {
            $config = "";
        }
        if (preg_match("/IPV6_AUTOCONF\s*=\s*(.*)/", $config, $matches)) {
            $config = preg_replace(
                "/IPV6_AUTOCONF\s*=\s*(.*)/",
                "IPV6_AUTOCONF=" . ($this->disableSLAAC ? "no" : "yes"),
                $config
            );
        } else {
            $config =
                "IPV6_AUTOCONF=" .
                ($this->disableSLAAC ? "no" : "yes") .
                "\n" .
                $config;
        }
        file_put_contents($this->fnameSLAAC, $config);

        Tools::addReason(
            "SLAAC " . ($this->disableSLAAC ? "disabled" : "enabled")
        );
    }
    function saveSafeModeConf()
    {
        // copy settings to failsafe-data

        if (
            !Tools::setSafeModeData(
                "netiface",
                $this->netiface == self::NET_WIFI ? "wlan" : "eth",
                false
            )
        ) {
            $this->addError(
                'file',
                'Failed saving network configuration for recovery console, please try again.'
            );
            return false;
        }

        if ($this->dhcp) {
            $netconf = "dhcp\n";
        } else {
            $netconf =
                "static=" .
                $this->address .
                ";" .
                $this->netmask .
                ";" .
                $this->broad($this->address, $this->netmask) .
                ";";
            if ($this->gateway != "") {
                $netconf .= $this->gateway;
            }
            $netconf .= ";";
            if ($this->nameserver1 != "") {
                $netconf .= $this->formatDNSServer($this->nameserver1);
            }
            $netconf .= ";";
            if ($this->nameserver2 != "") {
                $netconf .= $this->formatDNSServer($this->nameserver2);
            }
            $netconf .= ";";
            if ($this->nameserver3 != "") {
                $netconf .= $this->formatDNSServer($this->nameserver3);
            }
            $netconf .= ";";
            if ($this->domain != "") {
                $netconf .= $this->domain;
            }
            $netconf .= "\n";
        }
        if (!Tools::setSafeModeData("network", $netconf, false)) {
            $this->addError(
                'file',
                'Failed saving network configuration for recovery console, please try again.'
            );
            return false;
        }

        if ($this->getIpv6('ipv6Configuration') === "dhcp") {
            $netconfv6 = "dhcp\n";
        } elseif ($this->getIpv6('ipv6Configuration') === "static") {
            $netconfv6 =
                "static=" . $this->getIpv6('ipv6Address') . ";" . 
                $this->getIpv6('ipv6Netmask') . ";";
            if ($this->getIpv6('ipv6Gateway') != "") {
                $netconfv6 .= $this->getIpv6('ipv6Gateway');
            }
        } else {
            $netconfv6 = null;
        }
        if (!Tools::setSafeModeData("network6", $netconfv6, false)) {
            $this->addError(
                'file',
                'Failed saving network-IPv6 configuration for recovery console, please try again.'
            );
            return false;
        }

        if ($this->dnsConfiguration == "manual") {
            $dnsconf = "";
            if ($this->nameserver1 != "") {
                $dnsconf .= $this->formatDNSServer($this->nameserver1);
            }
            $dnsconf .= ";";
            if ($this->nameserver2 != "") {
                $dnsconf .= $this->formatDNSServer($this->nameserver2);
            }
            $dnsconf .= ";";
            if ($this->nameserver3 != "") {
                $dnsconf .= $this->formatDNSServer($this->nameserver3);
            }
            $dnsconf .= ";";
            if ($this->domain != "") {
                $dnsconf .= $this->domain;
            }
        } else {
            $dnsconf = null;
        }
        if (!Tools::setSafeModeData("dns", $dnsconf, false)) {
            $this->addError(
                'file',
                'Failed saving dns configuration for recovery console, please try again.'
            );
            return false;
        }

        $netopts = $this->disableSLAAC ? "no_ipv6_autoconf" : "";
        // if there are other option for the $netopts this should be separated by a ":"
        if (!Tools::setSafeModeData("netopts", $netopts, false)) {
            $this->addError(
                'file',
                'Failed saving netopts configuration for recovery console, please try again.'
            );
            return false;
        }

        return true;
    }

    function load()
    {
        $this->loadIP();
        $this->loadDNS();
        $this->loadSLAAC();
    }

    function save($refreshCredentials = true, $refreshNTP = true)
    {
        $this->saveIP();
        $this->saveDNS();
        $this->saveSLAAC();
        $this->saveSafeModeConf();

        if ($this->netiface != $this->currentNetiface) {
            $player = new PlayerConfig();
            $player->refresh($this);
            if ($refreshCredentials) {
                $credentials = new CredentialsConfig();
                $credentials->refresh($this);
            }
            if ($refreshNTP) {
                $ntp = new NTPSettings();
                $ntp->refresh($this);
            }
        }

        $name = $this->netiface == ipconfig::NET_WIFI ? "Wi-Fi" : "Ethernet";

        if ($this->dhcp) {
            $message = "<h4>DHCP will be used on $name.</h4>";
            $hostname = trim(file_get_contents("/etc/hostname"));
            $message .= "The unit is always reachable at <a href='http://$hostname.local/'>http://$hostname.local</a> from Zeroconf enabled computers on the local network.";
            Tools::addMessage($message);
        } else {
            Tools::addMessage(
                "New ip will be <a href='http://{$this->address}/'>{$this->address}</a> on $name"
            );
        }

        Tools::addReason(
            "network config change (" .
                $this->netiface .
                ", " .
                ($this->dhcp ? "IPv4 DHCP" : $this->address) .
                ", IPv6 " .
                $this->getIpv6('ipv6Configuration') .
                ")"
        );
    }

    public function checkConfig()
    {
        if ($this->dhcp) {
            return true;
        }

        $a = ip2long($this->address);
        $m = ip2long($this->netmask);
        if ($a === false || $a == -1) {
            $this->addError(
                'address',
                'Valid format xxx.xxx.xxx.xxx, where xxx is between 0-255'
            );
            return false;
        }
        if ($m === false || $m == -1) {
            $this->addError(
                'netmask',
                'Valid format xxx.xxx.xxx.xxx, where xxx is between 0-255'
            );
            return false;
        }
        if (($a & ip2long("255.0.0.0")) == ip2long("127.0.0.0")) {
            $this->addError(
                'address',
                '127.0.0.0/8 is reserved for loopback addresses'
            );
            return false;
        }

        if (($a & $m) == $a) {
            $this->addError(
                'address',
                'IP address is incompatible with netmask'
            );
            $this->addError('netmask', "");
            return false;
        }
        if (($a & ~$m) == ~$m) {
            $x = ~$m;
            $this->addError(
                'address',
                'Address and netmask combination is not allowed'
            );
            $this->addError('netmask', "");
            return false;
        }

        if ($this->gateway != "") {
            $g = ip2long($this->gateway);
            if (($a & $m) != ($g & $m)) {
                $this->addError(
                    'address',
                    'Gateway not in the same subnet as the address.'
                );
                $this->addError('netmask', "");
                $this->addError('gateway', "");

                return false;
            }
            if ($a == $g) {
                $this->addError(
                    'address',
                    'Gateway cannot be equal to the address.'
                );
                $this->addError('gateway', "");
                return false;
            }
            if (($g & $m) == $g || ($g & ~$m) == ~$m) {
                $this->addError('gateway', 'Gateway value not correct');
                return false;
            }
        }
        for ($i = 1; $i < 4; $i++) {
            $name = 'nameserver' . $i;
            if (
                !filter_var($this->$name, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)
            ) {
                // no validation on the DNS server name if they are IPv6 addresses
                continue;
            }
            $d = ip2long($this->$name);
            if ($this->$name != "") {
                if ($a == $d) {
                    $this->addError(
                        'address',
                        'DNS cannot be equal to the address.'
                    );
                    $this->addError($name, "");
                    return false;
                }
                if (($a & $m) == ($d & $m)) {
                    if (($d & $m) == $d) {
                        $this->addError($name, 'Reserved addresses');
                        return false;
                    }
                    if (($d & ~$m) == ~$m) {
                        $this->addError($name, 'Reserved addresses');
                        return false;
                    }
                }
            }
        }
        return true;
    }

    public function setIpv6($netiface, $values) {
        if (!is_array($values)) {
            return;
        }
        $this->setAttributes($values);
        $this->ipv6[$netiface] = $values;
        $this->ipv6[$netiface]['ipv6Configuration'] = 
            $this->ipv6Configuration;
    }

    protected function getIpv6($attribute) {
        return isset($this->ipv6[$this->netiface][$attribute]) ? 
            $this->ipv6[$this->netiface][$attribute] : $this->$attribute;
    }
}
