﻿var system = require('system');
var webserver = require('webserver');
var webpage = require('webpage');
var fs = require('fs');

function usage() {
    console.log('Usage: webservice.js <portnumber> <png_folder>');
    phantom.exit(1);
}

if (system.args.length < 3)
    usage();

var desiredPortNumber = 0, actualPortNumber = 0;
if ( system.args[1] == 'auto' ) {
    desiredPortNumber = 0;
} else {
    desiredPortNumber = parseInt( system.args[1] );
    if ( !isFinite( desiredPortNumber ) || desiredPortNumber <= 0 || desiredPortNumber > 65536 )
        usage();
}

var pngFolder = system.args[2];
if (!pngFolder.length)
    usage();
if (pngFolder.substring(pngFolder.length - 1) == '/' || pngFolder.substring(pngFolder.length - 1) == '\\')
    pngFolder = pngFolder.substring(0, pngFolder.length - 1);
if (!fs.isDirectory(pngFolder))
    usage();
var list = fs.list(pngFolder);
for (var i = 0; i < list.length; i++)
    if (list[i].match(/^\.?[0-9]+\.png$/))
        try { fs.remove(pngFolder + '/' + list[i]); } catch(e) {}

var server = webserver.create();
var rePath = /^\/([a-z0-9A-z_]+)\?(.*)$/;
var reUserAgent = /raperca\/([a-zA-Z0-9_]+)/;
var reHeader = /^([!#$&'*+.^`|~_0-9A-Za-z-]+):[ \t]*(.*)$/;

var pageById = {};

function failed(reason) {
    return {
        success: false,
        reason: reason
    };
}

function success() {
    return {
        success: true
    };
}

function render(id) {
    var page = pageById[id].page;
    try {
        page.render(pngFolder + '/.' + id + '.png', { format: 'png', quality: 80 });   // Do not waste too much time in zlib
    } catch ( e ) {
        console.log( 'Render exception: ' + e );
    }
    try {
        fs.remove( pngFolder + '/' + id + '.png' );
    } catch ( e ) {
    }
    try {
        fs.move( pngFolder + '/.' + id + '.png', pngFolder + '/' + id + '.png' );
    } catch ( e ) {
        console.log( 'File move exception: ' + e );
    }
}

var observeTimeNow = 0;

function mutate(id) {
    if (pageById[id].loaded && !pageById[id].pending)
        pageById[id].lastModified = observeTimeNow;
}

function observe(id, force) {
    var mutated = pageById[id].page.evaluate(function () {
        var mutated = window.spinetix_observer_mutated;
        window.spinetix_observer_mutated = false;
        if (mutated !== false) {
            if (!window.spinetix_observer) {
                window.spinetix_observer = new MutationObserver(function (mutations) {
                    window.spinetix_observer_mutated = true;
                    window.spinetix_observer.disconnect();
                });
            }
            window.spinetix_observer.observe(document, { childList: true, attributes: true, characterData:true, subtree: true });
        }
        return mutated;
    });
    //console.log('mutated=' + mutated + ' loaded=' + pageById[id].loaded + ' pending=' + pageById[id].pending);
    if (mutated || force) {
        pageById[id].lastModified = observeTimeNow;
    } else if (pageById[id].lastModified + 1 == observeTimeNow) {
        console.log(id + " NEW RENDER at " + observeTimeNow);
        render(id);
    }
}

function observeTimeout() {
    ++observeTimeNow;
    for (var id in pageById)
        if (pageById[id].loaded && !pageById[id].pending)
            observe(id);
}

function beforeLoad(id) {
    if (pageById[id]) {
        pageById[id].loaded = false;
        pageById[id].lastModified = 0;
    }
}

function afterLoad(id, status) {
    if (pageById[id] && !pageById[id].loaded) {
        console.log(id + ' LOADED status=' + status);
        pageById[id].loaded = true;
        pageById[id].lastModified = 0;
        render(id);
        if (!pageById[id].pending)
            observe(id);
    }
}

function resourceRequested(id, requestData, networkRequest) {
    if (pageById[id]) {
        var url = requestData.url;
        if (url.length > 120)
            url = url.substring(0, 117) + '...';
        console.log(id + ' --> (#' + requestData.id + ') ' + requestData.method + ' ' + url);
        pageById[id].pending += 1;
    }
}

function resourceReceived(id, response) {
    if (response.stage == 'end' && pageById[id]) {
        if ( response.status !== null ) {
            console.log( id + ' <-- (#' + response.id + ') ' + response.status + ' ' + response.statusText + ' ' + response.contentType );
            if ( response.id == 1 && response.status >= 400 )
                pageById[id].page.setContent( '<html><body><h1>' + response.status + ' ' + response.statusText + '</h1></body></html>', 'local:error' );
        }
        pageById[id].pending -= 1;
        if (pageById[id].pending <= 0) {
            pageById[id].pending = 0;
            if (pageById[id].loaded)
                observe(id, true);
        }
    }
}

function resourceError(id, error) {
    if (pageById[id])
        console.log(id + ' <-- (#' + error.id + ') Error ' + error.errorCode + ' ' + error.errorString);
}

function resourceTimeout(id, request) {
    if (pageById[id])
        console.log(id + ' <-- (#' + request.id + ') Timeout ' + request.errorCode + ' ' + request.errorString);
}

function setViewport(page, args) {
    var zoom = args.zoom || 1;
    var windowWidth = args.windowWidth || 960;
    var windowHeight = args.windowHeight || 960;
    var renderWidth = args.renderWidth || windowWidth;
    var renderHeight = args.renderHeight || windowHeight;
    page.viewportSize = {
        width: Math.max(windowWidth * zoom, renderWidth),
        height: Math.max(windowHeight * zoom, renderHeight)
    };
    page.zoomFactor = zoom;
    page.clipRect = {
        left: args.scrollX * zoom,
        top: args.scrollY * zoom,
        width: renderWidth,
        height: renderHeight
    };
    /* This causes more problems than it solves (mainly it would allow dynamically generated content like twitter to be aware of the scroll intention, but it somehow causes the bottom of the page to be truncated)
    page.scrollPosition = {
        left: args.scrollX,
        top: args.scrollY
    };
    */
}

var actions = {
    open: function (args, html) {
        if (!args.id)
            return failed( 'Missing id');
        if (pageById[args.id])
            return failed('Reusing id');
        var page = webpage.create();
        pageById[args.id] = {
            page: page,
            loaded: false,
            lastModified: 0,
            pending: 0
        };
        page.onLoadStarted = function () {
            beforeLoad(args.id);
        };
        page.onLoadFinished = function (status) {
            afterLoad(args.id, status);
        }
        page.onResourceRequested = function (requestData, networkRequest) {
            resourceRequested(args.id, requestData, networkRequest);
        }
        page.onResourceError = function (error) {
            resourceError(args.id, error);
        }
        page.onResourceReceived = function (response) {
            resourceReceived(args.id, response);
        }
        page.onResourceTimeout = function (request) {
            resourceTimeout(args.id, request);
        }
        if ( 'userAgent' in args )
            page.settings.userAgent = args.userAgent;
        if ( 'customHeaders' in args ) {
            var map = {};
            var headers = args.customHeaders.split( /[\r\n]+/ );
            for ( var i = 0; i < headers.length; i++ ) {
                var h = reHeader.exec( headers[i] );
                if ( h )
                    map[h[1]] = h[2];
            }
            page.customHeaders = map;
        }
        var allowSameOrigin = true, allowTopNavigation = true, allowForms = true, allowPopups = false, allowScripts = true, allowPointerLock = false;
        page.settings.javascriptEnabled = allowScripts;
        page.settings.localToRemoteUrlAccessEnabled = allowSameOrigin;
        page.settings.webSecurityEnabled = !allowSameOrigin;
        page.navigationLocked = false;  // As weird as it sounds, navigationLocked actually prevents a page.open()
        if ( args.uri ) {
            page.settings.userName = 'user' in args ? args.user : '';
            page.settings.password = 'pass' in args ? args.pass : '';
            page.open( args.uri );
        } else {
            page.setContent( html, args.base );
        }
        page.navigationLocked = !allowTopNavigation;
        setViewport( page, args );
        return success();
    },

    close: function (args) {
        if (!args.id)
            return failed('Missing id');
        if (!pageById[args.id])
            return failed('Id not used');
        var page = pageById[args.id].page;
        delete pageById[args.id];
        page.close();
        try { fs.remove(pngFolder + '/' + args.id + '.png'); } catch (e) { }
        console.log(args.id + ' CLOSED');
        return success();
    },

    resize: function (args) {
        if (!args.id)
            return failed('Missing id');
        if (!pageById[args.id])
            return failed('Id not used');
        var page = pageById[args.id].page;
        setViewport(page, args);
        mutate(args.id);
        return success();
    },

    mouseDown: function (args) {
        if (!args.id)
            return failed('Missing id');
        if (!pageById[args.id])
            return failed('Id not used');
        var page = pageById[args.id].page;
        var x = parseFloat(args.x) || 0;
        var y = parseFloat(args.y) || 0;
        var button = parseInt(args.button) || 0;
        page.sendEvent('mousedown', page.clipRect.left + x, page.clipRect.top + y, button == 1 ? 'middle' : button == '2' ? 'right' : 'left');
        mutate(args.id);
        return success();
    },

    mouseUp: function (args) {
        if (!args.id)
            return failed('Missing id');
        if (!pageById[args.id])
            return failed('Id not used');
        var page = pageById[args.id].page;
        var x = parseFloat(args.x) || 0;
        var y = parseFloat(args.y) || 0;
        var button = parseInt(args.button) || 0;
        page.sendEvent('mouseup', page.clipRect.left + x, page.clipRect.top + y, button == 1 ? 'middle' : button == '2' ? 'right' : 'left');
        mutate(args.id);
        return success();
    },

    click: function (args) {
        if (!args.id)
            return failed('Missing id');
        if (!pageById[args.id])
            return failed('Id not used');
        var page = pageById[args.id].page;
        var x = parseFloat(args.x) || 0;
        var y = parseFloat(args.y) || 0;
        var button = parseInt(args.button) || 0;
        page.sendEvent( 'click', page.clipRect.left + x, page.clipRect.top + y, button == 1 ? 'middle' : button == '2' ? 'right' : 'left' );
        mutate(args.id);
        return success();
    },

    keyDown: function (args) {
        if (!args.id)
            return failed('Missing id');
        if (!pageById[args.id])
            return failed('Id not used');
        var page = pageById[args.id].page;
        var code = parseInt(args.code) || page.event.key[args.code] || 0;
        var flags = parseInt(args.flags) || 0;
        page.sendEvent('keydown', code, null, null, flags);
        mutate(args.id);
        return success();
    },

    keyUp: function (args) {
        if (!args.id)
            return failed('Missing id');
        if (!pageById[args.id])
            return failed('Id not used');
        var page = pageById[args.id].page;
        var code = parseInt(args.code) || page.event.key[args.code] || 0;
        var flags = parseInt(args.flags) || 0;
        page.sendEvent('keyup', code, null, null, flags);
        mutate(args.id);
        return success();
    },

    keyPress: function (args) {
        if (!args.id)
            return failed('Missing id');
        if (!pageById[args.id])
            return failed('Id not used');
        var page = pageById[args.id].page;
        page.sendEvent('keypress', 'text' in args ? args.text : '', null, null, 0);
        mutate(args.id);
        return success();
    }
};

var portNumberMin = 8899, portNumberMax = 8909, service = null;
if ( desiredPortNumber )
    portNumberMin = portNumberMax = desiredPortNumber;

for ( var actualPortNumber = portNumberMin; actualPortNumber < portNumberMax; actualPortNumber++ ) {
    service = server.listen( '127.0.0.1:' + actualPortNumber, { keepAlive: true }, function ( request, response ) {
        function fail( reason ) {
            console.log( 'Request failed: ' + reason );
            var body = JSON.stringify( { success: false, reason: reason }, null, 4 );
            response.statusCode = 401;
            response.headers = {
                'Cache': 'no-cache',
                'Content-Type': 'text/json',
                'Connection': 'Keep-Alive',
                'Keep-Alive': 'timeout=60, max=600',
                'Content-Length': body.length
            };
            response.write( body );
            response.close();
            return null;
        }
        var ua = reUserAgent.exec( request.headers['User-Agent'] );
        if ( !ua || ua[1] != 'phantom1' )
            return fail( 'Unknown user agent:' + request.headers['User-Agent'] );
        var html = null;
        if ( request.method == 'POST' && request.headers['Content-Type'] == 'text/html' ) {
            html = request.post;
        }
        var r = rePath.exec( request.url );
        if ( !r )
            return fail( 'Malformed url:' + request.url );
        var action = r[1], args = {};
        var list = r[2].split( '&' );
        for ( var i = 0; i < list.length; i++ ) {
            var eq = list[i].indexOf( '=' );
            if ( eq >= 0 )
                args[decodeURIComponent( list[i].substring( 0, eq ) )] = decodeURIComponent( decodeURIComponent( list[i].substring( eq + 1 ) ) );
            else
                args[decodeURIComponent( list[i] )] = true;
        }
        if ( !actions[action] )
            return fail( 'Unknown action:' + action );
        var result;
        try {
            result = actions[action]( args, html );
        } catch ( e ) {
            result = {
                success: false,
                reason: 'Exception: ' + e
            };
        }
        if ( !result.success )
            return fail( result.reason );
        var body = JSON.stringify( result, null, 4 );
        response.statusCode = 200;
        response.headers = {
            'Cache': 'no-cache',
            'Content-Type': 'text/json',
            'Connection': 'Keep-Alive',
            'Keep-Alive': 'timeout=60, max=600',
            'Content-Length': body.length
        };
        response.write( body );
        response.close();
    } );
    if ( service )
        break;
}

if (service) {
    console.log('Now listening on port ' + actualPortNumber + ', png output to ' + pngFolder);
    window.setInterval(observeTimeout, 50);
} else {
    console.log('Error: Could not create web server on port ' + desirecPortNumber);
    phantom.exit();
}
