<?php
class WiredAuth extends CFormModel
{
    static $path = "/var/lib/ead/";

    public $network = "default";
    public $method = "8021x";

    public $username;
    public $password;
    public $pem;

    public $file;
    public $fileBase64;

    /**
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        return array(
            array('network', 'in', 'range' => array("default")),
            array('username', 'safe'),
            array('password', 'safe'),
            array('pem', 'safe'),
            array('network', 'safe'),
            array('file', 'safe'),
            array('fileBase64', 'safe'),
            array('method', 'in', 'range' => array("8021x")),
        );
    }

    public function attributeLabels()
    {
        return array();
    }
    public function createFile()
    {
        $file = "# Automatically generate by Control Center\n";
        if ($this->method === "8021x") {
            $file .= "[Security]\n";
            $file .= "EAP-Method=PEAP\n";
            $file .= "EAP-Identity=" . $this->username . "\n";
            $file .= "EAP-PEAP-Phase2-Method=MSCHAPV2\n";
            $file .= "EAP-PEAP-Phase2-Identity=" . $this->username . "\n";
            $file .= "EAP-PEAP-Phase2-Password=" . $this->password . "\n";
            if ($this->pem) {
                $file .= "[@pem@cacert]\n";
                $file .= $this->pem . "\n";
            }
        }
        return $file;
    }

    public function save()
    {
        if ($this->fileBase64) {
            $this->file = base64_decode($this->fileBase64);
        }
        if (!$this->file) {
            if (
                $this->method === "8021x" &&
                (!$this->password || !$this->username)
            ) {
                return $this->addError(
                    "file",
                    "A configuration file is required"
                );
            }
            $this->file = $this->createFile();
        }
        $name = WiredAuth::getFileName($this->network, $this->method);
        if (!file_exists(WiredAuth::$path . $name)) {
            Tools::addReason("Wired authentication enabled");
        }
        file_put_contents(WiredAuth::$path . $name, $this->file);
        if (
            !Tools::setSafeModeData(
                "ead/" . $name,
                WiredAuth::$path . $name,
                true
            )
        ) {
            $this->addError(
                'file',
                'Failed saving wired auth configuration for recovery console, please try again.'
            );
            return false;
        }
    }
    public function load()
    {
        $this->file = WiredAuth::loadFile($this->network, $this->method);
    }

    public static function getFileName($network, $method)
    {
        if (preg_match("/[^\w\- ]/", $network)) {
            $encodedSsid = "=";
            for ($pos = 0; $pos < strlen($network); $pos++) {
                $byte = substr($network, $pos);
                $encodedSsid .= dechex(ord($byte));
            }
            $network = $encodedSsid;
        }
        return $network . "." . $method;
    }
    public static function checkFileName($name)
    {
        return substr($name, -5) === "8021x";
    }
    public static function decodeFileName($name)
    {
        $data = explode(".", $name);
        if (substr($data[0], 0, 1) === "=") {
            $network = "";
            for ($pos = 1; $pos < strlen($data[0]); $pos += 2) {
                $byte = substr($data[0], $pos, 2);
                $network .= chr(hexdec($byte));
            }
            $data[0] = $network;
        }
        return $data;
    }

    public static function saveFile($network, $method, $file)
    {
        $name = WiredAuth::getFileName($network, $method);
        file_put_contents(WiredAuth::$path . $name, $file);
        return $name;
    }
    public static function loadFile($network, $method)
    {
        $name = WiredAuth::getFileName($network, $method);
        if (!file_exists(WiredAuth::$path . $name)) {
            return null;
        }

        return file_get_contents(WiredAuth::$path . $name);
    }
    public static function clearAll()
    {
        $dir = scandir(WiredAuth::$path);
        $modified = false;
        foreach ($dir as $file) {
            if (!WiredAuth::checkFileName($file)) {
                continue;
            }
            $modified = true;
            unlink(WiredAuth::$path . $file);
        }
        if ($modified) {
            Tools::addReason("Wired authentication disabled");
        }
        Tools::setSafeModeData("ead", null);
    }
    public static function listAll()
    {
        $list = array();
        $dir = scandir(WiredAuth::$path);

        foreach ($dir as $file) {
            if (!WiredAuth::checkFileName($file)) {
                continue;
            }
            $data = WiredAuth::decodeFileName($file);
            $conf = new WiredAuth();
            $conf->network = $data[0];
            $conf->method = $data[1];
            $list[] = $conf;
        }
        return $list;
    }
}
