/*!
 * jSignage.CustomEffects
 * https://www.spinetix.com/
 * Copyright SpinetiX S.A.
 * Released under the GPL Version 2 license.
 *
 * A jSignage library add-on implementing effects that are not provided "out of the box" by jSignage.
 * 
 * $Date: 2020-08-21 18:50:50 +0000 (Fri, 21 Aug 2020) $
 * $Revision: 36656 $
 */
( function () {

var version = new String( "1.1.0" );
version.major = 1;
version.minor = 1;
version.revision = 0;

/**
 *  Calculates a point on a circle, given a center point and a radius. 
 *  The angle can be provided or randomly selected.
 *  @param {number} cx
 *  @param {number} cy
 *  @param {number} radius
 *  @param {number=} angle Optional angle value in radians.
 *  @return {Object} An object defining the selected point.
 */
function _getPointOnCircle( cx, cy, radius, angle ){
    if ( angle === undefined ) {
        angle = 2 * Math.PI * jSignage.randomSafeNumber();
    }
    
    return { 
        angle: angle,
        x : cx + Math.cos( angle ) * radius, 
        y : cy + Math.sin( angle ) * radius
    };
}

/**
 * Implementation of CustomEffects jSignage add-on.
 */
jSignage.CustomEffects = {
    
    version: version,
    
    /**
     *  Applies the Ken Burns effect on the layer provided as input.
     *  The Ken Burns effect is a type of panning and zooming effect used usually in video production from still imagery.
     *  For more details about it, see https://en.wikipedia.org/wiki/Ken_Burns_effect .
     *  @param {Object} jsonLayer Object describing the layer to apply the Ken Burns effect on.
     *  @param {Object} jsonParams Widget parameters.
     *  @param {Object=} data Optional data to modify the jsonLayer.
     *  @return {Object} The layer with a chained effectIn implementing the Ken Burns effect.
     */
    KenBurns: function( jsonLayer, jsonParams, data ) {
        var factorStart   = 1 / Math.sqrt( ( jsonParams.cropSizeAtStart || 90 ) / 100 ), 
            factorEnd     = 1 / Math.sqrt( ( jsonParams.cropSizeAtEnd   || 70 ) / 100 ), 
            randomZoom    = jsonParams.randomZoom || false,
            randomSeed    = jsonParams.randomSeed || 5737,
            slideDur      = jSignage.durInSeconds( jsonLayer.args.dur, jSignage.durInSeconds( jsonParams.defaultSlideDur ) ),
            transitionDur = ( jsonParams.defaultTransition ? $.durInSeconds( jsonParams.defaultTransition.args.dur ) : 0 ),
            layer = ( !data ? $.uncan( jsonLayer ) : $.uncan( jsonLayer, 'data', data ) );

        
        return layer.effectIn( 
            function( trigger, inner, width, height, x, y, bbw, bbh, FX1, FY1, FX2, FY2 ) {
                var xStart, yStart, xEnd, yEnd, f, cx, cy, rx, ry, radius,
                    point = {}, 
                    randomSN = jSignage.randomSafeNumber( randomSeed ),
                    angle = 2 * Math.PI * randomSN;
                
                // check if random motion is selected 
                if ( randomZoom && randomSN < 0.5 ) {
                    f = factorStart;
                    factorStart = factorEnd; 
                    factorEnd = f; 
                }
                
                // calculate start animation center point
                cx = ( width  / 2 ) * ( factorStart - 1 );
                cy = ( height / 2 ) * ( factorStart - 1 ); 
                
                // get a point on a cercle around it to create the pan
                f = ( 1 - 1 / factorStart );
                rx = f * width  - cx;
                ry = f * height - cy;
                radius = ( rx > ry ? ry : rx );
                point = _getPointOnCircle( cx, cy, radius, angle );
                
                // set the start animation point                
                xStart = point.x;
                yStart = point.y;
                
                if ( factorEnd != factorStart ) {
                    // calculate end animation center point
                    cx   = width  / 2 * ( factorEnd   - 1 ); 
                    cy   = height / 2 * ( factorEnd   - 1 ); 
                    
                    // get a point on a cercle around it to create the pan
                    f = ( 1 - 1 / factorEnd );
                    rx = f * width  - cx;
                    ry = f * height - cy;
                    
                    radius = ( rx > ry ? ry : rx );
                    point = _getPointOnCircle( cx, cy, radius );
                } else {
                    // get the opposite point on the cercle to create the pan
                    point = _getPointOnCircle( cx, cy, radius, Math.PI + point.angle );
                }
                
                // set the end animation point
                xEnd = point.x;
                yEnd = point.y;                
               
                // add translate animation
                jSignage.svgAnimation( this, 'animateTransform', {
                  attributeName: 'transform',
                  type: 'translate',
                  additive: 'sum',
                  fill: 'freeze',
                  from: (-xStart)+','+(-yStart),
                  to: (-xEnd)+','+(-yEnd),
                  begin: trigger,
                  dur: slideDur + transitionDur
                });
                
                // add scale animation (for zoom)
                jSignage.svgAnimation( this, 'animateTransform', {
                  attributeName: 'transform',
                  type: 'scale',
                  additive: 'sum',
                  fill: 'freeze',
                  from: factorStart,
                  to: factorEnd,
                  begin: trigger,
                  dur: slideDur + transitionDur
                });
            }
        );
    }
}

} ) ( );
