<?php
class ApplicationsStreaming extends Streaming {

    public function attributeLabels() {
        $labels = parent::attributeLabels();
        $labels['file'] = 'Configuration';
        return $labels;
    }

    public function rules() {
        $rules = parent::rules();
        $rules[] = array(
            'resolution', 'validateResolution', 'on' => array(self::IPTV, 
            self::RTSP, self::RTMP));
        $rules[] = array(
            'frameRate', 'validateFrameRate', 'on' => array(self::IPTV, 
            self::RTSP, self::RTMP));
        $rules[] = array(
            'file', 'file', 'on' => self::CUSTOM, 'allowEmpty' => false, 
            'types' => 'xml');
        return $rules;
    }

    public function validateResolution($attribute, $params) {
        if (is_array($this->$attribute) || !isset(
            $this->getResolutions()[$this->$attribute])) {
            $this->addError($attribute, 'Invalid resolution.');
        }
    }

    public function validateFrameRate($attribute, $params) {
        if (is_array($this->$attribute) || !isset(
            $this->getFrameRates()[$this->$attribute])) {
            $this->addError($attribute, 'Invalid frame rate.');
        }
    }

    public function load() {
        parent::load();
        $screenInfo = new ScreenInfo();
        $screenInfo->load();
        $this->loadResolutions($screenInfo);
        $this->loadFrameRates($screenInfo);
    }

    public function save() {
        if ($this->protocol == self::CUSTOM) {
            if (($file = CUploadedFile::getInstance($this, 'file')) === null) {
                $this->addError('file', 'No configuration file uploaded.');
                return;
            };                
            if (!($XML = @file_get_contents($file->getTempName()))) {
                $this->addError(
                    'file', 'Configuration file cannot be read or is empty.');
                return;
            };
            $this->_XML = $XML;
        }
        parent::save();
        if (($bitRate = $this->getBitRate())) {
            Tools::addMessage(
                'Bit Rate: ' . intdiv($bitRate, 1000) . ' kbit/s');
        }
        Tools::addReason( 'streaming configuration change' );
    }

    public function getProtocols() {
        return array_combine( $this->_protocols, $this->_protocols );
    }

    public function getResolutions() {
        return $this->_resolutions;
    }

    public function getFrameRates() {
        return $this->_frameRates;
    }

    public function getQualities() {
        return array_combine( $this->_qualities, $this->_qualities );
    }

    public function getCodecs() {
        return array_combine( $this->_codecs, $this->_codecs );
    }

    public $file = null;

    private function loadResolutions($screenInfo) {
        list($maxWidth, $maxHeight) = explode('x', $screenInfo->resolution);
        $maxSize = max($maxWidth, $maxHeight);
        if (($size = self::maxSize()) < $maxSize) {
            $maxWidth = self::roundEven(($maxWidth * $size) / $maxSize);
            $maxHeight = self::roundEven(($maxHeight * $size) / $maxSize);
            $maxSize = $size;
        }
        $resolutions = array();
        $width = $maxWidth;
        $height = $maxHeight;
        $size = $maxSize;
        do {
            $resolutions[$size] = $width . 'x' . $height;
            $width = self::roundEven($width / 2);
            $height = self::roundEven($height / 2);
            $size = self::roundEven($size / 2);
        } while ($size >= 640);
        $width = self::roundEven($maxWidth / 1.5);
        $height = self::roundEven($maxHeight / 1.5);
        $size = self::roundEven($maxSize / 1.5);
        while ($size >= 640) {
            $resolutions[$size] = $width . 'x' . $height;
            $width = self::roundEven($width / 2);
            $height = self::roundEven($height / 2);
            $size = self::roundEven($size / 2);
        }
        if ($this->resolution && ($size = max(explode(
            'x', $this->resolution))) < $maxSize) {
            if (!isset($resolutions[$size])) {
                $width = self::roundEven(($maxWidth * $size) / $maxSize);
                $height = self::roundEven(($maxHeight * $size) / $maxSize);
                $resolutions[$size] = $width . 'x' . $height;
            }
            $this->resolution = $resolutions[$size];
        } else {
            $this->resolution = $resolutions[$maxSize];
        }
        krsort($resolutions);
        $resolutions = array_values ($resolutions);
        $this->_resolutions = array_combine ($resolutions, $resolutions);
    }

    private static function maxSize() {
        $cpuinfo = file_get_contents('/proc/cpuinfo');
        $family = preg_match(
            '/\s+cpu family\s*:\s+(\w+)/', $cpuinfo, $match) ? $match[1] : '';
        $model = preg_match(
            '/\s+model\s*:\s+(\w+)/', $cpuinfo, $match) ? $match[1] : '';
        // Limit Apollo Lake (family 6, model 92) and Gemini Lake/Gemini Lake 
        // Refresh (family 6, model 122) platforms to 1920 pixels. 
        return $family == '6' && ($model == '92' || $model == '122') ? 1920 : 
            3840;
    }

    private static function roundEven($float) {
        return round($float / 2) * 2;
    }

    private function loadFrameRates($screenInfo) {
        $refresh = floatval($screenInfo->refresh);
        $frameRates = array(1 => ceil($refresh));
        if ($frameRates[1] >= 50)
            $frameRates[2] = ceil($refresh / 2);
        if (!empty($this->frameRate)) {
            $divider = ceil($refresh / $this->frameRate);
            if (!isset($frameRates[$divider]))
                $frameRates[$divider] = ceil($refresh / $divider);
            $this->frameRate = $frameRates[$divider];
        } else {
            $this->frameRate = end($frameRates);
        }
        $frameRates = array_values ($frameRates);
        $this->_frameRates = array_combine($frameRates, $frameRates);
    }

    private $_resolutions = array();
    private $_frameRates = array();
}
