<?php
class SerialConfig extends CFormModel
{
    static $confpathsys = "/usr/share/raperca/comportfsm/";
    static $confpathusr = "/etc/raperca/comportfsm/";
    var $powersavefile = "power_save.fsm";
    var $simplefile = "simple.fsm";

    var $baudRateSet = array(
        110,
        300,
        600,
        1200,
        2400,
        4800,
        9600,
        14400,
        19200,
        38400,
        57600,
        115200,
        128000,
        256000,
    );
    var $paritySet = array("none", "even", "odd");
    var $flowControlSet = array("none", "XonXoff", "RTS/CTS", "RS485");
    var $byteSizeSet = array("7", "8");
    var $stopBitsSet = array("0", "1", "2");

    var $href = "";

    var $portNumber = 1;

    var $enableSerial = null;
    public $device;
    public $name;
    var $baudRate = 9600;
    var $byteSize = 8;
    var $stopBits = 1;
    var $parity = "none";
    var $flowControl = "none";

    var $uploadedfile;

    var $target = "";

    var $modified = null;

    var $playerConf = null;
    protected $_devices = array();
    protected $_simpleCommands = null;
    public function init()
    {
        if (Yii::app()->device->model === 'ikebana') {
            $this->portNumber = 2;
        }
    }
    public function rules()
    {
        $res = array(
            array(
                'href',
                'match',
                'pattern' => '/^(\[disabled\]|\[simple\]|[^\.]*.fsm)$/',
                'on' => 'io, backup',
            ),
            array(
                'target',
                'match',
                'pattern' => '/^[^\.]*.fsm$/',
                'on' => 'dump, delete',
            ),
            array('enableSerial', 'forceInt', 'on' => 'io, backup'),
            array('device', 'validateDevice', 'on' => 'io'),
            array('name', 'validateName', 'on' => 'io'),
            array('device, name', 'safe', 'on' => 'backup'),
            array(
                'baudRate',
                'in',
                'range' => array(
                    "110",
                    "300",
                    "600",
                    "1200",
                    "2400",
                    "4800",
                    "9600",
                    "14400",
                    "19200",
                    "38400",
                    "57600",
                    "115200",
                    "128000",
                    "256000",
                ),
                'on' => 'io, backup',
            ),
            array(
                'byteSize',
                'in',
                'range' => array(7, 8),
                'on' => 'io, backup',
            ),
            array(
                'stopBits',
                'numerical',
                'min' => 0,
                'max' => 2,
                'integerOnly' => true,
                'on' => 'io, backup',
            ),
            array(
                'parity',
                'in',
                'range' => array("none", "even", "odd"),
                'on' => 'io, backup',
            ),
            array('uploadedfile', 'file', 'types' => 'fsm', 'on' => 'upload'),
            array('simpleCommands', 'safe'),
        );
        return $res;
    }
    public function attributeLabels()
    {
        return array(
            'href' => 'Protocols',
            'setings' => "Settings",
            'target' => 'Protocols',
            'enableSerial' => 'Enable serial port',
            'device' => 'Device',
            'baudRate' => 'Baud rate',
            'byteSize' => 'Data bits',
            'parity' => 'Parity',
            'stopBits' => 'Stop bits',
            'simpleFSMCommand.name' => "Command",
            'simpleFSMCommand.value' => "Data",
        );
    }
    public function forceInt($attribute, $params)
    {
        if ($this->$attribute !== null) {
            $this->$attribute = intval($this->$attribute);
        }
    }
    public function validateDevice($attribute, $params)
    {
        if (Yii::app()->device->model === 'ikebana') {
            if ($this->$attribute !== null) {
                $this->addError($attribute, 'Invalid device.');
            }
        } else {
            if (!is_string($this->$attribute) 
                || !isset($this->devices[$this->$attribute])
            ) {
                $this->addError($attribute, 'Invalid device.');
            }
        }
    }
    public function validateName($attribute, $params)
    {
        if (Yii::app()->device->model === 'ikebana') {
            if ($this->$attribute !== null) {
                $this->addError($attribute, 'Invalid device.');
            }
        } else {
            if ($this->$attribute !== $this->devices[$this->device]) {
                $this->addError($attribute, 'Invalid device.');
            }
        }
    }

    public function getDevices()
    {
        return $this->_devices;
    }
    function toList($in)
    {
        $name = $in . 'Set';
        $ret = array();
        foreach ($this->$name as $val) {
            $ret[$val] = $val;
        }
        return $ret;
    }
    function getSettings()
    {
        if ($this->href == "[disabled]" || $this->href == "") {
            return 'disabled';
        } elseif (
            $this->href == "[simple]" ||
            $this->href == "usr/" . $this->simplefile
        ) {
            return 'simple';
        } else {
            return 'advanced';
        }
    }
    function getRealPath($href)
    {
        if (substr($href, 0, 3) == "sys") {
            return self::$confpathsys . substr($href, 4);
        } elseif (substr($href, 0, 3) == "usr") {
            return self::$confpathusr . substr($href, 4);
        } else {
            return "";
        }
    }
    function getBackupType()
    {
        if ($this->href == "[disabled]" || $this->href == "") {
            return "disabled";
        }
        if (
            $this->href == "[simple]" ||
            $this->href == "usr/" . $this->simplefile
        ) {
            return "simple";
        }
        if (substr($this->href, 0, 3) == "sys") {
            return "system";
        } else {
            return "inline";
        }
    }

    function getFileContent($href = "")
    {
        if ($href == "") {
            $href = $this->href;
        }
        if (substr($href, 0, 3) == "usr" || substr($href, 0, 3) == "sys") {
            return file_get_contents($this->getRealPath($href));
        }
        return "";
    }

    function getBackupName()
    {
        if ($this->href == "[disabled]" || $this->href == "") {
            return "[disabled]";
        }
        if ($this->href == "[simple]") {
            return $this->simplefile;
        }
        return substr($this->href, 4);
    }

    function setBackup($type, $name = "", $content = null)
    {
        if ($type == 'disabled') {
            if ($this->href !== "[disabled]") {
                $this->href = "[disabled]";
                $this->modified = true;
            }
            return;
        }
        if ($type == 'simple') {
            if ($this->href !== "[simple]") {
                $this->href = "[simple]";
                $this->modified = true;
            }
            return;
        }
        if ($name == "") {
            $this->addError(
                'href',
                'A name must be provided for the protocol.'
            );
            return;
        }
        if ($type == 'system') {
            if ($this->href !== "sys/" . $name) {
                $this->href = "sys/" . $name;
                if (!file_exists($this->getRealPath($this->href))) {
                    $this->addError('', 'System protocol do not exist.');
                }
                $this->modified = true;
            }
        } elseif ($content !== "" || $this->href !== "usr/" . $name) {
            $this->href = "usr/" . $name;
            if ($content == "") {
                if (!file_exists($this->getRealPath($this->href))) {
                    $this->addError(
                        '',
                        'Protocol do not exist. Must provide the protocol file.'
                    );
                }
            } else {
                file_put_contents($this->getRealPath($this->href), $content);
            }
            $this->modified = true;
        }
    }

    function setAttributes($inputs, $safeOnly = true)
    {
        if ($this->modified === null) {
            $this->load();
        }
        if (isset($inputs['href']) && $inputs['href'] == "") {
            $inputs['href'] = "[disabled]";
        }
        if (!is_array($inputs)) {
            return;
        }
        $attributes = array_flip(
            $safeOnly ? $this->getSafeAttributeNames() : $this->attributeNames()
        );
        foreach ($inputs as $name => $value) {
            if (isset($attributes[$name])) {
                if ($this->$name != $value) {
                    $this->modified = true;
                    $this->$name = $value;
                }
            } elseif ($safeOnly) {
                $this->onUnsafeAttribute($name, $value);
            }
        }
    }

    function load()
    {
        if ($this->playerConf === null) {
            $this->playerConf = new PlayerConfig();
        }
        if (!$this->playerConf->loadConfig()) {
            return false;
        }

        $domtree = $this->playerConf->domdoc;

        $ComPorts = $domtree->getElementsByTagName("serialPort");
        if ($ComPorts->length > 0) {
            $ComPorts = $ComPorts->item(0);

            if ($ComPorts->hasAttribute("href")) {
                $file = $ComPorts->getAttribute("href");
                if (strpos($file, self::$confpathsys) !== false) {
                    $this->href =
                        "sys/" . substr($file, strlen(self::$confpathsys));
                } else {
                    $this->href =
                        "usr/" . substr($file, strlen(self::$confpathusr));
                }
            } else {
                $this->href = "";
            }
            if ($this->href == "") {
                $this->href = "[disabled]";
            } elseif ($this->href == "sys/" . $this->powersavefile) {
                $this->href = "[disabled]";
            } elseif ($this->href == "usr/" . $this->simplefile) {
                $this->href = "[simple]";
            }

            $this->enableSerial = 1;
            if ($ComPorts->hasAttribute("portNumber")) {
                $this->portNumber = $ComPorts->getAttribute("portNumber");
            }
            if ($ComPorts->hasAttribute("dev")) {
                $this->device = $ComPorts->getAttribute("dev");
                $this->name = $ComPorts->getAttribute("name");
            }
            if ($ComPorts->hasAttribute("baudRate")) {
                $this->baudRate = $ComPorts->getAttribute("baudRate");
            }
            if ($ComPorts->hasAttribute("byteSize")) {
                $this->byteSize = $ComPorts->getAttribute("byteSize");
            }
            if ($ComPorts->hasAttribute("stopBits")) {
                $this->stopBits = $ComPorts->getAttribute("stopBits");
            }
            if ($ComPorts->hasAttribute("flowControl")) {
                $this->flowControl = $ComPorts->getAttribute("flowControl");
            }
            if ($ComPorts->hasAttribute("parity")) {
                $this->parity = $ComPorts->getAttribute("parity");
            }
        } else {
            $this->href = "[disabled]";
        }
        $this->modified = false;
    }

    public function loadDevices()
    {
        if ($this->device) {
            $this->_devices[$this->device] = $this->name;
        }
        foreach (Yii::app()->branding->serialPorts as $device => $name) {
            $this->_devices[$device] = $name;
        }
        $this->_devices['ttyUSB0'] = 'Any USB-to-Serial Adapter';
        $devices = array();
        if ($files = @scandir('/sys/class/tty')) {
            foreach ($files as $file) {
                if (!strncmp($file, 'ttyUSB', strlen('ttyUSB'))) {
                    $devices[] = '/sys/class/tty/' . $file;
                }
            }
        }
        //syslog(LOG_DEBUG, 'USB-to-serial devices: ' . print_r($devices, true));
        foreach ($devices as $device) {
            unset($output);
            if (exec('udevadm info -p ' . escapeshellarg($device) . 
                ' -q property', $output, $status) !== false && $status === 0
            ) {
                //syslog(LOG_DEBUG, 'udevadm info -p ' . $device . 
                //    ' -q property: ' . print_r($output, true));
                $properties = array();
                array_walk($output, 
                    function ($value, $index) use (&$properties) {
                        list($key, $value) = explode('=', $value);
                        $properties[$key] = $value;
                    }
                );
                //syslog(LOG_DEBUG, $device . ' properties: ' .
                //    print_r($properties, true));
                if (!isset($properties['DEVLINKS']) || !isset(
                    $properties['ID_SERIAL'])) {
                    continue;
                }
                $byId = strstr($properties['DEVLINKS'], 'serial/by-id');
                if ($byId === false) {
                    continue;
                }
                $byId = strtok($byId, ' ');
                $this->_devices[strtok($byId, ' ')] = 
                    str_replace('_', ' ', $properties['ID_SERIAL']);
            }
        }
        if (defined('YII_DEBUG') && YII_DEBUG) {
            if (!isset($this->devices['path/to/dummy0'])) {
                $this->_devices['path/to/dummy0'] = 'Dummy Device 0';
            }
            if (!isset($this->devices['path/to/dummy1'])) {
                $this->_devices['path/to/dummy1'] = 'Dummy Device 1';
            }
        }
        if (!$this->device) {
            $this->device = array_key_first($this->devices);
            $this->name = $this->devices[$this->device];
        }
    }

    function save()
    {
        $this->saveSimpleCommands();

        if (!$this->modified) {
            return false;
        }

        if ($this->playerConf === null) {
            $this->playerConf = new PlayerConfig();
        }
        if (!$this->playerConf->loadConfig()) {
            return false;
        }
        $domtree = $this->playerConf->domdoc;

        $serialPorts = $domtree->getElementsByTagName("serialPort");

        if ($this->enableSerial === 0) {
            // $this->enableSerial === 0 means that the serial port as been explicitelly disabled
            // In this case we need to remove the active fsm as it makes no sense to keep it.
            $this->href = "";
        }

        if ($this->href == "[disabled]") {
            $this->href = "";
        }
        if ($this->href == "[simple]") {
            $this->href = "usr/" . $this->simplefile;
        }

        $file = "";
        if ($this->href != "") {
            if (substr($this->href, 0, 3) == "sys") {
                $file = self::$confpathsys . substr($this->href, 4);
            } else {
                $file = self::$confpathusr . substr($this->href, 4);
            }
            // If there is an FSM then the serial must be enabled
            $this->enableSerial = 1;
        }

        $targetPort = null;
        if ($serialPorts->length > 0) {
            foreach ($serialPorts as $port) {
                if ($port->getAttribute("portNumber") == $this->portNumber) {
                    $targetPort = $port;
                    break;
                }
            }
        }

        if ($this->enableSerial) {
            if (!$targetPort) {
                $targetPort = $domtree->createElement("serialPort");
                $player = $domtree->documentElement->getElementsByTagName(
                    "player"
                );
                if ($player->length > 0) {
                    $targetPort = $player->item(0)->appendChild($targetPort);
                } else {
                    $targetPort = $domtree->documentElement->appendChild(
                        $targetPort
                    );
                    $this->addError(
                        "href",
                        "Warning no player tag found in the config, cannot add SerialPort tag."
                    );
                }
            }
            if (Yii::app()->device->model == 'ikebana') {
                // check if the serial port is actually free to be used
                $isSilent = shell_exec('fw_printenv -n silent');
                if (strcmp($isSilent, "yes\n") != 0) {
                    // Must disable the com port
                    $out = array();
                    exec('fw_setenv silent yes 2>&1', $out, $ret);
                    if ($ret !== 0) {
                        foreach ($out as &$val) {
                            $val = htmlspecialchars($val);
                        }
                        $this->addError(
                            "href",
                            "Warning failed to setup the RS232 interface ($ret): <br/>\n" .
                                implode("<br/>\n", $out)
                        );
                    }
                    Tools::addReason("disabling serial console");
                }
            }
            if ($file != "") {
                $targetPort->setAttribute("href", $file);
            } else {
                $targetPort->removeAttribute("href");
            }

            $targetPort->setAttribute("portNumber", $this->portNumber);
            if ($this->device) {
                $targetPort->setAttribute("dev", $this->device);
                $targetPort->setAttribute("name", $this->name);
            }
            $targetPort->setAttribute("baudRate", $this->baudRate);
            $targetPort->setAttribute("byteSize", $this->byteSize);
            $targetPort->setAttribute("stopBits", $this->stopBits);
            $targetPort->setAttribute("flowControl", $this->flowControl);
            $targetPort->setAttribute("parity", $this->parity);
        } else {
            $this->href = "";
            if ($targetPort) {
                $targetPort->parentNode->removeChild($targetPort);
            }
        }

        if (!$this->hasErrors()) {
            $this->playerConf->saveConfig(true);
            Tools::addReason("serial port config change");
            return true;
        } else {
            return false;
        }
    }

    function getSimpleCommands()
    {
        if ($this->_simpleCommands !== null) {
            return $this->_simpleCommands;
        }
        $file = self::$confpathusr . $this->simplefile;
        $commands = array();
        if (file_exists($file)) {
            $content = file_get_contents($file);

            $domdoc = new DOMDocument();
            $domdoc->preserveWhiteSpace = false;
            $domdoc->formatOutput = true;
            if ($domdoc->loadXML($content)) {
                $nodes = $domdoc->getElementsByTagName("onStart");
                foreach ($nodes as $node) {
                    $name = html_entity_decode($node->getAttribute("command"));
                    foreach ($node->childNodes as $child) {
                        if (
                            $child->nodeType == XML_ELEMENT_NODE &&
                            $child->nodeName == "write"
                        ) {
                            $value = html_entity_decode(
                                $child->getAttribute("data")
                            );
                        }
                    }
                    // fix the value: replace %c{decimal_number} by \hex
                    $simpleFSMCommand = array();
                    $simpleFSMCommand['name'] = $name;
                    $simpleFSMCommand['value'] = preg_replace_callback(
                        "/%c{(\d*)}/",
                        function ($matches) {
                            $val = intval($matches[1], 10);
                            if ($val < 16) {
                                return "\\x0" . strtoupper(dechex($val));
                            } else {
                                return "\\x" . strtoupper(dechex($val));
                            }
                        },
                        $value
                    );
                    $commands[] = $simpleFSMCommand;
                }
            }
        } else {
            $this->setSimpleCommands(array());
        }
        if (count($commands) == 0) {
            $commandsList = array(
                'PowerOn' => "",
                'PowerOff' => "",
            );
            foreach ($commandsList as $name => $value) {
                $commands[] = array('name' => $name, 'value' => $value);
            }
        }
        $this->_simpleCommands = $commands;
        return $commands;
    }
    function setSimpleCommands($commands)
    {
        $this->_simpleCommands = $commands;
        foreach ($this->_simpleCommands as &$simpleFSMCommand) {
            $value = preg_replace(
                '/&#x([A-Fa-f0-9][A-Fa-f0-9]?);/',
                "\\x$1",
                $simpleFSMCommand['value']
            );
            $value = preg_replace_callback(
                '/&#(\d+);/',
                function ($matches) {
                    return "\\x" . dechex($matches[1]);
                },
                $value
            );
            $value = preg_replace_callback(
                '/\\\\([bfnrtv0\'\\\\]|x[A-Fa-f0-9][A-Fa-f0-9]?)/',
                function ($matches) {
                    $val = $matches[1];
                    $map = array(
                        "b" => "08",
                        "f" => "0C",
                        "n" => "0A",
                        "r" => "0D",
                        "t" => "09",
                        "v" => "0B",
                        "0" => "00",
                        "'" => "27",
                        "\"" => "22",
                        "\\" => "5C",
                    );
                    if (strlen($val) == 1 && isset($map[$val])) {
                        return "\x" . $map[$val];
                    }
                    if (strlen($val) == 2) {
                        return "\\x0" . strtoupper(substr($val, 1));
                    }
                    if (strlen($val) == 3) {
                        return "\\x" . strtoupper(substr($val, 1));
                    }
                    return "???";
                },
                $value
            );
            $simpleFSMCommand['value'] = $value;
        }
    }
    function saveSimpleCommands()
    {
        if (!$this->_simpleCommands) {
            return true;
        }
        $xml =
            "<?xml version='1.0' encoding='utf-8'?>\n" .
            "<protocol xmlns='http://www.spinetix.com/namespace/1.0/spx' xml:id='test' desc='protocol' startup='start' target='monitor'>\n" .
            "<state xml:id='start'>\n";

        foreach ($this->_simpleCommands as &$simpleFSMCommand) {
            $name = $simpleFSMCommand['name'];
            $value = $simpleFSMCommand['value'];
            if (empty($name) || empty($value)) {
                continue;
            }
            $escapedval = preg_replace_callback(
                '/\\\\(x[A-F0-9][A-F0-9])/',
                function ($matches) {
                    return "%c{" . hexdec(substr($matches[1], 1)) . "}";
                },
                $value
            );
            $xml .=
                "<onStart command='" .
                htmlentities($name) .
                "' goto='start'>\n<write data='" .
                htmlentities($escapedval) .
                "'/>\n</onStart>\n";
        }
        $xml .= "</state>\n</protocol>";
        return file_put_contents(self::$confpathusr . $this->simplefile, $xml);
    }
    function getFileList($showSystem = true, $realpath = false)
    {
        $filelist = array();
        if ($showSystem) {
            $filelist1 = scandir(self::$confpathsys);
            $filelist['[disabled]'] = '[disabled]';
            foreach ($filelist1 as $file) {
                if (
                    $file == "." ||
                    $file == ".." ||
                    $file == $this->powersavefile
                ) {
                    continue;
                }
                if ($realpath) {
                    $filelist["sys/" . $file] = self::$confpathsys . $file;
                } else {
                    $filelist["sys/" . $file] =
                        "[sys] " . basename($file, ".fsm");
                }
            }
        }
        if (file_exists(self::$confpathusr)) {
            $filelist2 = scandir(self::$confpathusr);
            foreach ($filelist2 as $file) {
                if ($file != "." && $file != "..") {
                    if ($realpath && $file == $this->simplefile) {
                        continue;
                    }
                    if ($realpath) {
                        $filelist["usr/" . $file] = self::$confpathusr . $file;
                    } else {
                        $filelist["usr/" . $file] = substr($file, 0, -4);
                    }
                }
            }
        }
        return $filelist;
    }

    function saveFsm($file)
    {
        if (!$file->saveAs(self::$confpathusr . $file->name)) {
            $this->addError('uploadedfile', 'Saving the fsm file failed');
        }

        if ($this->href == "usr/" . $file->name) {
            Tools::addReason("update of active fsm file");
        }

        return array(
            'path' => "usr/" . $file->name,
            'name' => substr($file->name, 0, -4),
            'target' => substr($file->name, 0, -4),
            'system' => false,
            'disable' => false,
        );
    }
    function dump()
    {
        header("Content-type: text/xml");
        header(
            'Content-Disposition: attachment; filename=\'protocol.fsm\'; filename*=utf-8\'\'' .
                rawurlencode(substr($this->target, 4))
        );

        echo $this->getFileContent($this->target);
    }
    function delete()
    {
        if ($this->href == $this->target) {
            // must reset the config
            $this->href = "[disabled]";
            $this->modified = true;
            $this->save();
        }
        $file = $this->getRealPath($this->target);
        if (file_exists($file)) {
            unlink($file);
        }
    }
    function deleteAll()
    {
        if ($this->href !== "[disabled]") {
            // must reset the config
            $this->href = "[disabled]";
            $this->modified = true;
            $this->save();
        }
        $list = $this->getFileList(false);
        foreach ($list as $name => $username) {
            $file = $this->getRealPath($name);
            if (file_exists($file)) {
                unlink($file);
            }
        }
    }
}
