#!/bin/bash
#
# License: Copyright 2015 SpinetiX S.A. This file is licensed
#          under the terms of the GNU General Public License version 2.
#          This program is licensed "as is" without any warranty of any
#          kind, whether express or implied.
#
### BEGIN INIT INFO
# Required-Start:
# Required-Stop:
# Should-Start:
# Should-Stop:
# Default-Start: S
# Default-Stop: 0 1 2 3 4 5 6
# Short-Description: Populate volatile and temporary files and directories
# Description: Populate volatile and temporary files and directories
### END INIT INFO

# Init script information
NAME=spxtmpfiles.sh

# The directory where to find the tmpfile.d spec
#
# NOTE: the freedesktop.org specification has /etc/tmpfiles.d,
# /run/tmpfiles.d and /usr/lib/tmpfiles.d as possible locations, in
# order of precedence (*.conf files with the same name in the first
# location overriding others). Packages should install their files in
# /etc/tmpfiles.d. We do not implement the other locations.
TMPFILESCONFDIR=/etc/tmpfiles.d

# Source the init script functions
. /etc/init.d/functions

set_path_attrs() {
    local opts=()

    while [ $# -gt 0 ]; do
        case "$1" in
            --)
                shift
                break
                ;;
            -R)
                opts+=("-R")
                ;;
            -*)
                echo "invalid set_path_attrs option '$1'" >&2
                ;;
            *)
                break
                ;;
        esac
        shift
    done

    local mode="$1"
    local uid="$2"
    local gid="$3"
    shift 3

    [ -n "$mode" -a "$mode" != "-" ] && chmod "${opts[@]}" -- "$mode" "$@"
    [ -n "$uid" -a "$uid" != "-" ] && chown "${opts[@]}" -- "$uid" "$@"
    [ -n "$gid" -a "$gid" != "-" ] && chgrp "${opts[@]}" -- "$gid" "$@"
}

# Parses the tmpfiles specification (compatible with systemd) given as
# the first arg and performs all the required actions. The type
# identifiers handled are those of an old version of the
# specification, newer versions introduce extra types not handled
# here.
do_tmpfiles_conf() {
    local conf="$1"
    local type path mode uid gid age arg junk
    local p
    local actpaths=()

    while read type path mode uid gid age arg junk; do
        [ -z "$type" -o "$type" = "#" ] && continue # empty line or comment
        if [ -z "$path" ]; then
            echo "empty path in tmpfiles spec" >&2
            continue
        fi
        if [ "${path:0:1}" != "/" ]; then
            echo "relative path ('$path') in tmpfiles spec not allowed" >&2
            continue
        fi
        case "$type" in
            f) # create a file if it does not exist yet
                if [ ! -e "$path" ]; then
                    echo -n > "$path"
                    set_path_attrs -- "$mode" "$uid" "$gid" "$path"
                fi
                ;;
            F) # create or truncate a file
                echo -n > "$path"
                set_path_attrs -- "$mode" "$uid" "$gid" "$path"
                ;;
            d) # create a directory if it does not exist yet
                if [ ! -e "$path" ]; then
                    mkdir -- "$path"
                    set_path_attrs -- "$mode" "$uid" "$gid" "$path"
                fi
                ;;
            D) # create or empty a directory
                if [ ! -e "$path" ]; then
                    mkdir -- "$path"
                    set_path_attrs -- "$mode" "$uid" "$gid" "$path"
                else
                    find -- "$path" -mindepth 1 -print0 | xargs -0r rm -rf --
                fi
                ;;
            L) # create a symlink if it does not exist yet
                if [ -z "$arg" ]; then
                    echo "link for '$path' missing target argument" >&2
                elif [ ! -L "$path" -a ! -e "$path" ]; then
                    ln -Tns -- "$arg" "$path"
                fi
                ;;
            L+) # create a symlink if it does not exist yet or is a file
                if [ -z "$arg" ]; then
                    echo "link for '$path' missing target argument" >&2
                elif [ ! -L "$path" -a \( ! -e "$path" -o -f "$path" \) ]; then
                    ln -Tnfs -- "$arg" "$path"
                fi
                ;;
            x) # ignore a path; accepts globs
                ;;
            r) # remove a path; accepts globs
                for p in $path; do
                    if [ -d "$p" ]; then
                        rmdir -- "$p"
                    elif [ -e "$p" ]; then
                        rm -f -- "$p"
                    fi
                done
                ;;
            R) # recursively remove a path; accepts globs
                for p in $path; do
                    [ -e "$p" ] && rm -rf -- "$p"
                done
                ;;
            z) # adjust the access mode, group and user if path exists; accepts globs
                actpaths=()
                for p in $path; do
                    [ -e "$p" ] && actpaths+=("$p")
                done
                [ "${#actpaths[@]}" -gt 0 ] && \
                    set_path_attrs -- "$mode" "$uid" "$gid" "${actpaths[@]}"
                ;;
            Z) # recursively adjust the access mode, group and user if path exists; accepts globs
                actpaths=()
                for p in $path; do
                    [ -e "$p" ] && actpaths+=("$p")
                done
                [ "${#actpaths[@]}" -gt 0 ] && \
                    set_path_attrs -R -- "$mode" "$uid" "$gid" "${actpaths[@]}"
                ;;
            *) # unkown specification
                echo "invalid tmpfiles spec '$type'" >&2
                ;;
        esac
    done < "$conf"
}

# Do all the tmpfiles specifications found
do_tmpfiles_all() {
    for f in "$TMPFILESCONFDIR"/*.conf ; do
        [ -f "$f" ] && do_tmpfiles_conf "$f"
    done
}

#
# Main
#

# This script relies on matching names starting with a dot on globs
shopt -s dotglob

case "$1" in
    start)
        echo -n "Populating volatile and temporary paths: $NAME "

        do_tmpfiles_all

        success
        echo

        ;;
    stop)
        ;;
esac
