<?php

require_once './protected/utils/WebStorageAPI.php';
require_once './protected/utils/BrandingInfo.php'; // branding info

function isLoggedUser()
{
    if (!isset($_COOKIE["uisess"])) {
        return false;
    }
    $id = $_COOKIE["uisess"];
    $autentication_path = ini_get('session.save_path');
    return file_exists($autentication_path . "/auth_" . $id);
}

function checkWebstorage()
{
    $brandingInfo = new BrandingInfo();
    if ($brandingInfo->hasRight('netAPI')) {
        return true;
    }
    if (isLoggedUser()) {
        return true;
    }

    throw new Exception("Not supported by this model");
}

function webstorage_list()
{
    checkWebstorage();

    $store = new WebStorageAPI();

    $ret = $store->getList();

    if ($ret === false) {
        throw new Exception("Internal error");
    }

    $store->close();
    return $ret;
}
function webstorage_get($names = null)
{
    checkWebstorage();

    if ($names === null) {
        throw new Exception("Missing names");
    }

    if (isset($names['debug'])) {
        $names = $names['debug'];
    }
    if (!is_array($names) || isset($names['name'])) {
        throw new Exception("expect array as argument");
    }
    $store = new WebStorageAPI();
    $ret = array();
    foreach ($names as $name) {
        $val = $store->get($name);
        if ($val === false) {
            throw new Exception("Internal error");
        }
        $ret[] = $val;
    }
    $store->close();
    return $ret;
}
function webstorage_set($variables = null)
{
    checkWebstorage();

    if ($variables === null) {
        throw new Exception("Missing variables");
    }

    if (isset($variables['debug'])) {
        $variables = $variables['debug'];
    }
    if (!is_array($variables) || isset($variables['name'])) {
        throw new Exception("expect array as argument");
    }
    foreach ($variables as $name => $variable) {
        if (!is_array($variable)) {
            throw new Exception("name and value required");
            //$variables[] = array('value' => $variable, 'name' => $name );
            //unset( $variables[$name]);
            //continue;
        }
        if (!isset($variable['name'])) {
            throw new Exception("name is required");
        }
        if (!isset($variable['value'])) {
            throw new Exception("value is required");
        }
    }

    $store = new WebStorageAPI();
    $ret = array();
    foreach ($variables as $variable) {
        $timestamp = "+0";
        if (isset($variable['offset'])) {
            $timestamp = "+" . $variable['offset'];
        }
        if (isset($variable['timestamp'])) {
            $timestamp = $variable['timestamp'];
        }
        $val = $store->set($variable['name'], $variable['value'], $timestamp);
        if ($val === false) {
            $store->close();
            throw new Exception("Internal error");
        }
        $ret[] = $val;
    }
    $store->close();
    return $ret;
}
function webstorage_remove($names = null)
{
    checkWebstorage();

    if ($names === null) {
        throw new Exception("Missing names");
    }

    if (isset($names['debug'])) {
        $names = $names['debug'];
    }
    if (!is_array($names) || isset($names['name'])) {
        throw new Exception("expect array as argument");
    }
    $store = new WebStorageAPI();
    $ret = array();
    foreach ($names as $name) {
        $val = $store->remove($name);
        if ($val === false) {
            throw new Exception("Internal error");
        }
        $ret[] = $val;
    }
    $store->close();
    return $ret;
}

function webstorage_cmpxchg($variables = null)
{
    checkWebstorage();

    if ($variables === null) {
        throw new Exception("Missing variables");
    }

    if (isset($variables['debug'])) {
        $variables = $variables['debug'];
    }
    if (!is_array($variables) || isset($variables['name'])) {
        throw new Exception("expect array as argument");
    }
    foreach ($variables as $variable) {
        if (!isset($variable['name'])) {
            throw new Exception("name is required");
        }
        if (!isset($variable['value'])) {
            throw new Exception("value is required");
        }
    }
    $store = new WebStorageAPI();
    $ret = array();
    foreach ($variables as $variable) {
        $timestamp = "+0";
        if (isset($variable['offset'])) {
            $timestamp = "+" . $variable['offset'];
        }
        if (isset($variable['timestamp'])) {
            $timestamp = $variable['timestamp'];
        }

        if (!isset($variable['expect'])) {
            $val = $store->create(
                $variable['name'],
                $variable['value'],
                $timestamp
            );
        } else {
            $val = $store->cmpxchg(
                $variable['name'],
                $variable['expect'],
                $variable['value'],
                $timestamp
            );
        }
        if ($val === false) {
            throw new Exception("Internal error");
        }
        $ret[] = $val;
    }
    $store->close();
    return $ret;
}

function respond($result, $error = null)
{
    header("Content-Type: " . 'application/json');
    if ($error) {
        http_response_code(400);
        $body = json_encode(array("error" => $error));
    } else {
        $body = json_encode($result);
    }
    header("Content-Length: " . strlen($body));
    echo $body;
    flush();
}
function process()
{
    $verb = $_SERVER['REQUEST_METHOD'];

    if (isset($_SERVER['ORIG_PATH_INFO'])) {
        $path = $_SERVER['ORIG_PATH_INFO'];
        $paths = explode("/", $path);
    } else {
        $path = $_SERVER['SCRIPT_URL'];
        $paths = explode("/", $path);
    }
    if (count($paths) == 2 && isset($_GET['action'])) {
        $verb = $_GET['action'];
    }
    $emptyAnswer = false;

    try {
        if (count($paths) == 2 && isset($_GET['name'])) {
            $name = $_GET['name'];
        } else {
            $name = implode("/", array_slice($paths, 2));
        }
        if ($name === "") {
            switch ($verb) {
                case "GET":
                    $ret = json_encode(webstorage_list());
                    http_response_code(200);
                    header("Content-Type: " . 'application/json');
                    header("Content-Length: " . strlen($ret));
                    header("Cache-Control: " . 'no-cache');
                    echo $ret;
                    break;
                case "POST":
                    if ($_SERVER['CONTENT_TYPE'] === 'application/json') {
                        $input = file_get_contents("php://input");
                        $data = json_decode($input, true);
                    } else {
                        $data = array();
                        foreach ($_POST as $name => $value) {
                            $data[] = array("name" => $name, "value" => $value);
                        }
                    }
                    $ret = webstorage_set($data);
                    http_response_code(204);
                    break;
                case "PURGE":
                    webstorage_remove(webstorage_list());
                    http_response_code(204);
                    break;
            }
        } else {
            switch ($verb) {
                case "GET":
                    $ret = webstorage_get(array($name))[0];
                    if (isset($_GET['format'])) {
                        header("Content-Type: " . $_GET['format']);
                    } else {
                        header("Content-Type: " . 'application/octet-stream');
                    }
                    if ($ret['success']) {
                        http_response_code(200);
                        header(
                            "Last-Modified: " .
                                gmdate(
                                    "D, d M Y H:i:s",
                                    floor($ret['timestamp'] / 1000)
                                ) .
                                " GMT"
                        );
                        header("Cache-Control: " . 'no-cache');
                        echo $ret['value'];
                    } else {
                        http_response_code(404);
                        header("Content-Type: " . 'application/json');
                        $body = json_encode(array("error" => "Not found"));
                        header("Content-Length: " . strlen($body));
                        echo $body;
                    }
                    break;
                case "POST":
                case "PUT":
                    $input = file_get_contents("php://input");
                    $data = array("value" => $input);
                    $data['name'] = $name;
                    if (isset($_GET['expect'])) {
                        $data['expect'] = $_GET['expect'];
                        $ret = webstorage_cmpxchg(array($data))[0];
                    } else {
                        $ret = webstorage_set(array($data))[0];
                    }
                    if ($ret['success']) {
                        if ($ret['created']) {
                            header("Content-Type: " . 'application/octet-stream');
                            header("Cache-Control: " . 'no-cache');
                            http_response_code(201);
                        } else {
                            http_response_code(204);
                        }
                    } else {
                        http_response_code(412);
                        header("Cache-Control: " . 'no-cache');
                        if (isset($ret['prev_value'])) {
                            echo $ret['prev_value'];
                        }
                    }
                    break;
                case "DELETE":
                    $ret = webstorage_remove(array($name));
                    http_response_code(204);
                    break;
            }
        }
    } catch (Exception $e) {
        http_response_code(400);
        $body = json_encode(array("error" => $e->getMessage()));
        header("Content-Type: " . 'application/json');
        header("Content-Length: " . strlen($body));
        echo $body;
    }
    flush();
}

process();
