#!/bin/sh
#
# License: Copyright 2015 SpinetiX. This file is licensed
#          under the terms of the GNU General Public License version 2.
#          This program is licensed "as is" without any warranty of any
#          kind, whether express or implied.
#
# Copyright 1999-2003 MontaVista Software, Inc.
# Copyright 2002, 2003, 2004 Sony Corporation
# Copyright 2002, 2003, 2004 Matsushita Electric Industrial Co., Ltd.
#
### BEGIN INIT INFO
# Required-Start:
# Required-Stop:
# Should-Start:
# Should-Stop:
# Default-Start: S
# Default-Stop: 0 1 2 3 4 5 6
# Short-Description: Config reset mechanism.
# Description: Config reset mechanism.
### END INIT INFO

# Init script information
NAME=config
DESC=""

# The file that permanently stores configuration changes so as to be
# robust against reboots while resetting.
BOOTCONFIG=/boot-config

# The splash resources directory
RCDIR_SPLASH=/usr/share/resources/default/splash

# If set then the copying of the recovery splash screens to the
# failsafe-data is skipped, normally set on machines which have a
# modern splash screen and do not need these.
RECOVERY_SKIP_SPLASH=yes

# The file used to flag if the recovery console splash has been set
RECOVERY_SPLASH_SET=/var/lib/spinetix/recovery-splash-set

# The name of the persistent data partition (MTD)
PERSISTENT_DATA_PNAME=persistent-data
# Flag file to skip reset of persistent data partition
PERSISTENT_DATA_NORESET=/var/lib/spinetix/persistent-data.noreset

reset_persistent_data_mtd() {
    # the persistent data partition is mounted later, so we can
    # go ahead and reset it
    local mtdnr
    mtdnr="$(sed -ne '/\"'"$PERSISTENT_DATA_PNAME"'\"$/{s/mtd\([0-9]\+\).*/\1/;p}' /proc/mtd)"
    if [ -z "$mtdnr" ]; then
        echo "persistent data storage partition $PERSISTENT_DATA_PNAME not found"
        return 1
    fi
    flash_erase -q -j /dev/mtd"$mtdnr" 0 0
    if ! [ $? -eq 0 ]; then
        echo "persistent data storage reset failed"
        return 1
    fi
    rm -f "$PERSISTENT_DATA_NORESET"
}

reset_persistent_data_bdev() {
    if [ -z "$SPX_PERSISTENT_DATA_BDEV" ]; then
        [ -f /etc/spinetix/identifiers ] && . /etc/spinetix/identifiers
    fi
    if [ -z "$SPX_PERSISTENT_DATA_BDEV" ]; then
        echo "no persistent data storage block device"
        return 1
    fi
    [ -b "$SPX_PERSISTENT_DATA_BDEV" ] && \
        dd if=/dev/zero of="$SPX_PERSISTENT_DATA_BDEV" bs=4096 count=1 conv=fsync 2> /dev/null
    if ! [ $? -eq 0 ]; then
        echo "persistent data storage reset failed"
        return 1
    fi
    rm -f "$PERSISTENT_DATA_NORESET"
}

reset_persistent_data() {
    if [ "$HOSTTYPE" = "arm" ]; then
        reset_persistent_data_mtd
    else
        reset_persistent_data_bdev
    fi
}

do_config_changes() {
    config="$(sed -e 's/.*\bconfig=\([^[:space:]]\+\).*/\1/p;d' < /proc/cmdline)"
    if [ -n "$config" ]; then
        # append boot command line config to overall config orders atomically
        if [ -f "$BOOTCONFIG" -a -s "$BOOTCONFIG" ]; then
            cat "$BOOTCONFIG" > "$BOOTCONFIG".new && \
                echo "$config" >> "$BOOTCONFIG".new && sync && \
                mv -f "$BOOTCONFIG".new "$BOOTCONFIG" && sync
            [ $? -eq 0 ] || \
                echo "Failed adding configuration request '$config' to '$BOOTCONFIG'" >&2
        else
            echo "$config" > "$BOOTCONFIG".new && sync && \
                mv -f "$BOOTCONFIG".new "$BOOTCONFIG" && sync
            [ $? -eq 0 ] || \
                echo "Failed saving configuration request '$config' to '$BOOTCONFIG'" >&2
        fi
    elif [ ! -f "$BOOTCONFIG" ]; then
        # not a regular file or does not exist
        [ -e "$BOOTCONFIG" ] && rm -f "$BOOTCONFIG"
        return 0
    elif [ ! -s "$BOOTCONFIG" ]; then
        # file exists but empty
        rm -f "$BOOTCONFIG"
        return 0
    fi
    [ -e "$BOOTCONFIG".new ] && rm -f "$BOOTCONFIG".new

    # NOTE: the list of supported configuration tags must be entered in
    # /etc/spinetix/boot-config-tags.
    while read config; do
        case "$config" in
            format|format=*)
                # format is handled earlier in another boot script
                # but need to keep the tag
                echo "$config" >> "$BOOTCONFIG".new
                echo "WARNING: format requested but did not complete, will retry next boot"
                ;;
            reset)
                echo '!!!! RESETTING CONFIGURATION !!!!'
                for f in /etc/spinetix/config.d/*; do
                    echo DOING "$(basename $f)"
                    $f reset
                done
                echo 'DONE'
                ;;
            reset-persistent-data)
                echo '!!!! RESETTING PERSISTENT DATA !!!!'
                reset_persistent_data
                echo 'DONE'
                ;;
            dhcp-nfs)
                echo -n "Resetting network configuration to DHCP with NFS"
                /etc/spinetix/spxsysconf dhcp
                echo "."
                ;;
            dhcp|static=*)
                echo -n "Resetting network configuration to '$config'"
                /etc/spinetix/spxsysconf net "$config"
                echo "."
                ;;
            *)
                echo "Ignoring unknown config order $config" >&2
                ;;
        esac
    done < "$BOOTCONFIG"

    rm -f "$BOOTCONFIG"
    if [ -s "$BOOTCONFIG".new ]; then
        sync && mv -f "$BOOTCONFIG".new "$BOOTCONFIG"
        [ $? -eq 0 ] || \
            echo "Failed keeping unhandled configuration requests" >&2
    fi
    [ -e "$BOOTCONFIG".new ] && rm -f "$BOOTCONFIG".new
    sync

}

clean_upgrade_pkg() {
    if [ -f /var/cache/upgrade.pkg ]; then
        echo "removing existing upgrade.pkg file from cache directory"
        rm /var/cache/upgrade.pkg
    fi
}

case "$1" in
    start)
        # First process all config changes
        do_config_changes

        # Clean any leftover upgrade.pkg file now, they are not needed any
        # longer and it frees space for the subsequent operations.
        clean_upgrade_pkg

        # Handle ID remapping for /srv (already mounted content partition)
        if [ -e /srv/.remap-ids ]; then

            echo "INFO: doing ID remapping for /srv"
            [ -x /etc/init.d/splash ] && /etc/init.d/splash updating
            # Add a small delay for user to notice splash screen before we start
            sleep 5

            # the remapping process may take long depending on the number of paths to process,
            # we thus need to start a watchdog daemon to avoid the hardware watchdog from triggering;
            # the watchdog configuration checks for an uptime limit, which we set to one hour
            mkdir -p /run/spxsysconf
            echo 3600 > /run/spxsysconf/wd-max-uptime
            start-stop-daemon -S --quiet --exec /usr/sbin/watchdog -- -c /etc/spinetix/uptime-watchdog.conf

            /usr/libexec/spxsysconf/path-remap-ids /srv
            if [ $? -ne 0 ]; then
                echo "ERROR: incomplete remapping, problems may occur"
                if [ -f /srv/.remap-ids.errors ]; then
                    echo "INFO: logged errors are"
                    cat /srv/.remap-ids.errors
                fi
            fi

            # Update the maximum uptime limit to be a fixed time from now (atomic update!)
            # First /proc/uptime entry is seconds of uptime, with possible decimal part
            UPTIME_SECS="$(< /proc/uptime)"
            UPTIME_SECS="${UPTIME_SECS%% *}"
            UPTIME_SECS="${UPTIME_SECS%%.*}"
            UPTIME_SECS=$(( UPTIME_SECS + 900 ))
            echo "$UPTIME_SECS" > /run/spxsysconf/wd-max-uptime.new
            mv /run/spxsysconf/wd-max-uptime.new /run/spxsysconf/wd-max-uptime

        elif [ -f /srv/.remap-ids.errors ]; then
            echo "WARNING: a past ID remapping of /srv logged the following errors, problems may occur"
            cat /srv/.remap-ids.errors
        fi

        # Create initial network configuration if missing
        if ! grep -q '# START-spxsysconfig' /etc/network/interfaces; then
            /etc/spinetix/spxsysconf net-default
        fi

        # Run the recovery console splash screen update
        if [ -z "$RECOVERY_SKIP_SPLASH" ] && \
            [ ! -f "$RECOVERY_SPLASH_SET" ] && \
            [ -f "$RCDIR_SPLASH"/recovery.png ] && \
            [ -f "$RCDIR_SPLASH"/recovery-shutdown.png ]; then
            failsafe-data \
                remove start.jpg \
                remove shutdown.jpg \
                copy start.png "$RCDIR_SPLASH"/recovery.png \
                copy shutdown.png "$RCDIR_SPLASH"/recovery-shutdown.png
            touch "$RECOVERY_SPLASH_SET"
        fi

        ;;
    stop)
        ;;
esac
