/*!
 * jSignage.Graph
 * http://www.spinetix.com/
 * Copyright 2013, SpinetiX S.A.
 * Released under the GPL Version 2 license.

 *
 * $Date: 2014-06-26 17:19:26 +0200 (Thu, 26 Jun 2014) $
 * $Revision: 21805 $
 */
 
 ( function () {
    // Private declarations

    function measureLabels( ctx, textStyle, textAlign, textBaseline, labels ) {
        ctx.font = textStyle.font();
        ctx.textAlign = textAlign;
        ctx.textBaseline = textBaseline;
        var maxLeft = 0, maxRight = 0, maxAscent = 0, maxDescent = 0;
        for ( var i=0; i < labels.length; i++ ) {
            var text = labels[i];
            if ( typeof(text)=='object' && text ) {
                ctx.font = ( text.style ? new TextStyle( text.style, textStyle ) : textStyle ).font();
                text = text.text;
            }
            if ( text!==null ) {
                var m = jSignage.measureText( ctx, text );
                maxRight = Math.max( maxRight, m.actualBoundingBoxRight );
                maxLeft = Math.max( maxLeft, m.actualBoundingBoxLeft );
                var fs = m.fontBoundingBoxAscent + m.fontBoundingBoxDescent;
                maxAscent = Math.max( maxAscent, m.actualBoundingBoxAscent+fs*0.05 );
                maxDescent = Math.max( maxDescent, m.actualBoundingBoxDescent+fs*0.05 );
            }
        }
        return { left: maxLeft, right: maxRight, ascent: maxAscent, descent: maxDescent };
    }

    function measureLegend( labels, ctx, textStyle, orientation, series ) {
        var fs = textStyle.fontSize, inc = fs*1.4;
        if ( series ) {
            labels = [];
            for ( var i=0; i < series.length; i++ )
                labels.push( series[i].label );
        }
        if ( orientation=='horizontal' ) {
            ctx.font = textStyle.font();
            ctx.textAlign = 'left';
            ctx.textBaseline = 'alphabetic';
            var width = 0;
            for ( var i=0; i < labels.length; i++ ) {
                var m = jSignage.measureText( ctx, labels[i] );
                width += inc + m.actualBoundingBoxRight + inc;
            }
            return { width: width ? width-inc : 0, height: inc };
        } else {
            var right = measureLabels( ctx, textStyle, 'left', 'alphabetic', labels ).right;
            return {
                width: inc + right + fs*0.2,
                height: labels.length*inc
            };
        }
    }

    function drawLegend( labels, ctx, textStyle, strokeStyle, colors, frameStyle, x, y, width, height, orientation, series, dataSize ) {
        var fs = textStyle.fontSize, inc = fs*1.4, label;

        if ( frameStyle ) {
            setStrokeStyle( ctx, frameStyle );
            ctx.strokeRect( x, y, width, height );
        }
        if ( strokeStyle && !series )
            setStrokeStyle( ctx, strokeStyle );
        ctx.font = textStyle.font();
        for ( var i=0; i < (series||labels).length; i++ ) {
            ctx.beginPath();
            if ( series ) {
                if ( !dataSize && series[i].fillOpacity ) {
                    ctx.rect( x+fs*0.2, y+fs*0.2, fs, fs );
                    ctx.fillStyle = series[i].fillColor;
                    ctx.globalAlpha = series[i].fillOpacity;
                    ctx.fill();
                    ctx.globalAlpha = 1;
                    setStrokeStyle( ctx, series[i].strokeStyle );
                    ctx.stroke();                    
                } else {
                    if ( !dataSize && series[i].line!='none' ) {
                        ctx.moveTo( x+fs*0.2, y+fs*0.7 );
                        ctx.lineTo( x+fs*1.2, y+fs*0.7 );
                        setStrokeStyle( ctx, series[i].strokeStyle );
                        ctx.stroke();
                    }
                    if ( series[i].marker.type!=='none' ) {
                        ctx.fillStyle = dataSize ? series[i].fillColor : series[i].marker.color;
                        drawMarker( ctx, series[i].marker.type, x+fs*0.7, y+fs*0.7, dataSize ? fs*0.9 : fs*0.5 );
                        if ( dataSize )
                            ctx.globalAlpha = series[i].fillOpacity;
                        ctx.fill();
                        if ( dataSize )
                            ctx.globalAlpha = 1;
                        if ( series[i].marker.strokeStyle ) {
                            setStrokeStyle( ctx, series[i].marker.strokeStyle );
                            ctx.stroke();
                        }
                    }
                }
                label = series[i].label;
            } else {
                ctx.rect( x+fs*0.2, y+fs*0.2, fs, fs );
                ctx.fillStyle = colors[i];
                ctx.fill();
                if ( strokeStyle )
                    ctx.stroke();
                label = labels[i];
            }
            ctx.fillStyle = textStyle.color;
            ctx.fillText( label, x+inc, y+fs*1.1 );
            if ( orientation=='horizontal' ) {
                var m = jSignage.measureText( ctx, label );
                x += inc + m.actualBoundingBoxRight + inc;
            } else {
                y += inc;
            }
        }
    }

    function addLegend( width, height, labels, legend, ctx, colors, series, dataSize ) {
        var graphArea = { x: 0, y: 0, width: width, height: height };

        if ( legend && legend.position!=='none' ) {
            var orientation = legend.orientation || ( legend.position=='top' || legend.position=='bottom' ? 'horizontal' : 'vertical' );
            var alignment = legend.alignment || ( legend.position=='top' || legend.position=='bottom' ? 'center' : 'top' );
            var bbox = measureLegend( labels, ctx, legend.textStyle, orientation, series );
            var ll = 0, lt = 0, lw = width, lh = height, overlay = false;
            if ( legend.position=='right' ) {
                ll = width-bbox.width;
                lw = bbox.width;
            } else if ( legend.position=='left' ) {                
                lw = bbox.width;
            } else if ( legend.position=='top' ) {
                lh = bbox.height;
            } else if ( legend.position=='bottom' ) {
                lt = height-bbox.height;
                lh = bbox.height;
            } else if ( legend.position=='in' ) {
                overlay = true;
                ll = width - bbox.width;
                lw = bbox.width;
            } else {
                overlay = true;
                ll = jSignage.relAbs( legend.position.left, width, 0 );
                lt = jSignage.relAbs( legend.position.top, height, 0 );
                lw = jSignage.relAbs( legend.position.width, width, width-ll );
                lh = jSignage.relAbs( legend.position.height, height, height-lt );
            }
            if ( !overlay ) {
                if ( ll > 0 ) {
                    graphArea.width = ll;
                } else if ( lw < width ) {
                    graphArea.width -= lw;
                    graphArea.x += lw;
                }
                if ( lt > 0 ) {
                    graphArea.height = lt;
                } else if ( lh < height ) {
                    graphArea.height -= lh;
                    graphArea.y += lh;
                }
            }
            if ( orientation=='horizontal' ) {
                if ( alignment=='center' )
                    ll = (width-bbox.width)/2;
                else if ( alignment=='right' || alignment=='end' )
                    ll = width-bbox.width;
            } else {
                if ( alignment=='center' )
                    lt = (height-bbox.height)/2;
                else if ( alignment=='bottom' || alignment=='end' )
                    lt = height-bbox.height;
            }
            drawLegend( labels, ctx, legend.textStyle, legend.strokeStyle, colors, legend.frameStyle, ll, lt, bbox.width, bbox.height, orientation, series, dataSize );
        }
        return graphArea;
    }

    function getAxisLabels( axis ) {
        var labels;
        if ( axis.ticks ) {
            labels = [];
            for ( var i=0; i < axis.ticks.length; i++ )
                labels.push( axis.ticks[i].f );
        } else {
            labels = axis.categories;
        }
        return labels;
    }

    function measureCircularAxis( ctx, axis ) {
        var tick = 0;
        var margin = { alpha: 0, maxTextWidth: 0, maxTextHeight: 0 };

        if ( !axis )
            return margin;

        if ( axis.axisStrokeStyle )
            margin.alpha = Math.max( margin.alpha, axis.axisStrokeStyle.lineWidth/2 );

        if ( axis.tickMarks ) {
            var l = axis.tickMarks.length;
            if ( axis.minorTickMarks )
                l = Math.max( l, axis.minorTickMarks.length );
            if ( axis.position < 0 ? axis.tickMarks.placement!='inside' : axis.tickMarks.placement!='outside' )
                margin.alpha = Math.max( margin.alpha, l );
            tick = ( axis.position < 0 ? axis.tickMarks.placement!='inside' : axis.tickMarks.placement!='outside' ) ? axis.tickMarks.length : 0;
        }

        if ( axis.textPlacement!='none' && ( axis.position < 0 ? axis.textPlacement=='outside' : axis.textPlacement!='outside' ) ) {
            var m = measureLabels( ctx, axis.textStyle, 'center', 'middle', getAxisLabels( axis ) );
            margin.alpha = Math.max( margin.alpha, tick + axis.textStyle.fontSize * 0.2 );
            margin.maxTextWidth = m.left + m.right;
            margin.maxTextHeight = m.ascent + m.descent;
        }
        
        margin.alpha -= axis.offset * axis.position;

        return margin;
    }

    function measureAxis( ctx, axis, orientation, xDir, yDir ) {
        var maxLineWidth = 0, tick1 = 0, tick2 = 0;
        var margin = { left: 0, right: 0, top: 0, bottom: 0 };

        if ( !axis )
            return margin;

        if ( axis.axisStrokeStyle )
            maxLineWidth = Math.max( maxLineWidth, axis.axisStrokeStyle.lineWidth );
        if ( axis.baselineSrokeStyle )
            maxLineWidth = Math.max( maxLineWidth, axis.baselineSrokeStyle.lineWidth );
        if ( axis.gridlinesStrokeStyle ) {
            maxLineWidth = Math.max( maxLineWidth, axis.gridlinesStrokeStyle.lineWidth );
            if ( axis.minorGridLines )
                maxLineWidth = Math.max( maxLineWidth, axis.minorGridLines.strokeStyle.lineWidth );
        }
        if ( axis.tickMarks ) {
            maxLineWidth = Math.max( maxLineWidth, axis.tickMarks.strokeStyle.lineWidth );
            if ( axis.minorTickMarks )
                maxLineWidth = Math.max( maxLineWidth, axis.minorTickMarks.strokeStyle.lineWidth );
        }

        if ( orientation=='y' ) {
            margin.top = Math.max( margin.top, maxLineWidth/2 );
            margin.bottom = Math.max( margin.bottom, maxLineWidth/2 );
            if ( axis.axisStrokeStyle ) {
                margin.left = Math.max( margin.left, axis.axisStrokeStyle.lineWidth/2 );
                margin.right = Math.max( margin.right, axis.axisStrokeStyle.lineWidth/2 );
            }
        } else {
            margin.left = Math.max( margin.left, maxLineWidth/2 );
            margin.right = Math.max( margin.right, maxLineWidth/2 );
            if ( axis.axisStrokeStyle ) {
                margin.top = Math.max( margin.top, axis.axisStrokeStyle.lineWidth/2 );
                margin.bottom = Math.max( margin.bottom, axis.axisStrokeStyle.lineWidth/2 );
            }
        }

        if ( axis.tickMarks ) {
            var l = axis.tickMarks.length;
            if ( axis.minorTickMarks )
                l = Math.max( l, axis.minorTickMarks.length );
            if ( orientation=='y' ) {
                if ( xDir < 0 ) {
                    if ( axis.tickMarks.placement!='inside' )
                        margin.right = Math.max( margin.right, l );
                    if ( axis.tickMarks.placement!='outside' )
                        margin.left = Math.max( margin.left, l );
                } else {
                    if ( axis.tickMarks.placement!='inside' )
                        margin.left = Math.max( margin.left, l );
                    if ( axis.tickMarks.placement!='outside' )
                        margin.right = Math.max( margin.right, l );
                }
            } else {
                if ( yDir < 0 ) {
                    if ( axis.tickMarks.placement!='inside' )
                        margin.top = Math.max( margin.top, l );
                    if ( axis.tickMarks.placement!='outside' )
                        margin.bottom = Math.max( margin.bottom, l );
                } else {
                    if ( axis.tickMarks.placement!='inside' )
                        margin.bottom = Math.max( margin.bottom, l );
                    if ( axis.tickMarks.placement!='outside' )
                        margin.top = Math.max( margin.top, l );
                }
            }
            tick1 = axis.tickMarks.placement=='outside' || axis.tickMarks.placement=='across' ? axis.tickMarks.length : 0;
            tick2 = axis.tickMarks.placement=='inside' || axis.tickMarks.placement=='across' ? axis.tickMarks.length : 0;
        }

        if ( axis.textPlacement!='none' ) {
            var m, texAlign, textBaseline;
            var textPlacement = axis.ticks ? axis.textPlacement : 'outside';
            if ( orientation=='y' ) {
                if ( textPlacement=='outside' ) {
                    textAlign = xDir < 0 ? 'left': 'right';
                    textBaseline = 'middle';
                } else {
                    textAlign = xDir < 0 ? 'right': 'left';
                    if ( textPlacement=='above' )
                        textBaseline = 'bottom';
                    else if ( textPlacement=='below' )
                        textBaseline = 'top';
                    else
                        textBaseline = 'middle';
                }
            } else {
                if ( textPlacement=='outside' ) {
                    textAlign = 'center';
                    textBaseline = yDir < 0 ? 'bottom' : 'top';
                } else {
                    textBaseline = yDir < 0 ? 'top' : 'bottom';
                    textAlign = 'center';
                }
            }
            m = measureLabels( ctx, axis.textStyle, textAlign, textBaseline, getAxisLabels( axis ) );
            if ( orientation=='y' ) {
                margin.top = Math.max( margin.top, m.ascent );
                margin.bottom = Math.max( margin.bottom, m.descent );
                if ( xDir < 0 ) {
                    if ( textPlacement=='outside' )
                        margin.right = Math.max( margin.right, m.right + ( axis.ticks ? tick1 : 0 ) + axis.textStyle.fontSize * 0.2 );
                    else
                        margin.left = Math.max( margin.left, m.left + ( axis.ticks ? tick2 : 0 ) + axis.textStyle.fontSize * 0.2 );
                } else {
                    if ( textPlacement=='outside' )
                        margin.left = Math.max( margin.left, m.left + ( axis.ticks ? tick1 : 0 ) + axis.textStyle.fontSize * 0.2 );
                    else
                        margin.right = Math.max( margin.right, m.right + ( axis.ticks ? tick2 : 0 ) + axis.textStyle.fontSize * 0.2 );
                }
            } else {
                margin.left = Math.max( margin.left, m.left );
                margin.right = Math.max( margin.right, m.right );
                if ( yDir < 0 ) {
                    if ( textPlacement=='outside' )
                        margin.top = Math.max( margin.top, m.ascent + ( axis.ticks ? tick1 : 0 ) + axis.textStyle.fontSize * 0.2 );
                    else
                        margin.bottom = Math.max( margin.bottom, m.descent + ( axis.ticks ? tick2 : 0 ) + axis.textStyle.fontSize * 0.2 );
                } else {
                    if ( textPlacement=='outside' )
                        margin.bottom = Math.max( margin.bottom, m.descent + ( axis.ticks ? tick1 : 0 ) + axis.textStyle.fontSize * 0.2 );
                    else
                        margin.top = Math.max( margin.top, m.ascent + ( axis.ticks ? tick2 : 0 ) + axis.textStyle.fontSize * 0.2 );
                }
            }
        }

        return margin;
    }

    function drawGridlines( ctx, axis, otherAxis, orientation, h, v, width, height ) {
        var size = orientation=='y' ? height : width;

        if ( axis.gridlinesStrokeStyle ) {
            setStrokeStyle( ctx, axis.gridlinesStrokeStyle );
            ctx.beginPath();
            if ( axis.ticks ) {
                for ( var i=0; i < axis.ticks.length; i++ ) {
                    var w = axis.ticks[i].v;
                    var grid = ( w - axis.minValue ) / ( axis.maxValue - axis.minValue ) * size;
                    if ( otherAxis.axisStrokeStyle && w==axis.minValue || axis.baselineStrokeStyle && w==axis.baseline )
                        continue;
                    if ( orientation=='y' ) {
                        ctx.moveTo( h(0), v(grid) );
                        ctx.lineTo( h(width), v(grid) );
                    } else {
                        ctx.moveTo( h(grid), v(0) );
                        ctx.lineTo( h(grid), v(height) );
                    }
                }
            } else {
                for ( var i=0; i <= axis.categories.length; i++ ) {
                    if ( i==0 && ( otherAxis.axisStrokeStyle || axis.baselineStrokeStyle ) )
                        continue;
                    var grid =  i / axis.categories.length * size;
                    if ( orientation=='y' ) {
                        ctx.moveTo( h(0), v(grid) );
                        ctx.lineTo( h(width), v(grid) );
                    } else {
                        ctx.moveTo( h(grid), v(0) );
                        ctx.lineTo( h(grid), v(height) );
                    }
                }
            }
            ctx.stroke();
        }

        if ( axis.minorGridlines && axis.tickInterval && axis.minorGridlines.count > 0 ) {
            setStrokeStyle( ctx, axis.minorGridlines.strokeStyle );
            ctx.beginPath();
            var minorInterval = axis.tickInterval / ( axis.minorGridlines.count + 1 );
            var iMin = Math.ceil( axis.minValue / minorInterval ), iMax = Math.floor( axis.maxValue / minorInterval );
            for ( var i=iMin; i <= iMax; i++ ) {
                var w = axis.baseline + i * minorInterval;
                var grid = ( w - axis.minValue ) / ( axis.maxValue - axis.minValue ) * size;
                if ( i%(axis.minorGridlines.count+1)==0 || otherAxis.axisStrokeStyle && w==axis.minValue || axis.baselineStrokeStyle && w==axis.baseline )
                    continue;
                if ( orientation=='y' ) {
                    ctx.moveTo( h(0), v(grid) );
                    ctx.lineTo( h(width), v(grid) );
                } else {
                    ctx.moveTo( h(grid), v(0) );
                    ctx.lineTo( h(grid), v(height) );
                }
            }
            ctx.stroke();
        }

        if ( axis.baselineStrokeStyle && ( !otherAxis.axisStrokeStyle || axis.baseline!=axis.minValue ) ) {
            var baseline = ( axis.baseline - axis.minValue ) / ( axis.maxValue - axis.minValue ) * size;
            setStrokeStyle( ctx, axis.baselineStrokeStyle );
            ctx.beginPath();
            if ( orientation=='y' ) {
                ctx.moveTo( h(0), v(baseline) );
                ctx.lineTo( h(width), v(baseline) );
            } else {
                ctx.moveTo( h(baseline), v(0) );
                ctx.lineTo( h(baseline), v(height) );
            }
            ctx.stroke();
        }
    }

    function drawCircularThresholds( ctx, thresholds, cx, cy, radius, startAngle, barRange, reverse, min, max ) {
        if ( thresholds.step ) {
            var i = 0;
            var n = Math.ceil( ( max - min ) / thresholds.step );
            var fan = barRange / n / 2 * thresholds.stepSize;
            for ( var j=0; j < n; j++ ) {
                var x = min + ( max - min ) * ( j + .5 ) / n;
                while ( i < thresholds.zones.length && ( thresholds.zones[i].from > x || thresholds.zones[i].to <= x ) )
                    ++i;
                if ( i >= thresholds.zones.length )
                    break;
                var z = thresholds.zones[i], f = (x-z.from)/(z.to-z.from);
                if ( thresholds.fill=='gradient' ) {
                    ctx.fillStyle = z.startColor;
                    var c1 = getRGBA( ctx.fillStyle );
                    ctx.fillStyle = z.endColor;
                    var c2 = getRGBA( ctx.fillStyle );
                    ctx.fillStyle = putRGBA3( c1, c2, f );
                } else {
                    ctx.fillStyle = z.color;
                }
                if ( thresholds.position < 0 ) {
                    r1 = radius + thresholds.offset;
                    r2 = radius + thresholds.offset + z.startSize*(1-f) + z.endSize*f;
                } else if ( thresholds.position > 0 ) {
                    r1 = radius - thresholds.offset;
                    r2 = radius - thresholds.offset - z.startSize*(1-f) - z.endSize*f;
                } else {
                    r1 = radius + thresholds.offset - z.startSize*(1-f)/2 - z.endSize*f/2;
                    r2 = radius + thresholds.offset + z.startSize*(1-f)/2 + z.endSize*f/2;
                }
                ctx.beginPath();
                var angle = ( j + .5 ) * barRange / n;
                ctx.arc( cx, cy, r1, startAngle+angle-fan, startAngle+angle+fan, reverse );
                ctx.arc( cx, cy, r2, startAngle+angle+fan, startAngle+angle-fan, !reverse );
                ctx.fill();
            }
        } else {
            for ( var i=0; i < thresholds.zones.length; i++ ) {
                var z = thresholds.zones[i];
                var from = ( z.from - min ) / ( max - min ) * barRange;
                var to = ( z.to - min ) / ( max - min ) * barRange;
                ctx.fillStyle = z.color;
                var rs1, re1, rs2, re2;
                if ( thresholds.position < 0 ) {
                    rs1 = re1 = radius + thresholds.offset;
                    rs2 = radius + thresholds.offset + z.startSize;
                    re2 = radius + thresholds.offset + z.endSize;
                } else if ( thresholds.position > 0 ) {
                    rs1 = re1 = radius - thresholds.offset;
                    rs2 = radius - thresholds.offset - z.startSize;
                    re2 = radius - thresholds.offset - z.endSize;
                } else {
                    rs1 = radius + thresholds.offset - z.startSize/2;
                    re1 = radius + thresholds.offset - z.endSize/2;
                    rs2 = radius + thresholds.offset + z.startSize/2;
                    re2 = radius + thresholds.offset + z.endSize/2;
                }
                ctx.beginPath();
                if ( rs1==re1 ) {
                    ctx.arc( cx, cy, rs1, startAngle+from, startAngle+to, reverse );
                } else {
                    var n = Math.min( Math.ceil( Math.abs( to - from ) / .07 ), 50 );
                    ctx.moveTo( cx + Math.cos( startAngle+from ) * rs1, cy + Math.sin( startAngle+from ) * rs1 );
                    for ( var j=1; j <= n; j++ ) {
                        var t = startAngle + from + ( to - from ) * j / n;
                        var r = rs1 + ( re1 - rs1 ) * j / n;
                        ctx.lineTo( cx + Math.cos( t ) * r, cy + Math.sin( t ) * r );
                    }
                }
                if ( rs2==re2 ) {
                    ctx.arc( cx, cy, rs2, startAngle+to, startAngle+from, !reverse );
                } else {
                    var n = Math.min( Math.ceil( Math.abs( from - to ) / .07 ), 50 );
                    ctx.lineTo( cx + Math.cos( startAngle+to ) * re2, cy + Math.sin( startAngle+to ) * re2 );
                    for ( var j=1; j <= n; j++ ) {
                        var t = startAngle + to + ( from - to ) * j / n;
                        var r = re2 + ( rs2 - re2 ) * j / n;
                        ctx.lineTo( cx + Math.cos( t ) * r, cy + Math.sin( t ) * r );
                    }
                }
                ctx.closePath();
                ctx.fill();
            }
        }
    }

    function drawThresholds( ctx, thresholds, orientation, h, v, barWidth, barRange, min, max, cutoffStart, cutoffEnd, layerWidth, layerHeight ) {
        if ( thresholds.step ) {
            var i = 0;
            var n = Math.ceil( ( max - min ) / thresholds.step );
            var fan = barRange / n / 2 * thresholds.stepSize;
            for ( var j=0; j < n; j++ ) {
                var x = min + ( max - min ) * ( j + .5 ) / n;
                if ( cutoffEnd!==undefined && x >= cutoffEnd )
                    break;
                if ( cutoffStart!==undefined && x < cutoffStart )
                    continue;
                while ( i < thresholds.zones.length && ( thresholds.zones[i].from > x || thresholds.zones[i].to <= x ) )
                    ++i;
                if ( i >= thresholds.zones.length )
                    break;
                var z = thresholds.zones[i], f = (x-z.from)/(z.to-z.from);
                if ( thresholds.fill=='gradient' ) {
                    ctx.fillStyle = z.startColor;
                    var c1 = getRGBA( ctx.fillStyle );
                    ctx.fillStyle = z.endColor;
                    var c2 = getRGBA( ctx.fillStyle );
                    ctx.fillStyle = putRGBA3( c1, c2, f );
                } else {
                    ctx.fillStyle = z.color;
                }
                var size = z.startSize*(1-f) + z.endSize*f;
                var pos = ( j + .5 ) * barRange / n;
                if ( orientation=='y' ) {
                    if ( thresholds.position )
                        ctx.fillRect( h(-thresholds.offset), v(pos-fan), h(-thresholds.offset-size)-h(-thresholds.offset), v(pos+fan)-v(pos-fan) );
                    else
                        ctx.fillRect( h((barWidth-size)/2+thresholds.offset), v(pos-fan), h((barWidth+size)/2+thresholds.offset)-h((barWidth-size)/2+thresholds.offset), v(pos+fan)-v(pos-fan) );
                } else {
                    if ( thresholds.position )
                        ctx.fillRect( h(pos-fan), v(-thresholds.offset), h(pos+fan)-h(pos-fan), v(-thresholds.offset-size)-v(-thresholds.offset) );
                    else
                        ctx.fillRect( h(pos-fan), v((barWidth-size)/2+thresholds.offset), h(pos+fan)-h(pos-fan), v((barWidth+size)/2+thresholds.offset)-v((barWidth-size)/2+thresholds.offset) );
                }
            }
        } else {
            if ( cutoffStart!==undefined ) {
                ctx.save();
                ctx.beginPath();
                cutoffStart = ( cutoffStart - min ) / ( max - min ) * barRange;
                cutoffEnd = ( cutoffEnd - min ) / ( max - min ) * barRange;
                if ( orientation=='y' )
                    ctx.rect( 0, v(cutoffStart), layerWidth, v(cutoffEnd)-v(cutoffStart) );
                else
                    ctx.rect( h(cutoffStart), 0, h(cutoffEnd)-h(cutoffStart), layerHeight );
                ctx.clip();
            }
            for ( var i=0; i < thresholds.zones.length; i++ ) {
                var z = thresholds.zones[i];
                var from = ( z.from - min ) / ( max - min ) * barRange;
                var to = ( z.to - min ) / ( max - min ) * barRange;
                if ( thresholds.fill=='gradient' ) {
                    var gr;
                    if ( orientation=='y' )
                        gr = ctx.createLinearGradient( h(thresholds.offset+barWidth/2), v(from), h(thresholds.offset+barWidth/2), v(to) );
                    else
                        gr = ctx.createLinearGradient( h(from), v(thresholds.offset+barWidth/2), h(to), v(thresholds.offset+barWidth/2) );
                    gr.addColorStop( 0, z.startColor );
                    gr.addColorStop( 1, z.endColor );
                    ctx.fillStyle = gr;
                } else {
                    ctx.fillStyle = z.color;
                }
                if ( z.endSize==z.startSize ) {
                    if ( orientation=='y' ) {
                        if ( thresholds.position )
                            ctx.fillRect( h(-thresholds.offset), v(from), h(-thresholds.offset-z.startSize)-h(-thresholds.offset), v(to)-v(from) );
                        else
                            ctx.fillRect( h((barWidth-z.startSize)/2+thresholds.offset), v(from), h((barWidth+z.startSize)/2+thresholds.offset)-h((barWidth-z.startSize)/2+thresholds.offset), v(to)-v(from) );
                    } else {
                        if ( thresholds.position )
                            ctx.fillRect( h(from), v(-thresholds.offset), h(to)-h(from), v(-thresholds.offset-z.startSize)-v(-thresholds.offset) );
                        else
                            ctx.fillRect( h(from), v((barWidth-z.startSize)/2+thresholds.offset), h(to)-h(from), v((barWidth+z.startSize)/2+thresholds.offset)-v((barWidth-z.startSize)/2+thresholds.offset) );
                    }
                } else {
                    ctx.beginPath();
                    if ( thresholds.position ) {
                        if ( orientation=='y' ) {
                            ctx.moveTo( h(-thresholds.offset), v(from) );
                            ctx.lineTo( h(-thresholds.offset-z.startSize), v(from) );
                            ctx.lineTo( h(-thresholds.offset-z.endSize), v(to) );
                            ctx.lineTo( h(-thresholds.offset), v(to) );
                        } else {
                            ctx.moveTo( h(from), v(-thresholds.offset) );
                            ctx.lineTo( h(from), v(-thresholds.offset-z.startSize) );
                            ctx.lineTo( h(to), v(-thresholds.offset-z.endSize) );
                            ctx.lineTo( h(to), v(-thresholds.offset) );
                        }
                    } else {
                        if ( orientation=='y' ) {
                            ctx.moveTo( h((barWidth+z.startSize)/2+thresholds.offset), v(from) );
                            ctx.lineTo( h((barWidth-z.startSize)/2+thresholds.offset), v(from) );
                            ctx.lineTo( h((barWidth-z.endSize)/2+thresholds.offset), v(to) );
                            ctx.lineTo( h((barWidth+z.endSize)/2+thresholds.offset), v(to) );
                        } else {
                            ctx.moveTo( h(from), v((barWidth+z.startSize)/2+thresholds.offset) );
                            ctx.lineTo( h(from), v((barWidth-z.startSize)/2+thresholds.offset) );
                            ctx.lineTo( h(to), v((barWidth-z.endSize)/2+thresholds.offset) );
                            ctx.lineTo( h(to), v((barWidth+z.endSize)/2+thresholds.offset) );
                        }
                    }
                    ctx.closePath();
                    ctx.fill();
                }
            }
            if ( cutoffStart!==undefined )
                ctx.restore();
        }
    }

    function autoSetCircularIndicator( i, layerMax, offsetRef, defaultLength ) {
        indicator = {};
        i = typeof(i)=='object' ? i : {};
        indicator.shape = i.shape || 'needle';
        indicator.position = i.position=='in' || i.position=='out' || i.position=='bar' ? i.position : 'center';
        if ( indicator.position=='center' || indicator.shape=='needle' ) {
            indicator.length = jSignage.relAbs( i.length, offsetRef, defaultLength );
            indicator.width = jSignage.relAbs( i.width, offsetRef, indicator.length*.1 );
        } else {
            indicator.length = jSignage.relAbs( i.length, offsetRef, defaultLength );
            indicator.width = jSignage.relAbs( i.width, offsetRef, indicator.length );
        }
        autoSetIndicator( indicator, i, layerMax, offsetRef );
        return indicator;
    }

    function autoSetLinearIndicator( i, orientation, layerMax, offsetRef, defaultLength ) {
        indicator = {};
        i = typeof(i)=='object' ? i : {};
        indicator.shape = i.shape || 'needle';
        if ( orientation=='y' )
            indicator.position = i.position=='left' || i.position=='right' ? i.position : 'bar';
        else
            indicator.position = i.position=='top' || i.position=='bottom' ? i.position : 'bar';
        indicator.length = jSignage.relAbs( i.length, offsetRef, defaultLength );
        if ( indicator.shape=='needle' )
            indicator.width = jSignage.relAbs( i.width, offsetRef, indicator.length*.2 );
        else
            indicator.width = jSignage.relAbs( i.width, offsetRef, indicator.length );
        autoSetIndicator( indicator, i, layerMax, offsetRef );
        return indicator;
    }

    function autoSetIndicator( indicator, i, layerMax, offsetRef ) {
        var indicatorStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMax * .001, 1 ), color: '#000' }, null );
        indicator.strokeStyle = i.strokeStyle ? new StrokeStyle( i.strokeStyle, indicatorStrokeStyle ) : null;
        indicator.color = i.color || '#888';
        indicator.offset = jSignage.relAbs( i.offset, offsetRef, 0 );
        indicator.placement = i.placement || 'outside';
        if ( indicator.shape=='needle' ) {
            var tip = i.tip || {}, back = i.back || {};
            indicator.tip = {
                width: jSignage.relAbs( tip.width, indicator.width, 0 ),
                shape: tip.shape || 'sharp'
            };
            indicator.tip.size = jSignage.relAbs( tip.size, indicator.tip.width, indicator.tip.width/2 );
            indicator.back = {
                length: jSignage.relAbs( back.length, indicator.length, indicator.length*.25 ),
                shape: back.shape || 'straight',
                inversed: back.inversed || false
            };
            indicator.back.width = jSignage.relAbs( back.width, indicator.width, indicator.width + (indicator.width-indicator.tip.width)/indicator.length*indicator.back.length );
            indicator.back.size = jSignage.relAbs( back.size, indicator.back.width, indicator.back.width/2 );
        }
        if ( 'button' in i && !i.button ) {
            indicator.button = null;
        } else if ( i.button || indicator.position=='center' ) {
            var b = typeof(i.button)=='object' ? i.button : {};
            indicator.button = {
                radius: jSignage.relAbs( b.radius, offsetRef, indicator.width ),
                strokeStyle:  b.strokeStyle ? new StrokeStyle( b.strokeStyle, indicatorStrokeStyle ) : indicatorStrokeStyle,
                color: b.color || '#888'
            };
        }
    }

    function measureCircularIndicator( indicator, radius ) {
        var area = measureIndicator( indicator );
        if ( indicator.position=='out' && indicator.placement!='inside' || indicator.position=='in' && indicator.placement=='inside' )
            area.x += radius+indicator.offset;
        else if ( indicator.position=='in' && indicator.placement!='inside' || indicator.position=='out' && indicator.placement=='inside' )
            area.x += radius-indicator.offset-indicator.length;
        else if ( indicator.position=='bar' )
            area.x += radius+indicator.offset-indicator.length/2;
        return area;        
    }

    function drawCircularIndicator( ctx, indicator, cx, cy, angle, radius ) {
        ctx.save();
        ctx.translate( cx, cy );
        ctx.rotate( angle );
        if ( indicator.position=='out' && indicator.placement!='inside' || indicator.position=='in' && indicator.placement=='inside' ) {
            ctx.translate( radius+indicator.offset+indicator.length, 0 );
            ctx.transform( -1, 0, 0, 1, 0, 0 );
        } else if ( indicator.position=='in' && indicator.placement!='inside' || indicator.position=='out' && indicator.placement=='inside' ) {
            ctx.translate( radius-indicator.offset-indicator.length, 0 );
        } else if ( indicator.position=='bar' ) {
            ctx.translate( radius+indicator.offset-indicator.length/2, 0 );
        }
        drawIndicator( ctx, indicator );
        ctx.restore();
    }

    function measureLinearIndicator( indicator, orientation, chartArea ) {
        var area = measureIndicator( indicator );
        if ( orientation=='y' ) {
            if ( indicator.position=='left' ) {
                if ( indicator.placement=='inside' )
                    area.x += chartArea.x + indicator.offset;
                else
                    area.x += chartArea.x - indicator.length - indicator.offset;
            } else if ( indicator.position=='right' ) {
                if ( indicator.placement=='inside' )
                    area.x += chartArea.x + chartArea.width - indicator.length - indicator.offset;
                else
                    area.x += chartArea.x + chartArea.width + indicator.offset;
            } else {
                area.x += chartArea.x + chartArea.width/2 - indicator.length/2 + indicator.offset;
            }
        } else {
            area = { x: area.y, y: area.x, width: area.height, height: area.width };
            if ( indicator.position=='top' ) {
                if ( indicator.placement=='inside' )
                    area.y += chartArea.y + indicator.offset;
                else
                    area.y += chartArea.y - indicator.length - indicator.offset;
            } else if ( indicator.position=='bottom' ) {
                if ( indicator.placement=='inside' )
                    area.y += chartArea.y + chartArea.height - indicator.length - indicator.offset;
                else
                    area.y += chartArea.y + chartArea.height + indicator.offset;
            } else {
                area.y += chartArea.y + chartArea.height/2 - indicator.length/2 + indicator.offset;
            }
        }
        return area;
    }

    function drawLinearIndicator( ctx, indicator, orientation, chartArea, where ) {
        ctx.save();
        if ( orientation=='y' ) {
            if ( indicator.position=='left' ) {
                if ( indicator.placement=='inside' ) {
                    ctx.translate( chartArea.x+indicator.length+indicator.offset, where );
                    ctx.transform( -1, 0, 0, 1, 0, 0 );
                } else {
                    ctx.translate( chartArea.x-indicator.length-indicator.offset, where );
                }
            } else if ( indicator.position=='right' ) {
                if ( indicator.placement=='inside' ) {
                    ctx.translate( chartArea.x+chartArea.width-indicator.length-indicator.offset, where );
                } else {
                    ctx.translate( chartArea.x+chartArea.width+indicator.length+indicator.offset, where );
                    ctx.transform( -1, 0, 0, 1, 0, 0 );
                }
            } else {
                ctx.translate( chartArea.x+chartArea.width/2-indicator.length/2+indicator.offset, where );
            }
        } else {
            if ( indicator.position=='top' ) {
                if ( indicator.placement=='inside' ) {
                    ctx.translate( where, chartArea.y+indicator.length+indicator.offset );
                    ctx.transform( 0, -1, 1, 0, 0, 0 );
                } else {
                    ctx.translate( where, chartArea.y-indicator.length-indicator.offset );
                    ctx.transform( 0, 1, -1, 0, 0, 0 );
                }
            } else if ( indicator.position=='bottom' ) {
                if ( indicator.placement=='inside' ) {
                    ctx.translate( where, chartArea.y+chartArea.height-indicator.length-indicator.offset );
                    ctx.transform( 0, 1, -1, 0, 0, 0 );
                } else {
                    ctx.translate( where, chartArea.y+chartArea.height+indicator.length+indicator.offset );
                    ctx.transform( 0, -1, 1, 0, 0, 0 );
                }
            } else {
                ctx.translate( where, chartArea.y+chartArea.height/2-indicator.length/2+indicator.offset );
                ctx.transform( 0, 1, -1, 0, 0, 0 );
            }
        }
        drawIndicator( ctx, indicator );
        ctx.restore();
    }

    function measureIndicator( indicator ) {
        var x1, y1, x2, y2;
        if ( indicator.shape=='rect' || indicator.shape=='ellipse' || indicator.shape=='triangle' || indicator.shape=='arrow' ) {
            x1 = 0;
            y1 = -indicator.width/2;
            x2 = indicator.length;
            y2 = indicator.width/2;
        } else {
            x1 = -indicator.back.length;
            if ( !indicator.back.inversed ) {
                if ( indicator.back.shape=='round' ) {
                    var a = ( indicator.width - indicator.back.width ) / ( 2 * indicator.back.length );
                    var x = a * ( indicator.back.width / 2 );
                    var r = Math.sqrt( x*x + indicator.back.width*indicator.back.width/4 );
                    x1 -= r - x;
                } else if ( indicator.back.shape=='sharp' ) {
                    x1 -= indicator.tip.size;
                }
            }
            x2 = indicator.length;
            y1 = -Math.max( indicator.width, indicator.back.width, indicator.tip.width ) / 2;
            y2 = -y1;
        }
        if ( indicator.strokeStyle ) {
            x1 -= indicator.strokeStyle.lineWidth / 2;
            y1 -= indicator.strokeStyle.lineWidth / 2;
            x2 += indicator.strokeStyle.lineWidth / 2;
            y2 += indicator.strokeStyle.lineWidth / 2;
        }
        if ( indicator.button ) {
            var r = indicator.button.radius + ( indicator.button.strokeStyle.lineWidth > 0 ? indicator.button.strokeStyle.lineWidth / 2 : 0 );
            x1 = Math.min( x1, -r );
            y1 = Math.min( y1, -r );
            x2 = Math.max( x2, r );
            y2 = Math.max( y2, r );
        }
        return { x: x1, y: y1, width: x2-x1, height: y2-y1 };
    }

    function drawIndicator( ctx, indicator, cx, cy, angle ) {
        ctx.beginPath();
        if ( indicator.shape=='rect') {
            ctx.rect( 0, -indicator.width/2, indicator.length, indicator.width );
        } else if ( indicator.shape=='ellipse' ) {
            ctx.ellipse( indicator.length/2, 0, indicator.length/2, indicator.width/2, 0, 0, 2*Math.PI, false );
        } else if ( indicator.shape=='triangle' ) {
            ctx.moveTo( 0, -indicator.width/2 );
            ctx.lineTo( indicator.length, 0 );
            ctx.lineTo( 0, indicator.width/2 );
        } else if ( indicator.shape=='arrow' ) {
            ctx.moveTo( 0, -indicator.width/4 );
            ctx.lineTo( indicator.length-indicator.width/2, -indicator.width/4 );
            ctx.lineTo( indicator.length-indicator.width/2, -indicator.width/2 );
            ctx.lineTo( indicator.length, 0 );
            ctx.lineTo( indicator.length-indicator.width/2, indicator.width/2 );
            ctx.lineTo( indicator.length-indicator.width/2, indicator.width/4 );
            ctx.lineTo( 0, indicator.width/4 );
        } else {
            ctx.moveTo( 0, -indicator.width/2 );
            if ( indicator.tip.width <= 0 ) {
                ctx.lineTo( indicator.length, 0 );
            } else {
                var length = indicator.length;
                if ( indicator.tip.shape=='round' ) {
                    var a = ( indicator.width - indicator.tip.width ) / ( 2 * indicator.length );
                    var x = a * ( indicator.tip.width / 2 );
                    var r = Math.sqrt( x*x + indicator.tip.width*indicator.tip.width/4 );
                    length += x - r;
                } else if ( indicator.tip.shape=='sharp' ) {
                    length -= indicator.tip.size;
                }
                ctx.lineTo( length, -indicator.tip.width/2 );
                if ( indicator.tip.shape=='round' ) {
                    var a = ( indicator.width - indicator.tip.width ) / ( 2 * length );
                    var x = a * ( indicator.tip.width / 2 );
                    var r = Math.sqrt( x*x + indicator.tip.width*indicator.tip.width/4 );
                    var b = Math.PI/2 - Math.atan( a );
                    ctx.arc( length - x, 0, r, -b, b );
                } else if ( indicator.tip.shape=='sharp' ) {
                    ctx.lineTo( length+indicator.tip.size, 0 );
                    ctx.lineTo( length, indicator.tip.width/2 );
                } else {
                    ctx.lineTo( length, indicator.tip.width/2 );
                }
            }
            ctx.lineTo( 0, indicator.width/2 );
            if ( indicator.back.width/2 <= 0 ) {
                ctx.lineTo( -indicator.back.length, 0 );
            } else {
                ctx.lineTo( -indicator.back.length, indicator.back.width/2 );
                var dir = indicator.back.inversed ? -1 : 1;
                if ( indicator.back.shape=='round' ) {
                    var a = ( indicator.width - indicator.back.width ) / ( 2 * indicator.back.length );
                    var x = a * ( indicator.back.width / 2 );
                    var r = Math.sqrt( x*x + indicator.back.width*indicator.back.width/4 );
                    var b = Math.PI/2 - Math.atan( a );
                    if ( dir < 0 ) {
                        if ( indicator.back.width < indicator.width )
                            ctx.arc( -indicator.back.length - x, 0, r, b, -b, true );
                        else
                            ctx.arc( -indicator.back.length + x, 0, r, Math.PI-b, Math.PI+b, true );
                    } else {
                        ctx.arc( -indicator.back.length + x, 0, r, Math.PI-b, Math.PI+b );
                    }
                } else if ( indicator.back.shape=='sharp' ) {
                    ctx.lineTo( -indicator.back.length-dir*indicator.back.size, 0 );
                    ctx.lineTo( -indicator.back.length, -indicator.back.width/2 );
                } else {
                    ctx.lineTo( -indicator.back.length, -indicator.back.width/2 );
                }
            }
        }
        ctx.closePath();
        ctx.fillStyle = indicator.color;
        ctx.fill();
        if ( indicator.strokeStyle ) {
            setStrokeStyle( ctx, indicator.strokeStyle );
            ctx.stroke();
        }
        if ( indicator.button ) {
            ctx.beginPath();
            ctx.arc( 0, 0, indicator.button.radius, 0, 2*Math.PI );
            ctx.fillStyle = indicator.button.color;
            ctx.fill();
            if ( indicator.button.strokeStyle.lineWidth > 0 ) {
                setStrokeStyle( ctx, indicator.button.strokeStyle );
                ctx.stroke();
            }
        }
    }

    function getTextOnCircleArea( ctx, cx, cy, textRadius, textAngle, textPlacement, text ) {
        var metrics = jSignage.measureText( ctx, text );
        var nx = Math.cos( textAngle ), ny = Math.sin( textAngle );
        if ( textPlacement=='inside' || textPlacement=='outside' ) {
            var d1 = nx * -metrics.actualBoundingBoxLeft + ny * -metrics.actualBoundingBoxAscent;
            var d2 = nx * metrics.actualBoundingBoxRight + ny * -metrics.actualBoundingBoxAscent;
            var d3 = nx * metrics.actualBoundingBoxRight + ny * metrics.actualBoundingBoxDescent;
            var d4 = nx * -metrics.actualBoundingBoxLeft + ny * metrics.actualBoundingBoxDescent;
            var d = textPlacement=='outside' ? Math.min( d1, d2, d3, d4 ) : Math.max( d1, d2, d3, d4 );
            textRadius = Math.max( textRadius - d, 0 );
        }
        var tx = cx + nx * textRadius, ty = cy + ny * textRadius;
        return {
            textX: tx,
            textY: ty,
            x: tx - metrics.actualBoundingBoxLeft,
            y: ty - metrics.actualBoundingBoxAscent,
            width: metrics.actualBoundingBoxRight + metrics.actualBoundingBoxLeft,
            height: metrics.actualBoundingBoxDescent + metrics.actualBoundingBoxAscent,
            addY: -metrics.alphabeticBaseline
        };
    }

    function drawTextOnCircle( ctx, cx, cy, textRadius, textAngle, out, text ) {
        var metrics = jSignage.measureText( ctx, text );
        var nx = Math.cos( textAngle ), ny = Math.sin( textAngle );
        var d1 = nx * -metrics.actualBoundingBoxLeft + ny * -metrics.actualBoundingBoxAscent;
        var d2 = nx * metrics.actualBoundingBoxRight + ny * -metrics.actualBoundingBoxAscent;
        var d3 = nx * metrics.actualBoundingBoxRight + ny * metrics.actualBoundingBoxDescent;
        var d4 = nx * -metrics.actualBoundingBoxLeft + ny * metrics.actualBoundingBoxDescent;
        var d = out ? Math.min( d1, d2, d3, d4 ) : Math.max( d1, d2, d3, d4 );
        textRadius = Math.max( textRadius - d, 0 );
        ctx.fillText( text, cx + nx * textRadius, cy + ny * textRadius );
    }

    function drawCircularAxis( ctx, axis, cx, cy, radius, startAngle, sizeAngle ) {
        var tickOut = 0, tickIn = 0, aw = 0;

        radius -= axis.offset * axis.position;

        if ( axis.axisStrokeStyle ) {
            aw = axis.axisStrokeStyle.lineWidth / 2;
            setStrokeStyle( ctx, axis.axisStrokeStyle );
            ctx.beginPath();
            ctx.arc( cx, cy, radius, startAngle, startAngle+sizeAngle, axis.reverse );
            ctx.stroke();
        }

        if ( axis.tickMarks ) {
            tickOut = ( axis.position < 0 ? axis.tickMarks.placement!='inside' : axis.tickMarks.placement!='outside' ) ? axis.tickMarks.length : aw;
            tickIn = ( axis.position < 0 ? axis.tickMarks.placement!='outside' : axis.tickMarks.placement!='inside' ) ? axis.tickMarks.length : aw;
            setStrokeStyle( ctx, axis.tickMarks.strokeStyle );
            ctx.beginPath();
            for ( var i=0; i < axis.ticks.length; i++ ) {
                var tickAngle = ( axis.ticks[i].v - axis.minValue ) / ( axis.maxValue - axis.minValue ) * sizeAngle + startAngle;
                ctx.moveTo( cx + Math.cos( tickAngle ) * ( radius + tickOut ), cy + Math.sin( tickAngle ) * ( radius + tickOut ) );
                ctx.lineTo( cx + Math.cos( tickAngle ) * ( radius - tickIn ), cy + Math.sin( tickAngle ) * ( radius - tickIn ) );
            }
            ctx.stroke();
        }

        if ( axis.minorTickMarks && axis.tickInterval && axis.minorTickMarks.count > 0 ) {
            var minorTickOut = ( axis.position < 0 ? axis.tickMarks.placement!='inside' : axis.tickMarks.placement!='outside' ) ? axis.minorTickMarks.length : aw;
            var minorTickIn = ( axis.position < 0 ? axis.tickMarks.placement!='outside' : axis.tickMarks.placement!='inside' ) ? axis.minorTickMarks.length : aw;
            setStrokeStyle( ctx, axis.minorTickMarks.strokeStyle );
            ctx.beginPath();
            var minorInterval = axis.tickInterval / ( axis.minorTickMarks.count + 1 );
            var iMin = Math.ceil( axis.minValue / minorInterval ), iMax = Math.floor( axis.maxValue / minorInterval );
            for ( var i=iMin; i <= iMax; i++ ) {
                if ( i%(axis.minorTickMarks.count+1)==0 )
                    continue;
                var tickAngle = ( axis.baseline + i * minorInterval - axis.minValue ) / ( axis.maxValue - axis.minValue ) * sizeAngle + startAngle;
                ctx.moveTo( cx + Math.cos( tickAngle ) * ( radius + minorTickOut ), cy + Math.sin( tickAngle ) * ( radius + minorTickOut ) );
                ctx.lineTo( cx + Math.cos( tickAngle ) * ( radius - minorTickIn ), cy + Math.sin( tickAngle ) * ( radius - minorTickIn ) );
            }
            ctx.stroke();
        }
        
        if ( axis.textPlacement!='none' ) {
            var out = axis.position < 0 ? axis.textPlacement=='outside' : axis.textPlacement!='outside';
            var textRadius = out ? radius + tickOut + axis.textStyle.fontSize * 0.2 : radius - tickIn - axis.textStyle.fontSize * 0.2;
            ctx.font = axis.textStyle.font();
            ctx.fillStyle = axis.textStyle.color;
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            for ( var i=0; i < axis.ticks.length; i++ ) {
                var textAngle = ( axis.ticks[i].v - axis.minValue ) / ( axis.maxValue - axis.minValue ) * sizeAngle + startAngle;
                drawTextOnCircle( ctx, cx, cy, textRadius, textAngle, out, axis.ticks[i].f );
            }
        }
    }

    function drawAxis( ctx, axis, orientation, h, v, xDir, yDir, size ) {
        var tick1 = 0, tick2 = 0;

        if ( axis.axisStrokeStyle ) {
            setStrokeStyle( ctx, axis.axisStrokeStyle );
            ctx.beginPath();
            if ( orientation=='y' ) {
                ctx.moveTo( h(0), v(0) );
                ctx.lineTo( h(0), v(size) );
            } else {
                ctx.moveTo( h(0), v(0) );
                ctx.lineTo( h(size), v(0) );
            }
            ctx.stroke();
        }

        if ( axis.tickMarks ) {
            tick1 = axis.tickMarks.placement=='outside' || axis.tickMarks.placement=='across' ? -axis.tickMarks.length : 0;
            tick2 = axis.tickMarks.placement=='inside' || axis.tickMarks.placement=='across' ? axis.tickMarks.length : 0;
            setStrokeStyle( ctx, axis.tickMarks.strokeStyle );
            ctx.beginPath();
            if ( axis.ticks ) {
                for ( var i=0; i < axis.ticks.length; i++ ) {
                    var tick = ( axis.ticks[i].v - axis.minValue ) / ( axis.maxValue - axis.minValue ) * size;
                    if ( orientation=='y' ) {
                        ctx.moveTo( h(tick1), v(tick) );
                        ctx.lineTo( h(tick2), v(tick) );
                    } else {
                        ctx.moveTo( h(tick), v(tick1) );
                        ctx.lineTo( h(tick), v(tick2) );
                    }
                }
            } else {
                for ( var i=0; i <= axis.categories.length; i++ ) {
                    var tick =  i / axis.categories.length * size;
                    if ( orientation=='y' ) {
                        ctx.moveTo( h(tick1), v(tick) );
                        ctx.lineTo( h(tick2), v(tick) );
                    } else {
                        ctx.moveTo( h(tick), v(tick1) );
                        ctx.lineTo( h(tick), v(tick2) );
                    }
                }
            }
            ctx.stroke();
        }

        if ( axis.minorTickMarks && axis.tickInterval && axis.minorTickMarks.count > 0 ) {
            var minorTick1 = axis.tickMarks.placement=='outside' || axis.tickMarks.placement=='across' ? -axis.minorTickMarks.length : 0;
            var minorTick2 = axis.tickMarks.placement=='inside' || axis.tickMarks.placement=='across' ? axis.minorTickMarks.length : 0;
            setStrokeStyle( ctx, axis.minorTickMarks.strokeStyle );
            ctx.beginPath();
            var minorInterval = axis.tickInterval / ( axis.minorTickMarks.count + 1 );
            var iMin = Math.ceil( axis.minValue / minorInterval ), iMax = Math.floor( axis.maxValue / minorInterval );
            for ( var i=iMin; i <= iMax; i++ ) {
                if ( i%(axis.minorTickMarks.count+1)==0 )
                    continue;
                var tick = ( axis.baseline + i * minorInterval - axis.minValue ) / ( axis.maxValue - axis.minValue ) * size;
                if ( orientation=='y' ) {
                    ctx.moveTo( h(minorTick1), v(tick) );
                    ctx.lineTo( h(minorTick2), v(tick) );
                } else {
                    ctx.moveTo( h(tick), v(minorTick1) );
                    ctx.lineTo( h(tick), v(minorTick2) );
                }
            }
            ctx.stroke();
        }

        if ( axis.textPlacement!='none' ) {
            ctx.font = axis.textStyle.font();
            ctx.fillStyle = axis.textStyle.color;
            var textPlacement = axis.ticks ? axis.textPlacement : 'outside';
            if ( orientation=='y' ) {
                if ( textPlacement=='outside' ) {
                    ctx.textAlign = xDir < 0 ? 'left': 'right';
                    ctx.textBaseline = 'middle';
                } else {
                    ctx.textAlign = xDir < 0 ? 'right': 'left';
                    if ( textPlacement=='above' )
                        ctx.textBaseline = 'bottom';
                    else if ( textPlacement=='below' )
                        ctx.textBaseline = 'top';
                    else
                        ctx.textBaseline = 'middle';
                }
            } else {
                if ( textPlacement=='outside' ) {
                    ctx.textAlign = 'center';
                    ctx.textBaseline = yDir < 0 ? 'bottom' : 'top';
                } else {
                    ctx.textBaseline = yDir < 0 ? 'top' : 'bottom';
                    ctx.textAlign = 'center';
                }
            }

            var textA = textPlacement=='outside' ? ( axis.ticks ? tick1 : 0 ) - axis.textStyle.fontSize*0.2 : ( axis.ticks ? tick2 : 0 ) + axis.textStyle.fontSize*0.2;
            if ( axis.ticks ) {
                for ( var i=0; i < axis.ticks.length; i++ ) {
                    var textB = ( axis.ticks[i].v - axis.minValue ) / ( axis.maxValue - axis.minValue ) * size;
                    if ( orientation=='y' )
                        ctx.fillText( axis.ticks[i].f, h(textA), v(textB) );
                    else
                        ctx.fillText( axis.ticks[i].f, h(textB), v(textA) );
                }
            } else {
                for ( var i=0; i < axis.categories.length; i++ ) {
                    var text = axis.categories[i];
                    if ( typeof(text)=='object' && text ) {
                        var style = text.style ? new TextStyle( text.style, axis.textStyle ) : axis.textStyle;
                        ctx.font = style.font();
                        ctx.fillStyle = style.color;
                        text = text.text;
                    }
                    var textB = ( i + 0.5 ) / axis.categories.length * size;
                    if ( orientation=='y' )
                        ctx.fillText( text, h(textA), v(textB) );
                    else
                        ctx.fillText( text, h(textB), v(textA) );
                }
            }
        }
    }

    function addLegendAndAxes( ctx, width, height, legend, yAxis, xAxis, colors, labels, series, dataSize ) {
        var chartArea = addLegend( width, height, labels, legend, ctx, colors, series, dataSize );

        var xMargin = measureAxis( ctx, xAxis, 'x', xAxis.direction, yAxis.direction );
        var yMargin = measureAxis( ctx, yAxis, 'y', xAxis.direction, yAxis.direction );
        var marginLeft = xAxis.direction < 0 ? xMargin.left : Math.max( yMargin.left, xMargin.left );
        var marginRight = xAxis.direction < 0 ? Math.max( yMargin.right, xMargin.right ) : xMargin.right;
        var marginTop = yAxis.direction < 0 ? Math.max( xMargin.top, yMargin.top ) : yMargin.top;
        var marginBottom = yAxis.direction < 0 ? yMargin.bottom : Math.max( xMargin.bottom, yMargin.bottom );

        chartArea.x += marginLeft;
        chartArea.width -= marginLeft + marginRight;
        chartArea.y += marginTop;
        chartArea.height -= marginTop + marginBottom;

        function h( x ) { return xAxis.direction < 0 ? chartArea.x + chartArea.width - x : chartArea.x + x; }
        function v( y ) { return yAxis.direction < 0 ? chartArea.y + y : chartArea.y + chartArea.height - y; }

        drawGridlines( ctx, xAxis, yAxis, 'x', h, v, chartArea.width, chartArea.height );
        drawGridlines( ctx, yAxis, xAxis, 'y', h, v, chartArea.width, chartArea.height );
        drawAxis( ctx, xAxis, 'x', h, v, xAxis.direction, yAxis.direction, chartArea.width );
        drawAxis( ctx, yAxis, 'y', h, v, xAxis.direction, yAxis.direction, chartArea.height );

        return chartArea;
    }

    function findTickLimit( v ) {
        if ( v <= 0 )
            return v;

        var log = Math.log(v)/Math.LN10;
        var v1 = Math.floor( log ), v2 = log-v1, mul;

             if ( v2==0 )          mul = 1;
        else if ( v2 < 0.3010300 ) mul = 2;
        else if ( v2 < 0.3979400 ) mul = 2.5;
        else if ( v2 < 0.6989700 ) mul = 5;
        else                       mul = 10;

        return Math.pow( 10, v1 ) * mul;
    }

    function sum( v ) {
        var s = 0;
        for ( var i=0; i < v.length; i++ )
            s += v[i];
        return s;
    }

    function findMinMax( data, isGrouped, isStacked, groupSize ) {
        var max = null, min = null;
        for ( var i=0; i < data.length; i++ ) {
            var group = data[i];
            for ( var j=0; j < groupSize; j++ ) {
                var stack = isGrouped ? group[j] : group;
                value = isStacked ? sum( stack ) : stack;
                if ( max===null || max < value )
                    max = value;
                if ( min===null || min > value )
                    min = value;
            }
        }
        if ( max===null )
            return [ 0, 0 ];
        return [ min, max ];
    }

    function findMinMax2( data, isStacked ) {
        if ( isStacked ) {
            var stacks = [];
            for ( var i=0; i < data.length; i++ )
                for ( var j=0; j < data[i].length; j++ )
                    stacks[j] = ( stacks[j] || 0 ) + ( data[i][j] || 0 );
            data = [ stacks ];
        }
        var max = null, min = null, value;
        for ( var i=0; i < data.length; i++ ) {
            for ( var j=0; j < data[i].length; j++ ) if ( j in data[i] ) {
                value = data[i][j];
                if ( max===null || max < value )
                    max = value;
                if ( min===null || min > value )
                    min = value;
            }
        }
        if ( max===null )
            return [ 0, 0 ];
        return [ min, max ];
    }

    function TextStyle( A, B ) {
        if ( typeof(B)=='object' ) {
            this.color = A.color || B.color;
            this.fontSize = A.fontSize || B.fontSize;
            this.fontFamily = A.fontFamily || B.fontFamily;
            this.fontStyle = A.fontStyle || B.fontStyle;
            this.fontVariant = A.fontVariant || B.fontVariant;
            this.fontWeight = A.fontWeight || B.fontWeight;
        } else {
            this.color = 'black';
            this.fontSize = A.fontSize || B;
            this.fontFamily = A.fontFamily || 'Arial';
            this.fontStyle = A.fontStyle;
            this.fontVariant = A.fontVariant;
            this.fontWeight = A.fontWeight;
        }
    }

    TextStyle.prototype = {
        font: function() {
            var css = this.fontSize + 'px ' + this.fontFamily;
            if ( this.fontWeight )
                css = this.fontWeight + ' ' + css;
            if ( this.fontVariant )
                css = this.fontVariant + ' ' + css;
            if ( this.fontStyle )
                css = this.fontStyle + ' ' + css;
            return css;
        }
    };

    function StrokeStyle( A, B ) {
        if ( B ) {
            this.color = A.color || B.color;
            this.lineWidth = 'lineWidth' in A ? A.lineWidth : B.lineWidth;
            this.lineCap = A.lineCap || B.lineCap;
            this.lineJoin = A.lineJoin || B.lineJoin;
            this.miterLimit = 'miterLimit'in A ? A.miterLimit : B.miterLimit;
            this.lineDashOffset = 'lineDashOffset' in A ? A.lineDashOffset : B.lineDashOffset;
            this.lineDash = A.lineDash || B.lineDash;
        } else {
            this.color = A.color || 'black';
            this.lineWidth = 'lineWidth' in A ? A.lineWidth : 1;
            this.lineCap = A.lineCap || 'butt';
            this.lineJoin = A.lineJoin || 'miter';
            this.miterLimit = 'miterLimit'in A ? A.miterLimit : 10;
            this.lineDashOffset = A.lineDashOffset || 0;
            this.lineDash = A.lineDash || [];
        }
    }

    function setStrokeStyle( ctx, stroke ) {
        ctx.strokeStyle = stroke.color;
        ctx.lineWidth = stroke.lineWidth;
        ctx.lineCap = stroke.lineCap;
        ctx.lineJoin = stroke.lineJoin;
        ctx.miterLimit = stroke.miterLimit;
        ctx.lineDashOffset = stroke.lineDashOffset;
        if ( !jSignage.isArray( stroke.lineDash ) ) {
            if ( stroke.lineDash=='dots' ) {
                jSignage.setLineDash( ctx, [ 0, 2*stroke.lineWidth ] );
                ctx.lineCap = 'round';
            } else if ( stroke.lineDash=='squareDots' ) {
                jSignage.setLineDash( ctx, [ stroke.lineWidth, stroke.lineWidth ] );
                ctx.lineCap = 'butt';
            } else if ( stroke.lineDash=='dashes' ) {
                jSignage.setLineDash( ctx, [ stroke.lineWidth*2, stroke.lineWidth ] );
                ctx.lineCap = 'butt';
            } else if ( stroke.lineDash=='alternated' ) {
                jSignage.setLineDash( ctx, [ stroke.lineWidth*2, stroke.lineWidth, stroke.lineWidth, stroke.lineWidth ] );
                ctx.lineCap = 'butt';
            } else {
                jSignage.setLineDash( ctx, [] );
                ctx.lineCap = 'butt';
            }
        } else {
            jSignage.setLineDash( ctx, stroke.lineDash );
        }
    }

    function arePointsInPath( ctx, points ) {
        var M = ctx.currentTransform;
        var a = M.getComponent(0), b=M.getComponent(1), c=M.getComponent(2), d=M.getComponent(3), e=M.getComponent(4), f=M.getComponent(5);
        for ( var i=0; i < points.length; i+=2 ) {
            var x = points[i], y = points[i+1] ;
            var tx = a * x + c * y + e;
            var ty = b * x + d * y + f;
            if ( !ctx.isPointInPath( tx, ty ) )
                return false;
        }
        return true;
    }

    function isTextInPath( ctx, x, y, metrics, margin ) {
        return arePointsInPath( ctx, [
            x-metrics.actualBoundingBoxLeft-margin, y-metrics.actualBoundingBoxAscent-margin,
            x+metrics.actualBoundingBoxRight+margin, y-metrics.actualBoundingBoxAscent-margin,
            x+metrics.actualBoundingBoxRight+margin, y+metrics.actualBoundingBoxDescent+margin,
            x-metrics.actualBoundingBoxLeft-margin, y+metrics.actualBoundingBoxDescent+margin
        ]);
    }

    function applyVisibilityThreshold( data, visibilityThreshold, colors, residueColor, labels, residueLabel ) {
        var residue = 0, c = [], l = [], d = [];
        for ( var i=0; i < data.length; i++ ) {
            if ( data[i] < visibilityThreshold ) {
                residue += data[i];
            } else {
                d.push( data[i] );
                l.push( labels[i] || i+1 );
                c.push( getColor( colors, i ) );
            }
        }
        if ( residue ) {
            d.push( residue );
            l.push( residueLabel );
            c.push( residueColor );
        }
        return { data: d, labels: l, colors: c };
    }

    function getRGBA( color ) {
        var r, g, b, a;
        if ( color.substring( 0, 1 )=='#' ) {
            r = parseInt( color.substring( 1, 3 ), 16 );
            g = parseInt( color.substring( 3, 5 ), 16 );
            b = parseInt( color.substring( 5, 7 ), 16 );
            a = 1;
        } else if ( color.substring( 0, 5 )=='rgba(' ) {
            var rgba = color.substring( 5, color.length-1 ).split( ',' );
            r = parseInt( rgba[0], 10 );
            g = parseInt( rgba[1], 10 );
            b = parseInt( rgba[2], 10 );
            a = parseFloat( rgba[3] );
        } else {
            return color;
        }
        return [ r, g, b, a ];
    }

    function putRGBA( c, f ) {
        if ( f===undefined ) f = 1;
        if ( c.length )
            return 'rgba(' + Math.min( Math.floor(c[0]*f), 255 ) + ',' + Math.min( Math.floor(c[1]*f), 255 ) + ',' + Math.min( Math.floor(c[2]*f), 255 ) + ',' + c[3] + ')';
        else
            return c;
    }

    function putRGBA2( c, f ) {
        if ( f===undefined ) f = 1;
        if ( c.length )
            return 'rgba(' + c[0] + ',' + c[1] + ',' + c[2] + ',' + c[3]*f + ')';
        else
            return c;
    }

    function putRGBA3( c1, c2, f ) {
        if ( c1.length && c2.length )
            return 'rgba(' + Math.floor(c1[0]*(1-f)+c2[0]*f) + ',' + Math.floor(c1[1]*(1-f)+c2[1]*f) + ',' + Math.floor(c1[2]*(1-f)+c2[2]*f) + ',' + Math.floor(c1[3]*(1-f)+c2[3]*f) + ')';
        else
            return c1;
    }

    function addLumaGradientStops( gr, color, lumaGradient ) {
        var c = getRGBA( color );
        gr.addColorStop( 0, putRGBA( c, lumaGradient >= 0 ? 1 : -lumaGradient ) );
        gr.addColorStop( 1, putRGBA( c, lumaGradient >= 0 ? lumaGradient : 1 ) );
    }

    function addOpacityGradientStops( gr, color, opacityGradient, base ) {
        var c = getRGBA( color );
        gr.addColorStop( 0, putRGBA2( c, opacityGradient >= 0 ? 1 : -opacityGradient ) );
        gr.addColorStop( base, putRGBA2( c, opacityGradient >= 0 ? opacityGradient : 1 ) );
        gr.addColorStop( 1, putRGBA2( c, opacityGradient >= 0 ? 1 : -opacityGradient ) );
    }

    function getColor( colors, i, data ) {
        if ( !colors.length )
            return jSignage.Graph.grey;
        if ( data!==undefined && typeof(colors[0])=='object' ) {
            for ( var i=0; i < colors.length; i++ )
                if ( data>=colors[i].from && data < colors[i].to )
                    return colors[i].color;
            return jSignage.Graph.grey;
        }
        return colors[i%colors.length];
    }

    function autoSetColors( colors ) {
        if ( !colors.length || typeof(colors[0])!='object' )
            return colors;
        var cs = [], last = Number.NEGATIVE_INFINITY;
        for ( var i=0; i < colors.length; i++ ) {
            var x = colors[i];
            var c = {
                from: 'from' in x ? x.from : last,
                to: 'to' in x ? x.to : i+1 < colors.length && 'from' in colors[i+1] ? colors[i+1].from : Number.POSITIVE_INFINITY,
                color: x.color || jSignage.Graph.red
            }
            last = c.to;
            cs.push( c );
        }
        return cs;
    }

    function autoSetThresholds( thresholds, defaultSize, min, max, offsetRef ) {
        var r = { };
        r.size = jSignage.relAbs( thresholds.size, offsetRef, defaultSize );
        r.offset = jSignage.relAbs( thresholds.offset, offsetRef, 0 );
        r.maxSize = r.size;
        r.step = thresholds.step || 0;
        r.stepSize = thresholds.stepSize || .85;
        var startSize = jSignage.relAbs( thresholds.startSize, r.size, r.size );
        var endSize = jSignage.relAbs( thresholds.endSize, r.size, r.size );
        r.fill = thresholds.fill || 'solid';
        r.zones = [];
        var last = min;
        for ( var i=0; i < thresholds.zones.length; i++ ) {
            var z = { }, x = thresholds.zones[i] || { };
            z.from = 'from' in x ? x.from : last;
            z.to = 'to' in x ? x.to : i+1 < thresholds.zones.length && thresholds.zones[i+1] && 'from' in thresholds.zones[i+1] ? thresholds.zones[i+1].from : max;
            last = z.to;
            z.color = x.color || jSignage.Graph.red;
            z.startSize = 'startSize' in x ? x.startSize : startSize + (z.from-min)/(max-min)*(endSize-startSize);
            z.endSize = 'endSize' in x ? x.endSize : startSize + (z.to-min)/(max-min)*(endSize-startSize);
            r.maxSize = Math.max( r.maxSize, z.startSize, z.endSize );
            if ( r.fill=='gradient' ) {
                z.startColor = x.startColor || z.color;
                z.endColor = x.endColor || i+1 < thresholds.zones.length && thresholds.zones[i+1] && thresholds.zones[i+1].color || jSignage.Graph.red;
            }
            r.zones.push( z );
        }
        r.maxSize += r.offset;
        return r;
    }

    function autoSetAxis( axis, sizeRef, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, withAxisByDefault, withGridlinesByDefault ) {
        var r = { };
        r.direction = axis.direction || 1;
        r.title = 'title' in axis ? axis.title : '';
        r.titleTextStyle = axis.titleTextStyle ? new TextStyle( axis.titleTextStyle, defaultTextStyle ) : defaultTextStyle;
        r.textPlacement = axis.textPlacement || 'outside';
        r.textStyle = axis.textStyle ? new TextStyle( axis.textStyle, defaultTextStyle ) : defaultTextStyle;
        var axisStrokeStyle = axis.axis && typeof(axis.axis)=='object' && axis.axis.strokeStyle ? new StrokeStyle( axis.axis.strokeStyle, axisStrokeStyle ) : axisStrokeStyle;
        r.axisStrokeStyle = ( 'axis' in axis ? !axis.axis : !withAxisByDefault ) ? null : axisStrokeStyle;
        r.tickMarks = ( !r.axisStrokeStyle && !axis.tickMarks ) || 'tickMarks' in axis && !axis.tickMarks ? null : {
            placement: typeof(axis.tickMarks)=='object' && axis.tickMarks.placement || 'outside',
            length: jSignage.relAbs( typeof(axis.tickMarks)=='object' && axis.tickMarks.length, sizeRef, r.textStyle.fontSize * 0.5 ),
            strokeStyle: typeof(axis.tickMarks)=='object' && axis.tickMarks.strokeStyle ? new StrokeStyle( axis.tickMarks.strokeStyle, axisStrokeStyle ) : axisStrokeStyle
        };
        r.minorTickMarks = !r.tickMarks || !axis.minorTickMarks ? null : {
            count: typeof(axis.minorTickMarks)=='object' && axis.minorTickMarks.count || 4,
            length: jSignage.relAbs( typeof(axis.minorTickMarks)=='object' && axis.minorTickMarks.length, r.tickMarks.length, r.tickMarks.length / 2 ),
            strokeStyle: typeof(axis.minorTickMarks)=='object' && axis.minorTickMarks.strokeStyle ? new StrokeStyle( axis.minorTickMarks.strokeStyle, r.tickMarks.strokeStyle ) : r.tickMarks.strokeStyle
        };
        r.gridlinesStrokeStyle = ( 'gridlines' in axis ? !axis.gridlines : !withGridlinesByDefault ) ? null : axis.gridlines && typeof(axis.gridlines)=='object' && axis.gridlines.strokeStyle ? new StrokeStyle( axis.gridlines.strokeStyle, gridlinesStrokeStyle ) : gridlinesStrokeStyle;
        if ( !r.gridlinesStrokeStyle || !axis.minorGridlines ) {
            r.minorGridlines = null;
        } else {
            var minorGridlinesStrokeStyle = new StrokeStyle( r.gridlinesStrokeStyle );
            minorGridlinesStrokeStyle.lineWidth /= 2;
            r.minorGridlines = {
                count: typeof(axis.minorGridlines)=='object' && axis.minorGridlines.count || 4,
                strokeStyle: typeof(axis.minorGridlines)=='object' && axis.minorGridlines.strokeStyle ? new StrokeStyle( axis.minorGridlines.strokeStyle, minorGridlinesStrokeStyle ) : minorGridlinesStrokeStyle
            };
        }
        r.offset = jSignage.relAbs( axis.offset, sizeRef, 0 );
        return r;
    }

    function getFormatter( pattern, locale ) {
        var formatter;
        if ( jSignage.isFunction( pattern ) ) {
            formatter = pattern;
        } else {
            if ( pattern=='' )
                pattern = jSignage.NumberFormat.Format.DECIMAL;
            else if ( pattern=='E' )
                pattern = jSignage.NumberFormat.Format.SCIENTIFIC;
            else if ( pattern=='%' )
                pattern = jSignage.NumberFormat.Format.PERCENT;
            else if ( pattern=='$' )
                pattern = jSignage.NumberFormat.Format.CURRENCY;
            else if ( pattern in jSignage.NumberFormat.Format )
                pattern = jSignage.NumberFormat.Format[pattern];
            var nf = new jSignage.NumberFormat( pattern, locale );
            formatter = function( y ) { return nf.format( y ); }
        }
        return formatter;
    }

    function autoSetContinuousAxis( axis, min, max, locale, sizeRef, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, baselineStrokeStyle, withAxisByDefault, withGridlinesByDefault, withBaselineByDefault, isPercentage, isGauge ) {
        var r = autoSetAxis( axis, sizeRef, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, withAxisByDefault, withGridlinesByDefault ), min, max;

        if ( !isGauge ) {
            // Prefer sticking the baseline at zero unless we use only one third of the chart area
            if ( min > 0 && max > 1.5*min )
                min = 0;
            else if ( max < 0 && min < 1.5*max )
                max = 0;
        }

        var autoMin = min, autoMax = max, autoTick = 0;
        if ( !axis.tick && ( !axis.tickInterval || axis.tickInterval==='small' || axis.tickInterval==='medium' || axis.tickInterval==='large' ) ) {
            var minTicks, maxTicks;
            if ( axis.tickInterval==='small' ) {
                minTicks = 8;
                maxTicks = 12;
            } else if ( axis.tickInterval==='large' ) {
                minTicks = 3;
                maxTicks = 4;
            } else {
                minTicks = 4;
                maxTicks = 6;
            }
            var bestFit = null;
            for ( var k = ( max - min ) / minTicks; k > 0; k = k/1.25 ) {
                var myTick = findTickLimit( k );
                var iMin = Math.floor( min / myTick ), iMax = Math.ceil( max / myTick );
                if ( iMax - iMin > maxTicks )
                    break;
                var myFit = ( iMax - iMin ) * myTick / ( max - min );
                if ( bestFit===null || myFit <= bestFit ) {
                    bestFit = myFit;
                    autoMin = iMin * myTick;
                    autoMax = iMax * myTick;
                    autoTick = myTick;
                }
            }
        }

        r.minValue = isGauge ? min : 'minValue' in axis ? axis.minValue : autoMin;
        r.maxValue = isGauge ? max : 'maxValue' in axis ? axis.maxValue : autoMax;

        if ( isGauge )
            r.baseline = r.minValue;
        else if ( 'baseline' in axis )
            r.baseline = axis.baseline;
        else if ( r.minValue > 0 )
            r.baseline = r.minValue;
        else if ( r.maxValue < 0 )
            r.baseline = r.maxValue;
        else
            r.baseline = 0;

        r.baselineStrokeStyle = ( 'baselineStrokeStyle' in axis ? !axis.baselineStrokeStyle : !withBaselineByDefault ) ? null : axis.baselineStrokeStyle ? new StrokeStyle( axis.baselineStrokeStyle, baselineStrokeStyle ) : baselineStrokeStyle;

        var formatter = getFormatter( 'format' in axis ? axis.format : isPercentage ? '%' : '', locale );
        var ticks = axis.ticks;
        if ( !ticks ) {
            if ( axis.tickInterval==='max' ) {
                if ( r.baseline==r.minValue || r.baseline==r.maxValue ) {
                    ticks = [ r.minValue, r.maxValue ];
                    r.tickInterval = r.maxValue - r.minValue;
                } else {
                    ticks = [ r.minValue, r.baseline, r.maxValue ];
                    if ( baseline-r.minValue==r.maxValue-r.baseline )
                        r.tickInterval = r.maxValue - r.baseline;
                }
            } else {
                var tickInterval = autoTick || axis.tickInterval || ( r.maxValue - r.minValue ) / 5;
                ticks = [];
                var minTick = Math.ceil( (r.minValue-r.baseline) / tickInterval ), maxTick = Math.floor( (r.maxValue-r.baseline) / tickInterval );
                for ( var i = minTick; i <= maxTick; i++ )
                    ticks.push( r.baseline + i * tickInterval );
                r.tickInterval = tickInterval;
            }
        }
        r.ticks = [];
        for ( var i=0; i < ticks.length; i++ ) {
            var v = typeof(ticks[i])=='object' ? ticks[i].v : ticks[i];
            r.ticks.push({
                v: v,
                f: formatter( v )
            });
        }

        r.referenceLevels = [];
        if ( axis.referenceLevels ) {
            var levelStrokeStyle = new StrokeStyle( baselineStrokeStyle );
            levelStrokeStyle.lineDash = [ r.textStyle.fontSize * 0.5 ];
            for ( var i = 0; i < axis.referenceLevels.length; i++ ) {
                var v = typeof(axis.referenceLevels[i])=='object' ? axis.referenceLevels[i].v : axis.referenceLevels[i];
                r.referenceLevels.push( {
                    v: v,
                    f: typeof(axis.referenceLevels[i])=='object' && 'f' in axis.referenceLevels[i] ? axis.referenceLevels[i].f : formatter( v ),
                    strokeStyle: typeof(axis.referenceLevels[i])=='object' && axis.referenceLevels[i].strokeStyle ? new StrokeStyle( axis.referenceLevels[i].strokeStyle, levelStrokeStyle ) : levelStrokeStyle,
                    fillColor: typeof(axis.referenceLevels[i])=='object' && axis.referenceLevels[i].fill || 'transparent',
                    linearGradient: typeof(axis.referenceLevels[i])=='object' && axis.referenceLevels[i].linearGradient || false
                });
            }
        }

        return r;
    }

    function autoSetDiscreteAxis( axis, numCategories, sizeRef, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, baselineStrokeStyle, withAxisByDefault, withGridlinesByDefault ) {
        var r = autoSetAxis( axis, sizeRef, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, withAxisByDefault, withGridlinesByDefault );
        r.categories = [];
        for ( var i=0; i < numCategories; i++ )
            r.categories.push( axis.categories && i < axis.categories.length ? axis.categories[i] : i+1 );
        return r;
    }

    function makePercentageData( data ) {
        var r = [];
        for ( var i=0; i < data.length; i++ ) {
            var sum = 0;
            for ( var j=0; j < data[i].length; j++ )
                sum += data[i][j];
            if ( sum > 0 ) {
                var row = [];
                for ( var j=0; j < data[i].length; j++ )
                    row.push( data[i][j] / sum );
                r.push( row );
            } else {
                r.push( data[i] );
            }
        }
        return r;
    }

    function makePercentageData2( data ) {
        var sum = [], r = [];
        for ( var i=0; i < data.length; i++ )
            for ( var j=0; j < data[i].length; j++ )
                sum[j] = ( sum[j] || 0 ) + ( data[i][j] || 0 );
        for ( var i=0; i < data.length; i++ ) {
            var row = []
            for ( var j=0; j < data[i].length; j++ )
                row.push( sum[j] && ( data[i][j] || 0 ) / sum[j] );
            r.push( row );
        }
        return r;
    }

    function drawMarker( ctx, type, x, y, size ) {
        ctx.beginPath();
        if ( type=='circle' ) {
            ctx.arc( x, y, size/2, 0, 2*Math.PI );
        } else if ( type=='square' ) {
            size /= 1.5 * Math.SQRT2;
            ctx.moveTo( x-size, y-size );
            ctx.lineTo( x+size, y-size );
            ctx.lineTo( x+size, y+size );
            ctx.lineTo( x-size, y+size );
            ctx.closePath();
        } else if ( type=='diamond' ) {
            size /= 1.5;
            ctx.moveTo( x, y-size );
            ctx.lineTo( x+size, y );
            ctx.lineTo( x, y+size );
            ctx.lineTo( x-size, y );
            ctx.closePath();
        } else if ( type=='triangle' ) {
            size /= 1.5;
            ctx.moveTo( x, y-size );
            ctx.lineTo( x+0.866*size, y+size/2 );
            ctx.lineTo( x-0.866*size, y+size/2 );
            ctx.closePath();
        } else if ( type=='star' ) {
            size /= 1.5;
            ctx.moveTo( x, y-size );
            ctx.lineTo( x+.2566*size, y-.2389*size );
            ctx.lineTo( x+1.053*size, y-.2389*size );
            ctx.lineTo( x+.4159*size, y+.2389*size );
            ctx.lineTo( x+.6460*size, y+size );
            ctx.lineTo( x, y+.5820*size );
            ctx.lineTo( x-.6460*size, y+size );
            ctx.lineTo( x-.4159*size, y+.2389*size );
            ctx.lineTo( x-1.053*size, y-.2389*size );
            ctx.lineTo( x-.2566*size, y-.2389*size );
            ctx.closePath();
        }
    }

    var richTextToStyle = { fill: 'color', fontFamily: 'fontFamily', fontStyle: 'fontStyle', fontWeight: 'fontWeight', fontVariant: 'fontVariant', fontSize: 'fontSize' };

 jSignage.Graph = {
    baseColors: ['#3366cc', '#dc3912', '#ff9900', '#109618', '#990099', '#0099c6', '#dd4477', '#66aa00', '#b82e2e'],
    baseMarkers: [ 'circle', 'diamond', 'square', 'triangle', 'star' ],
    green : "#109618",
    orange : "#ff9900",
    yellow: "#dc9610",
    red : "#dc3912",
    blue: "#4684ee",
    white: "#f7f7f7",
    grey: "#cccccc",
    gray: "#cccccc",
    black: "#000000",

    pieChart: function ( args ) {
        var layerWidth, layerHeight, layerMin, layerG2, canvas;

        var layer = jSignage.customLayer( 'pieChart', args, null, function ( width, height, x, y, bbw, bbh, parent ) {
            layerWidth = width;
            layerHeight = height;
            layerMin = Math.min( width, height );
            layerG2 = this;
            canvas = jSignage.appendCanvasElement( this, 0, 0, width, height );
        } );

        layer.endEvent( function () {
            layerG2.textContent = '';
        } );

        layer.beginEvent( function () {
            var ctx = jSignage.getContext2D( canvas );
            var startAngle = ( args.startAngle || 0 ) * Math.PI / 180;
            var endAngle = ( args.endAngle || args.startAngle || 0 ) * Math.PI / 180;
            var colors = args.colors || jSignage.Graph.baseColors;
            var reverseCategories = args.reverseCategories || false;
            var data = args.data || [];
            var borderSize = data.length==1 ? 0 : 'borderSize' in args ? args.borderSize : layerMin * .01;
            var defaultTextStyle = new TextStyle( args, layerMin * .05 );
            var pieSliceStrokeStyle = args.strokeStyle ? new StrokeStyle( args.strokeStyle, null ) : null;
            var textInside = args.textInside || 'none'; // "percentage"|"label"|"value"|"none"
            var textOutside = args.textOutside || 'none'; // "percentage"|"label"|"value"|"none"
            var pieHole = Math.min( args.pieHole || 0, 1 );
            var textInsideStyle = args.textInsideStyle ? new TextStyle( args.textInsideStyle, defaultTextStyle ) : defaultTextStyle;
            var textOutsideStyle = args.textOutsideStyle ? new TextStyle( args.textOutsideStyle, defaultTextStyle ) : defaultTextStyle;
            var sliceGradient = args.sliceGradient || 1;
            var legend = false;
            if ( 'legend' in args ) {
                if ( typeof(args.legend)=='object' && args.legend!==null ) {
                    legend = {
                        position: args.legend.position || 'right',
                        alignment: args.legend.alignment,
                        orientation: args.legend.orientation,
                        textStyle: args.legend.textStyle ? new TextStyle( args.legend.textStyle, defaultTextStyle ) : defaultTextStyle,
                        strokeStyle: args.legend.strokeStyle ? new StrokeStyle( args.legend.strokeStyle, pieSliceStrokeStyle ) : pieSliceStrokeStyle,
                        frameStyle: args.legend.frameStyle ? new StrokeStyle( args.legend.frameStyle, null ) : null
                    }
                }
            } else {
                legend = { position: 'right', textStyle: defaultTextStyle, strokeStyle: pieSliceStrokeStyle, frameStyle: null };
            }
            var pieResidueSliceColor = args.pieResidueSliceColor || jSignage.Graph.grey;
            var pieResidueSliceLabel = args.pieResidueSliceLabel || 'Other';
            var slices = args.slices || null, labels = args.labels || [], sum = 0;
            for ( var i = 0; i < data.length; i++ )
                sum += data[i];
            var sliceVisibilityThreshold = ( 'sliceVisibilityThreshold' in args ? args.sliceVisibilityThreshold : 0.02 ) * sum;
            var r = applyVisibilityThreshold( data, sliceVisibilityThreshold, colors, pieResidueSliceColor, labels, pieResidueSliceLabel );
            data = r.data; colors = r.colors; labels = r.labels;
            var chartArea = addLegend( layerWidth, layerHeight, labels, legend, ctx, colors );
            if ( args.chartArea ) {
                chartArea.x = jSignage.relAbs( args.chartArea.left, layerWidth, chartArea.x );
                chartArea.y = jSignage.relAbs( args.chartArea.top, layerHeight, chartArea.y );
                chartArea.width = jSignage.relAbs( args.chartArea.width, layerWidth, chartArea.width );
                chartArea.height = jSignage.relAbs( args.chartArea.height, layerHeight, chartArea.height );
            }
            var cx = chartArea.width/2 + chartArea.x;
            var cy = chartArea.height/2 + chartArea.y;
            var maxOffset = 0, maxLineWidth = pieSliceStrokeStyle ? pieSliceStrokeStyle.lineWidth : 0, maxFontSize = textOutsideStyle.fontSize;
            if ( slices ) for ( var i=0; i < data.length; i++ ) if ( slices[i] ) {
                if ( 'offset' in slices[i] && slices[i].offset > 0 )
                    maxOffset = Math.max( maxOffset, slices[i].offset );
                if ( slices[i].strokeStyle )
                    maxLineWidth = Math.max( maxLineWidth, slices[i].strokeStyle.lineWidth );
                if ( slices[i].textOutsideStyle && slices[i].textOutsideStyle.fontSize )
                    maxFontSize = Math.max( maxFontSize, slices[i].textOutsideStyle.fontSize );
            }
            var radius = Math.min( chartArea.width, chartArea.height ) / 2, valueToRad;
            radius = Math.max( 0, radius - maxLineWidth / 2 );
            if ( textOutside!='none' )
                radius = Math.max( 0, radius - maxFontSize ) / 1.05;
            radius = radius / ( 1 + maxOffset );
            startAngle = startAngle - Math.floor( startAngle/2/Math.PI ) * 2 * Math.PI;
            endAngle = endAngle - Math.floor( endAngle/2/Math.PI ) * 2 * Math.PI;
            if ( endAngle==startAngle ) {
                valueToRad = 2 * Math.PI / sum;
            } else if ( reverseCategories ) {
                if ( startAngle < endAngle )
                    startAngle += 2 * Math.PI;
                valueToRad = ( startAngle - endAngle ) / sum;
            } else {
                if ( endAngle < startAngle )
                    endAngle += 2 * Math.PI;
                valueToRad = ( endAngle - startAngle ) / sum;
            }
            var sign = reverseCategories ? -1 : 1;
            var deltaOuter = borderSize > 0 ? Math.asin( borderSize / (2*radius) ) : 0;
            var deltaInner = pieHole > 0 && borderSize > 0 ? Math.asin( borderSize / (2*radius*pieHole) ) : 0;
            var start = startAngle - Math.PI/2;
            var textInsideRadius = radius * 0.9, textOutsideRadius = radius * 1.05;

            for ( var i=0; i < data.length; i++ ) {
                var offsetX = 0, offsetY = 0, fill = getColor( colors, i ), strokeStyle = pieSliceStrokeStyle;
                var value = data[i], angle = valueToRad * value;
                if ( angle < 2*deltaOuter )
                    continue;
                var midAngle = start + sign*angle/2;
                if ( slices && slices[i] ) {
                    var offset = Math.min( slices[i].offset || 0, 1 );
                    if ( offset > 0 ) {
                        offsetX = Math.cos( midAngle ) * offset * radius;
                        offsetY = Math.sin( midAngle ) * offset * radius;
                    }
                    if ( slices[i].color )
                        fill = slices[i].color;
                    if ( slices[i].strokeStyle )
                        strokeStyle = slices[i].strokeStyle;
                }
                ctx.beginPath();
                if ( pieHole > 0 && angle > 2*deltaInner ) {
                    ctx.arc( cx+offsetX, cy+offsetY, radius*pieHole, start+deltaInner*sign, start+(angle-deltaInner)*sign, reverseCategories );
                    ctx.arc( cx+offsetX, cy+offsetY, radius, start+(angle-deltaOuter)*sign, start+deltaOuter*sign, !reverseCategories );
                } else {
                    if ( borderSize > 0 ) {
                        var d = Math.min( borderSize / 2 / Math.sin( angle/2 ), radius/20 );
                        ctx.moveTo( cx+offsetX+Math.cos( midAngle )*d, cy+offsetY+Math.sin(midAngle)*d );
                    } else {
                        ctx.moveTo( cx+offsetX, cy+offsetY );
                    }
                    ctx.arc( cx+offsetX, cy+offsetY, radius, start+deltaOuter*sign, start+(angle-deltaOuter)*sign, reverseCategories );
                }
                ctx.closePath();
                ctx.fillStyle = fill;
                if ( sliceGradient!=1 ) {
                    var gr = ctx.createRadialGradient( cx+offsetX, cy+offsetY, radius*pieHole, cx+offsetX, cy+offsetY, radius );
                    addLumaGradientStops( gr, ctx.fillStyle, sliceGradient );
                    ctx.fillStyle = gr;
                }
                ctx.fill();
                if ( strokeStyle ) {
                    setStrokeStyle( ctx, strokeStyle );
                    ctx.stroke();
                }
                if ( textInside!='none' ) {
                    var text, style = slices && slices[i] && slices[i].textInsideStyle ? new TextStyle( slices[i].textInsideStyle, textInsideStyle ) : textInsideStyle;
                    if ( jSignage.isArray( textInside ) )
                        text = i in textInside ? textInside[i] : null;
                    else
                        text = textInside=='value' ? value : textInside=='percentage' ? (value/sum*100).toFixed(1) + '%' : labels[i];
                    if ( text!==null ) {
                        ctx.fillStyle = style.color;
                        ctx.textAlign = 'center';
                        ctx.textBaseline = 'middle';
                        ctx.font = style.font();
                        var metrics = jSignage.measureText( ctx, text ), bx, by;
                        midAngle = midAngle - Math.floor( midAngle/2/Math.PI ) * 2 * Math.PI;
                        if ( midAngle < Math.PI/2 ) {
                            bx = metrics.actualBoundingBoxRight;
                            by = metrics.actualBoundingBoxDescent;
                        } else if ( midAngle < Math.PI ) {
                            bx = -metrics.actualBoundingBoxLeft;
                            by = metrics.actualBoundingBoxDescent;
                        } else if ( midAngle < 3*Math.PI/2 ) {
                            bx = -metrics.actualBoundingBoxLeft;
                            by = -metrics.actualBoundingBoxAscent;
                        } else {
                            bx = metrics.actualBoundingBoxRight;
                            by = -metrics.actualBoundingBoxAscent;
                        }
                        if ( textInsideRadius*textInsideRadius > bx*bx+by*by ) {
                            var b = 2 * Math.cos( midAngle ) * bx + 2 * Math.sin( midAngle ) * by;
                            var c = bx*bx + by*by - textInsideRadius*textInsideRadius;
                            var tr = ( -b + Math.sqrt( b*b - 4*c ) ) / 2;
                            var x = cx+offsetX+Math.cos(midAngle)*tr, y = cy+offsetY+Math.sin(midAngle)*tr;
                            if ( isTextInPath( ctx, x, y, metrics, 2 ) )
                                ctx.fillText( text, x, y );
                        }
                    }
                }
                if ( textOutside!='none' ) {
                    var text, style = slices && slices[i] && slices[i].textOutsideStyle ? new TextStyle( slices[i].textOutsideStyle, textOutsideStyle ) : textOutsideStyle;
                    if ( jSignage.isArray( textOutside ) )
                        text = i in textOutside ? textOutside[i] : null;
                    else
                        text = textOutside=='value' ? value : textOutside=='percentage' ? (value/sum*100).toFixed(1) + '%' : labels[i];
                    if ( text!==null ) {
                        ctx.fillStyle = style.color;
                        ctx.font = style.font();
                        midAngle = midAngle - Math.floor( midAngle/2/Math.PI ) * 2 * Math.PI;
                        if ( midAngle < Math.PI/2 ) {
                            ctx.textAlign = 'left';
                            ctx.textBaseline = 'top';
                        } else if ( midAngle < Math.PI ) {
                            ctx.textAlign = 'right';
                            ctx.textBaseline = 'top';
                        } else if ( midAngle < 3*Math.PI/2 ) {
                            ctx.textAlign = 'right';
                            ctx.textBaseline = 'alphabetic';
                        } else {
                            ctx.textAlign = 'left';
                            ctx.textBaseline = 'alphabetic';
                        }
                        var x = cx+offsetX+Math.cos(midAngle)*textOutsideRadius, y = cy+offsetY+Math.sin(midAngle)*textOutsideRadius;
                        ctx.fillText( text, x, y );
                    }
                }
                start += angle*sign;
            }
            jSignage.flushContext2D( ctx );
        });

        return layer;
    },

    columnChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'columnChart', args, 'column', false, false, 'color' in args ? jSignage.isArray(args.color) ? args.color : [ args.color ] : [ jSignage.Graph.baseColors[0] ], 'label' in args ? [ args.label ] : null, args.barWidth );
    },

    groupedColumnChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'groupedColumnChart', args, 'column', true, false, args.colors, args.labels, args.groupWidth, args.barWidth );
    },

    stackedColumnChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'stackedColumnChart', args, 'column', false, true, args.colors, args.labels, args.barWidth );
    },

    stackedAndGroupedColumnChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'stackedAndGroupedColumnChart', args, 'column', true, true, args.colors, args.labels, args.groupWidth, args.barWidth );
    },

    percentageColumnChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'percentageColumnChart', args, 'column', false, true, args.colors, args.labels, args.barWidth );
    },

    barChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'barChart', args, 'bar', false, false, 'color' in args ? jSignage.isArray(args.color) ? args.color : [ args.color ] : [ jSignage.Graph.baseColors[0] ], 'label' in args ? [ args.label ] : null, args.barWidth );
    },

    groupedBarChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'groupedBarChart', args, 'bar', true, false, args.colors, args.labels, args.groupWidth, args.barWidth );
    },

    stackedBarChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'stackedBarChart', args, 'bar', false, true, args.colors, args.labels, args.barWidth );
    },

    stackedAndGroupedBarChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'stackedAndGroupedBarChart', args, 'bar', true, true, args.colors, args.labels, args.groupWidth, args.barWidth );
    },

    percentageBarChart: function( args ) {
        return jSignage.Graph._columnOrBarChart( 'percentageBarChart', args, 'bar', false, true, args.colors, args.labels, args.barWidth );
    },

    _columnOrBarChart: function ( name, args, type, isGrouped, isStacked, colors, labels, groupWidth, barWidth ) { 
        var layerWidth, layerHeight, layerMin, layerG2, canvas;
        var isPercentage = name.substring( 0, 10 )=='percentage';

        var layer = jSignage.customLayer( name, args, null, function ( width, height, x, y, bbw, bbh, parent ) {
            layerWidth = width;
            layerHeight = height;
            layerMin = Math.min( width, height );
            layerG2 = this;
            canvas = jSignage.appendCanvasElement( this, 0, 0, width, height );
        } );

        layer.endEvent( function () {
            layerG2.textContent = '';
        } );

        layer.beginEvent( function () {
            var ctx = jSignage.getContext2D( canvas );
            var data = isPercentage ? makePercentageData( args.data ) : args.data;
            var groupSize = isGrouped && data.length && data[0].length || 1;
            var numLabels = groupSize;
            if ( isStacked ) {
                if ( isGrouped ) {
                    numLabels = 0;
                    if ( data.length && data[0].length )
                        for ( var i=0; i < data[0].length; i++ )
                            numLabels += data[0][i].length ? data[0][i].length : 1;
                } else {
                    numLabels = data.length && data[0].length || 1;
                }
            }
            colors = colors || jSignage.Graph.baseColors;
            var groupBorderSize = 'borderSize' in args ? args.borderSize : isGrouped && !barWidth ? Math.max( layerMin * .003, 1 ) : 0;
            var stackBorderSize = 'borderSize' in args ? args.borderSize : Math.max( layerMin * .003, 1 );
            var defaultTextStyle = new TextStyle( args, layerMin * .04 );
            var textInside = args.textInside || 'none';
            var textInsideStyle = args.textInsideStyle ? new TextStyle( args.textInsideStyle, defaultTextStyle ) : defaultTextStyle;
            var textOutside = args.textOutside || 'none';
            var textOutsideStyle = args.textOutsideStyle ? new TextStyle( args.textOutsideStyle, defaultTextStyle ) : defaultTextStyle;
            var gridlinesStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMin * .0015, 1 ), color: '#888' }, null );
            if ( args.gridlinesStrokeStyle )
                gridlinesStrokeStyle = new StrokeStyle( args.gridlinesStrokeStyle, gridlinesStrokeStyle );
            var axisStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMin * .0015, 1 ), color: '#88f' }, null );
            if ( args.axisStrokeStyle )
                axisStrokeStyle = new StrokeStyle( args.axisStrokeStyle, axisStrokeStyle );
            var baselineStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMin * .0015, 1 ), color: '#000' }, null );
            if ( args.baselineStrokeStyle )
                baselineStrokeStyle = new StrokeStyle( args.baselineStrokeStyle, baselineStrokeStyle );
            var strokeStyle = args.strokeStyle ? new StrokeStyle( args.strokeStyle, null ) : null;
            var chartGradient = args.chartGradient || 1;
            var barGradient = args.barGradient || 1;
            var legend = false, defaultLegendPosition = type=='bar' ? 'bottom' : 'right';
            if ( 'legend' in args ) {
                if ( typeof(args.legend)=='object' && args.legend!==null ) {
                    legend = {
                        position: args.legend.position || defaultLegendPosition,
                        alignment: args.legend.alignment,
                        textStyle: args.legend.textStyle ? new TextStyle( args.legend.textStyle, defaultTextStyle ) : defaultTextStyle,
                        strokeStyle: args.legend.strokeStyle ? new StrokeStyle( args.legend.strokeStyle, strokeStyle ) : strokeStyle,
                        frameStyle: args.legend.frameStyle ? new StrokeStyle( args.legend.frameStyle, null ) : null
                    }
                }
            } else {
                legend = { position: defaultLegendPosition, textStyle: defaultTextStyle, strokeStyle: strokeStyle, frameStyle: null };
            }

            var minorAxis = ( type=='bar' ? args.xAxis : args.yAxis ) || { }, min, max;
            if ( !('minValue' in minorAxis) || !('maxValue' in minorAxis) ) {
                if ( isPercentage ) {
                    min = 0;
                    max = 1;
                } else {
                    var minmax = findMinMax( data, isGrouped, isStacked, groupSize );
                    min = minmax[0];
                    max = minmax[1];
                }
            }
            minorAxis = autoSetContinuousAxis( minorAxis, min, max, args.locale || null, layerMin, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, baselineStrokeStyle, false, true, true, isPercentage );
            var majorAxis = autoSetDiscreteAxis( ( type=='bar' ? args.yAxis : args.xAxis ) || { }, data.length, layerMin, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, true, false );
            if ( type=='bar' )
                majorAxis.direction = -majorAxis.direction;
            var xAxis = type=='bar' ? minorAxis : majorAxis;
            var yAxis = type=='bar' ? majorAxis : minorAxis;
            if ( !labels || labels.length < numLabels ) {
                var l = [];
                for ( var i=0; i < numLabels; i++ )
                    l.push( labels && i < labels.length ? labels[i] : i+1 );
                labels = l;
            }

            var chartArea = addLegendAndAxes( ctx, layerWidth, layerHeight, legend, yAxis, xAxis, colors, labels );
            if ( args.chartArea ) {
                chartArea.x = jSignage.relAbs( args.chartArea.left, layerWidth, chartArea.x );
                chartArea.y = jSignage.relAbs( args.chartArea.top, layerHeight, chartArea.y );
                chartArea.width = jSignage.relAbs( args.chartArea.width, layerWidth, chartArea.width );
                chartArea.height = jSignage.relAbs( args.chartArea.height, layerHeight, chartArea.height );
            }
            if ( !data.length ) return;
            var majorPixels = type=='bar' ? chartArea.height : chartArea.width;
            var minorPixels = type=='bar' ? chartArea.width : chartArea.height;
            var groupSpacing = majorPixels / data.length;
            var groupPixels = jSignage.relAbs( groupWidth, groupSpacing, groupSpacing*0.618 );
            var groupOffset = ( groupSpacing - groupPixels ) / 2;
            var barSpacing = groupPixels / groupSize;
            var barPixels = jSignage.relAbs( barWidth, barSpacing, barSpacing );
            var barOffset = ( barSpacing - barPixels ) / 2;
            var valueToPixels = minorPixels / ( minorAxis.maxValue-minorAxis.minValue );
            var baselinePixels = ( minorAxis.baseline - minorAxis.minValue ) * valueToPixels;
            var barNum = 0, stackNum = 0;
            if ( strokeStyle )
                setStrokeStyle( ctx, strokeStyle );
            for ( var i=0; i < data.length; i++ ) {
                var group = data[i], idx = 0;
                var groupMajorPosition = i * groupSpacing + groupOffset;
                for ( var j=0; j < groupSize; j++ ) {
                    var stack = isGrouped ? group[j] : group;
                    var barMajorPosition = groupMajorPosition + j * barSpacing + barOffset + ( j > 0 ? groupBorderSize/2 : 0 );
                    var barMajorPixels = barPixels - groupBorderSize + ( j==0 ? groupBorderSize/2 : 0 ) + ( j==groupSize-1 ? groupBorderSize/2 : 0 );
                    var barStart = baselinePixels, value = 0, borderStackStart = 0, borderStackEnd = 0;
                    var stackSize = isStacked && stack.length ? stack.length : 1;
                    var barPos = ( i * groupSize + j ) / ( data.length * groupSize - 1 );
                    var barLuma = chartGradient > 0 ? chartGradient * barPos + 1 - barPos : -chartGradient * ( 1 - barPos ) + barPos;
                    for ( var k=0; k < stackSize; k++ ) {
                        if ( isStacked && stack.length ) {
                            value += stack[k];
                            borderStackStart = k==0 ? 0 : stackBorderSize/2;
                            borderStackEnd = k==stackSize-1 ? 0 : stackBorderSize/2;
                        } else {
                            value = stack;
                        }
                        var barEnd = ( value - minorAxis.minValue ) * valueToPixels;
                        var x, y, w, h;
                        if ( type=='bar' ) {
                            x = chartArea.x + ( minorAxis.direction > 0 ? Math.min(barStart, barEnd)+borderStackStart : chartArea.width-Math.max(barStart, barEnd)+borderStackEnd );
                            y = chartArea.y + ( majorAxis.direction > 0 ? chartArea.height-barMajorPosition-barMajorPixels : barMajorPosition );
                            w = Math.max( Math.abs(barEnd-barStart)-borderStackEnd-borderStackStart, 0 );
                            h = barMajorPixels;
                        } else {
                            x = chartArea.x + ( majorAxis.direction > 0 ? barMajorPosition : chartArea.width-barMajorPosition-barMajorPixels );
                            y = chartArea.y + ( minorAxis.direction > 0 ? chartArea.height-Math.max(barStart, barEnd)+borderStackEnd : Math.min(barStart, barEnd)+borderStackStart );
                            w = barMajorPixels;
                            h = Math.max( Math.abs(barEnd-barStart)-borderStackEnd-borderStackStart, 0 );
                        }
                        ctx.beginPath();
                        ctx.rect( x, y, w, h );
                        ctx.fillStyle = isStacked || isGrouped ? getColor( colors, idx ) : getColor( colors, i );
                        if ( barLuma!=1 )
                            ctx.fillStyle = putRGBA( getRGBA( ctx.fillStyle ), barLuma );
                        if ( barGradient!=1 ) {
                            var gr;
                            if ( type=='bar' )
                                gr = ctx.createLinearGradient( x+w/2, majorAxis.direction > 0 ? y+h : y, x+w/2, majorAxis.direction > 0 ? y : y+h );
                            else
                                gr = ctx.createLinearGradient( majorAxis.direction > 0 ? x : x+w, y+h/2, majorAxis.direction > 0 ? x+w : x, y+h/2 );
                            addLumaGradientStops( gr, ctx.fillStyle, barGradient );
                            ctx.fillStyle = gr;
                        }
                        ctx.fill();
                        if ( strokeStyle )
                            ctx.stroke();
                        if ( textInside!=='none' ) {
                            var text, style = textInsideStyle;
                            if ( jSignage.isArray( textInside ) )
                                text = barNum in textInside ? textInside[barNum] : null;
                            else
                                text = textInside=='label' ? labels[idx] : textInside=='category' ? majorAxis.categories[i] : textInside=='percentage' ? ((isStacked ? stack[k] : stack)*100).toFixed(1) + '%' : isStacked ? stack[k] : stack;
                            if ( typeof(text)=='object' && text ) {
                                style = text.style ? new TextStyle( text.style, textInsideStyle ) : textInsideStyle;
                                text = text.text;
                            }
                            if ( text!==null ) {
                                ctx.font = style.font();
                                ctx.fillStyle = style.color;
                                ctx.textAlign = 'center';
                                ctx.textBaseline = 'middle';
                                var m = jSignage.measureText( ctx, text );
                                if ( m.actualBoundingBoxLeft <= w/2 && m.actualBoundingBoxRight <= w/2 && m.actualBoundingBoxAscent <= h/2 && m.actualBoundingBoxDescent <= h/2 )
                                    ctx.fillText( text, x+w/2, y+h/2 );
                            }
                        }
                        barStart = barEnd;
                        idx++;
                        barNum++;
                    }
                    if ( textOutside!=='none' ) {
                        var text, style = textOutsideStyle, sum;
                        if ( jSignage.isArray( textOutside ) ) {
                            text = stackNum in textOutside ? textOutside[stackNum] : null;
                        } else {
                            if ( textOutside=='label' ) {
                                text = labels[idx-1];
                            } else if ( textOutside=='category' ) {
                                text = majorAxis.categories[i];
                            } else {
                                if ( isStacked ) {
                                    sum = 0;
                                    for ( k=0; k < stack.length; k++ )
                                        sum += stack[k];
                                } else {
                                    sum = stack;
                                }
                                text = textOutside=='percentage' ? (sum*100).toFixed(1) + '%' : sum;
                            }
                        }
                        if ( typeof(text)=='object' && text ) {
                            style = text.style ? new TextStyle( text.style, textOutsideStyle ) : textOutsideStyle;
                            text = text.text;
                        }
                        if ( text!==null ) {
                            var x, y, w, h, tx, ty;
                            if ( type=='bar' ) {
                                x = chartArea.x + ( minorAxis.direction > 0 ? barEnd : 0 );
                                y = chartArea.y + ( majorAxis.direction > 0 ? chartArea.height-barMajorPosition-barMajorPixels : barMajorPosition );
                                w = chartArea.width - barEnd;
                                h = barMajorPixels;
                            } else {
                                x = chartArea.x + ( majorAxis.direction > 0 ? barMajorPosition : chartArea.width-barMajorPosition-barMajorPixels );
                                y = chartArea.y + ( minorAxis.direction > 0 ? 0 : barEnd );
                                w = barMajorPixels;
                                h = chartArea.height - barEnd;
                            }
                            ctx.font = style.font();
                            ctx.fillStyle = style.color;
                            if ( type=='bar' ) {
                                ctx.textAlign = minorAxis.direction > 0 ? 'left' : 'right';
                                ctx.textBaseline = 'middle';
                                tx = minorAxis.direction > 0 ? x + style.fontSize*0.2 : x + w - style.fontSize*0.2;
                                ty = y + h/2;
                            } else {
                                ctx.textAlign = 'center';
                                ctx.textBaseline = minorAxis.direction > 0 ? 'bottom' : 'top';
                                tx = x + w/2;
                                ty = minorAxis.direction > 0 ? y + h - style.fontSize*0.2 : y + style.fontSize*0.2;
                            }
                            ctx.fillText( text, tx, ty );
                        }
                    }
                    stackNum++;
                }
            }
            jSignage.flushContext2D( ctx );
        });

        return layer;
    },

    lineChart: function( args ) {
        var series = {};
        jSignage.copyProps( args, series, [ 'color', 'label', 'strokeStyle', 'marker', 'line' ] );
        return jSignage.Graph._areaOrlineChart( 'lineChart', args, 'line', false, [ series ], [ args.data ] );
    },

    multiLineChart: function( args ) {
        return jSignage.Graph._areaOrlineChart( 'multiLineChart', args, 'line', false, args.series, args.data );
    },

    areaChart: function( args ) {
        var series = {};
        jSignage.copyProps( args, series, [ 'color', 'label', 'strokeStyle', 'marker', 'line', 'fillColor', 'fillOpacity', 'fillOpacityGradient' ] );
        return jSignage.Graph._areaOrlineChart( 'areaChart', args, 'area', false, [ series ], [ args.data ] );
    },

    multiAreaChart: function( args ) {
        return jSignage.Graph._areaOrlineChart( 'multiAreaChart', args, 'area', false, args.series, args.data );
    },

    stackedAreaChart: function( args ) {
        return jSignage.Graph._areaOrlineChart( 'stackedAreaChart', args, 'area', true, args.series, args.data );
    },

    percentageAreaChart: function( args ) {
        return jSignage.Graph._areaOrlineChart( 'percentageAreaChart', args, 'area', true, args.series, args.data );
    },

    scatterChart: function( args ) {
        var series = {};
        jSignage.copyProps( args, series, [ 'color', 'label', 'strokeStyle', 'marker', 'line' ] );
        return jSignage.Graph._areaOrlineChart( 'scatterChart', args, 'line', false, [ series ], [ args.dataY ], [ args.dataX ] );
    },

    multiScatterChart: function( args ) {
        return jSignage.Graph._areaOrlineChart( 'multiScatterChart', args, 'line', false, args.series, args.dataY, args.dataX );
    },

    bubbleChart: function( args ) {
        var series = {};
        jSignage.copyProps( args, series, [ 'color', 'label', 'strokeStyle', 'marker', 'line', 'fillColor', 'fillOpacity', 'fillOpacityGradient' ] );
        return jSignage.Graph._areaOrlineChart( 'bubbleChart', args, 'line', false, [ series ], [ args.dataY ], [ args.dataX ], [ args.dataSize ] );
    },

    multiBubbleChart: function( args ) {
        return jSignage.Graph._areaOrlineChart( 'multiBubbleChart', args, 'line', false, args.series, args.dataY, args.dataX, args.dataSize );
    },

    _areaOrlineChart: function ( name, args, type, isStacked, argsSeries, data, dataX, dataSize ) {
        var layerWidth, layerHeight, layerMin, layerG2, canvas;
        var isPercentage = name.substring( 0, 10 )=='percentage';

        var layer = jSignage.customLayer( name, args, null, function ( width, height, x, y, bbw, bbh, parent ) {
            layerWidth = width;
            layerHeight = height;
            layerMin = Math.min( width, height );
            layerG2 = this;
            canvas = jSignage.appendCanvasElement( this, 0, 0, width, height );
        } );

        layer.endEvent( function () {
            layerG2.textContent = '';
        } );

        layer.beginEvent( function () {
            var ctx = jSignage.getContext2D( canvas );
            if ( isPercentage )
                data = makePercentageData2( data );
            var numSeries = data.length, numPoints = 0;
            for ( var i=0; i < numSeries; i++ )
                if ( data[i].length > numPoints )
                    numPoints = data[i].length;
            var colors = args.colors || jSignage.Graph.baseColors;
            var markers = args.markers || jSignage.Graph.baseMarkers;
            var defaultTextStyle = new TextStyle( args, layerMin * .04 );
            var pointText = args.pointText || 'none';
            var pointTextStyle = new TextStyle( { color: '#888', fontSize: defaultTextStyle/2 }, defaultTextStyle );
            if ( args.pointTextStyle )
                pointTextStyle = new TextStyle( args.pointTextStyle, pointTextStyle);
            var gridlinesStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMin * .0015, 1 ), color: '#888' }, null );
            if ( args.gridlinesStrokeStyle )
                gridlinesStrokeStyle = new StrokeStyle( args.gridlinesStrokeStyle, gridlinesStrokeStyle );
            var axisStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMin * .0015, 1 ), color: '#88f' }, null );
            if ( args.axisStrokeStyle )
                axisStrokeStyle = new StrokeStyle( args.axisStrokeStyle, axisStrokeStyle );
            var baselineStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMin * .0015, 1 ), color: '#000' }, null );
            if ( args.baselineStrokeStyle )
                baselineStrokeStyle = new StrokeStyle( args.baselineStrokeStyle, baselineStrokeStyle );
            var strokeStyle = new StrokeStyle( { lineWidth: layerMin * .01 }, null );
            if ( args.strokeStyle )
                strokeStyle = new StrokeStyle( args.strokeStyle, strokeStyle );
            var line = args.line || ( dataX ? 'none' : 'straight' );
            var marker = {
                type : typeof(args.marker)=='object' && args.marker.type || ( args.marker || dataSize ? markers : 'none' ),
                size: typeof(args.marker)=='object' && args.marker.size || layerMin * .02,
                strokeStyle: typeof(args.marker)=='object' && args.marker.strokeStyle ? new StrokeStyle( args.marker.strokeStyle, dataSize ? strokeStyle : null ) : dataSize ? strokeStyle : null
            };
            var legend = false;
            if ( 'legend' in args ) {
                if ( typeof(args.legend)=='object' && args.legend!==null ) {
                    legend = {
                        position: args.legend.position || defaultLegendPosition,
                        alignment: args.legend.alignment,
                        textStyle: args.legend.textStyle ? new TextStyle( args.legend.textStyle, defaultTextStyle ) : defaultTextStyle,
                        strokeStyle: args.legend.strokeStyle ? new StrokeStyle( args.legend.strokeStyle, strokeStyle ) : strokeStyle,
                        frameStyle: args.legend.frameStyle ? new StrokeStyle( args.legend.frameStyle, null ) : null
                    }
                }
            } else if ( numSeries > 1 ) {
                legend = { position: 'top', textStyle: defaultTextStyle, strokeStyle: strokeStyle, frameStyle: null };
            }
            var yAxis = args.yAxis || { }, min, max;
            if ( !('minValue' in yAxis) || !('maxValue' in yAxis) ) {
                if ( isPercentage ) {
                    min = 0;
                    max = 1;
                } else {
                    var minmax = findMinMax2( data, isStacked );
                    min = minmax[0];
                    max = minmax[1];
                }
            }
            yAxis = autoSetContinuousAxis( yAxis, min, max, args.locale || null, layerMin, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, baselineStrokeStyle, false, true, true, isPercentage );
            var xAxis = args.xAxis || { };
            if ( xAxis.categories ) {
                xAxis = autoSetDiscreteAxis( xAxis, numPoints, layerMin, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, true, false );
            } else if ( dataX ) {
                if ( !('minValue' in xAxis) || !('maxValue' in xAxis) ) {
                    var minmax = findMinMax2( dataX, false );
                    min = minmax[0];
                    max = minmax[1];
                }
                xAxis = autoSetContinuousAxis( xAxis, min, max, args.locale || null, layerMin, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, baselineStrokeStyle, true, false, true, false );                
            } else {
                var start = args.start || 0;
                var interval = args.interval || 1;
                xAxis = jSignage.extend({
                    minValue: start,
                    maxValue: start + interval * ( numPoints - 1 ),
                    tickInterval: interval
                }, xAxis );
                xAxis = autoSetContinuousAxis( xAxis, null, null, args.locale || null, layerMin, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, baselineStrokeStyle, true, false, true, false );
            }
            var series = [], stack = [];
            for ( var i=0; i < numSeries; i++ ) {
                var s = argsSeries && argsSeries[i] || {}, x = {};
                x.color = s.color || getColor( colors, i );
                x.label = 'label' in s ? s.label : args.labels && i in args.labels ? args.labels[i] : i+1;
                x.strokeStyle = new StrokeStyle( { color: x.color }, strokeStyle );
                if ( s.strokeStyle )
                    x.strokeStyle = new StrokeStyle( s.strokeStyle, x.strokeStyle );
                x.line = s.line || line;
                x.marker = {
                    type: s.marker && s.marker.type || ( jSignage.isArray(marker.type) ? getColor( marker.type, i ) : marker.type ),
                    size: s.marker && s.marker.size || marker.size,
                    color: s.marker && s.marker.color || x.color,
                    strokeStyle: dataSize ? new StrokeStyle( { color: x.color }, marker.strokeStyle ) : marker.strokeStyle
                };
                if ( s.marker && s.marker.StrokeStyle )
                    x.marker.strokeStyle =  new StrokeStyle( s.marker.StrokeStyle, x.marker.strokeStyle );
                x.pointText = s.pointText || pointText;
                x.pointTextStyle = s.pointTextStyle ? new TextStyle( s.pointTextStyle, pointTextStyle ) : pointTextStyle;
                if ( type=='area' || dataSize ) {
                    x.fillColor = s.fillColor || x.color;
                    x.fillOpacity = s.fillOpacity || args.fillOpacity || 1;
                    x.fillOpacityGradient = 'fillOpacityGradient' in s ? s.fillOpacityGradient : 'fillOpacityGradient' in args ? args.fillOpacityGradient : 1;
                } else {
                    x.fillOpacity = 0;
                }
                series.push( x );
            }
            var chartArea = addLegendAndAxes( ctx, layerWidth, layerHeight, legend, yAxis, xAxis, null, null, series, dataSize );
            if ( args.chartArea ) {
                chartArea.x = jSignage.relAbs( args.chartArea.left, layerWidth, chartArea.x );
                chartArea.y = jSignage.relAbs( args.chartArea.top, layerHeight, chartArea.y );
                chartArea.width = jSignage.relAbs( args.chartArea.width, layerWidth, chartArea.width );
                chartArea.height = jSignage.relAbs( args.chartArea.height, layerHeight, chartArea.height );
            }
            var baseY = ( yAxis.baseline - yAxis.minValue )  / ( yAxis.maxValue - yAxis.minValue );
            baseY = chartArea.y + ( yAxis.direction > 0 ? 1 - baseY : baseY ) * chartArea.height;
            var points = [];

            for ( var i=0; i < data.length; i++ ) {
                var sdata = data[i], row = [], move = true, t = null;
                if ( series[i].pointText!=='none' && !jSignage.isArray( series[i].pointText ) )
                    t = [];
                for ( var j=0; j < sdata.length; j++ ) {
                    if ( isStacked || j in sdata ) {
                        var x = xAxis.categories ?  ( j + 0.5 ) / xAxis.categories.length : ( ( dataX ? dataX[i][j] : start + j * interval ) - xAxis.minValue ) / ( xAxis.maxValue - xAxis.minValue );
                        var y = isStacked ? stack[j]=( stack[j] || 0 ) + ( sdata[j] || 0 ) : sdata[j];
                        y = ( y - yAxis.minValue ) / ( yAxis.maxValue - yAxis.minValue );
                        row.push({
                            j: j,
                            move: move,
                            x: chartArea.x + ( xAxis.direction > 0 ? x : 1-x ) * chartArea.width,
                            y: chartArea.y + ( yAxis.direction > 0 ? 1 - y : y ) * chartArea.height
                        });
                        if ( t ) {
                            if ( series[i].pointText=='category' ) {
                                t.push( xAxis.categories ? xAxis.categories[ j ] : null );
                            } else {
                                var value = isStacked ? stack[ j ] : sdata[ j ];
                                t.push( series[i].pointText=='percentage' ? (value*100).toFixed(1) + '%' : value );
                            }
                        }
                        move = false;
                    } else {
                        move = true;
                    }
                }
                points.push( row );
                if ( series[i].line=='spline' ) {
                    for ( var j=1; j < row.length; j++ ) {
                        if ( row[j-1].move ) {
                            row[j].cp1x = row[j-1].x;
                            row[j].cp1y = row[j-1].y;
                        } else {
                            if ( j>=2 && ( ( row[j-1].y > row[j-2].y && row[j].y > row[j-1].y ) || ( row[j-1].y < row[j-2].y && row[j].y < row[j-1].y ) ) && ( ( row[j-1].x > row[j-2].x && row[j-1].x > row[j].x ) || ( row[j-1].x < row[j-2].x && row[j-1].x < row[j].x ) ) )
                                row[j].cp1x = row[j-1].x;
                            else
                                row[j].cp1x = row[j-1].x + ( row[j].x - row[j>=2 ? j-2 : 0].x ) / 6;
                            if ( j>=2 && ( ( row[j-1].x > row[j-2].x && row[j].x > row[j-1].x ) || ( row[j-1].x < row[j-2].x && row[j].x < row[j-1].x ) ) && ( ( row[j-1].y > row[j-2].y && row[j-1].y > row[j].y ) || ( row[j-1].y < row[j-2].y && row[j-1].y < row[j].y ) ) )
                                row[j].cp1y = row[j-1].y;
                            else
                                row[j].cp1y = row[j-1].y + ( row[j].y - row[j>=2 ? j-2 : 0].y ) / 6;
                        }
                        if ( j+1==row.length || row[j+1].move ) {
                            row[j].cp2x = row[j].x;
                            row[j].cp2y = row[j].y;
                        } else {
                            if ( ( ( row[j].y > row[j-1].y && row[j+1].y > row[j].y ) || ( row[j].y < row[j-1].y && row[j+1].y < row[j].y ) ) && ( ( row[j].x > row[j-1].x && row[j].x > row[j+1].x ) || ( row[j].x < row[j-1].x && row[j].x < row[j+1].x ) ) )
                                row[j].cp2x = row[j].x;
                            else
                                row[j].cp2x = row[j].x - ( row[j+1].x - row[j-1].x ) / 6;
                            if ( ( ( row[j].x > row[j-1].x && row[j+1].x > row[j].x ) || ( row[j].x < row[j-1].x && row[j+1].x < row[j].x ) ) && ( ( row[j].y > row[j-1].y && row[j].y > row[j+1].y ) || ( row[j].y < row[j-1].y && row[j].y < row[j+1].y ) ) )
                                row[j].cp2y = row[j].y;
                            else
                                row[j].cp2y = row[j].y - ( row[j+1].y - row[j-1].y ) / 6;                            
                        }
                    }
                }
                if ( t )
                    series[i].pointText = t;
            }

            if ( !dataSize ) for ( var i=0; i < data.length; i++ ) {
                if ( series[i].fillOpacity ) {
                    ctx.fillStyle = series[i].fillColor;
                    if ( series[i].fillOpacityGradient!=1 ) {
                        var grTop = ctx.createLinearGradient( chartArea.x, yAxis.direction > 0 ? chartArea.y : chartArea.y+chartArea.height, chartArea.x, yAxis.direction > 0 ? chartArea.y+chartArea.height : chartArea.y );
                        addOpacityGradientStops( grTop, ctx.fillStyle, series[i].fillOpacityGradient, ( yAxis.direction > 0 ? baseY-chartArea.y : chartArea.y+chartArea.height-baseY)/chartArea.height );
                        ctx.fillStyle = grTop;
                    }
                    ctx.globalAlpha = series[i].fillOpacity;
                    ctx.beginPath();
                    if ( isStacked && i > 0 ) {
                        for ( var j=0; j < points[i].length; j++ ) {
                            var p = points[i][j];
                            if ( p.move )
                                ctx.moveTo( p.x, p.y );
                            else if ( series[i].line=='spline' )
                                ctx.bezierCurveTo( p.cp1x, p.cp1y, p.cp2x, p.cp2y, p.x, p.y );
                            else
                                ctx.lineTo( p.x, p.y );
                        }
                        if ( points[i-1].length >= 1 ) {
                            var p = points[i-1][points[i-1].length-1];
                            ctx.lineTo( p.x, p.y );
                        }
                        for ( var j=points[i-1].length-2; j>=0; --j ) {
                            var p = points[i-1][j];
                            if ( series[i].line=='spline' )
                                ctx.bezierCurveTo( points[i-1][j+1].cp2x, points[i-1][j+1].cp2y, points[i-1][j+1].cp1x, points[i-1][j+1].cp1y, p.x, p.y );
                            else
                                ctx.lineTo( p.x, p.y );
                        }
                    } else {
                        for ( var j=0; j < points[i].length; j++ ) {
                            var p = points[i][j];
                            if ( p.move )
                                ctx.moveTo( p.x, baseY );
                            if ( series[i].line=='spline' && !p.move )
                                ctx.bezierCurveTo( p.cp1x, p.cp1y, p.cp2x, p.cp2y, p.x, p.y );
                            else
                                ctx.lineTo( p.x, p.y );
                            if ( j+1==points[i].length || points[i][j+1].move )
                                ctx.lineTo( p.x, baseY );
                        }
                    }
                    ctx.closePath();
                    ctx.fill();
                }
                ctx.globalAlpha = 1;
            }

            for ( var i=0; i < data.length; i++ ) {
                if ( !dataSize && series[i].line!='none' )  {
                    setStrokeStyle( ctx, series[i].strokeStyle );
                    ctx.beginPath();
                    for ( var j=0; j < points[i].length; j++ ) {
                        var p = points[i][j];
                        if ( p.move )
                            ctx.moveTo( p.x, p.y );
                        else if ( series[i].line=='spline' )
                            ctx.bezierCurveTo( p.cp1x, p.cp1y, p.cp2x, p.cp2y, p.x, p.y );
                        else
                            ctx.lineTo( p.x, p.y );
                    }
                    ctx.stroke();
                }
                if ( series[i].marker.type!='none' ) {
                    if ( series[i].marker.strokeStyle )
                        setStrokeStyle( ctx, series[i].marker.strokeStyle );
                    ctx.fillStyle = series[i].marker.color;
                    for ( var j=0; j < points[i].length; j++ ) {
                        drawMarker( ctx, series[i].marker.type, points[i][j].x, points[i][j].y, dataSize ? dataSize[i][points[i][j].j] : series[i].marker.size );
                        if ( dataSize )
                            ctx.globalAlpha = series[i].fillOpacity;
                        ctx.fill();
                        if ( dataSize )
                            ctx.globalAlpha = 1;
                        if ( series[i].marker.strokeStyle )
                            ctx.stroke();
                    }
                }
            }

            for ( var i=0; i < data.length; i++ ) {
                if ( series[i].pointText!=='none' ) {
                    for ( var j=0; j < points[i].length; j++ ) {
                        var style = series[i].pointTextStyle, p = points[i][j];
                        var text = p.j in series[i].pointText ? series[i].pointText[ p.j ] : null;
                        if ( typeof(text)=='object' && text ) {
                            if ( text.style )
                                style = new TextStyle( text.style, series[i].pointTextStyle );
                            text = text.text;
                        }
                        if ( text!==null ) {
                            ctx.font = style.font();
                            ctx.fillStyle = style.color;
                            var angleLeft, angleRight, angleText, margin = 0;
                            if ( dataSize ) {
                                angleText = 0;
                                ctx.textBaseline = 'middle';
                                ctx.textAlign = 'center';
                            } else if ( series[i].line=='none' ) {
                                angleText = Math.PI;
                                ctx.textBaseline = 'bottom';
                                ctx.textAlign = 'center';                                
                            } else {
                                if ( j > 0 ) {
                                    angleLeft = Math.atan2( (p.x-points[i][j-1].x)*xAxis.direction, (points[i][j-1].y-p.y)*yAxis.direction );
                                    if ( angleLeft < 0 ) angleLeft += 2*Math.PI;
                                } else {
                                    angleLeft = Math.PI/2;
                                }
                                if ( j+1 < points[i].length ) {
                                    angleRight = Math.atan2( (p.x-points[i][j+1].x)*xAxis.direction, (points[i][j+1].y-p.y)*yAxis.direction );
                                    if ( angleRight < 0 ) angleRight += 2*Math.PI;
                                } else {
                                    angleRight = 3*Math.PI/2
                                }
                                var below = yAxis.direction > 0 ? p.y > baseY : p.y < baseY;
                                if ( below ) {
                                    var al = angleLeft - Math.PI;
                                    if ( al < 0 ) al += 2*Math.PI;
                                    var ar = angleRight - Math.PI;
                                    if ( ar < 0 ) ar += 2*Math.PI;
                                    angleText = ( angleLeft + angleRight ) / 2;
                                    angleText += Math.PI;
                                    if ( angleText > 2*Math.PI ) angleText -= 2*Math.PI;
                                } else {
                                    angleText = ( angleLeft + angleRight ) / 2;
                                }
                                if ( angleText >= Math.PI/2  && angleText <= 3*Math.PI/2 ) {
                                    ctx.textBaseline = yAxis.direction > 0 ? 'bottom' : 'top';
                                    if ( angleLeft > Math.PI/2 ) {
                                        if ( angleRight < 3*Math.PI/2 ) {
                                            angleText += Math.PI;
                                            ctx.textBaseline = yAxis.direction > 0 ? 'top' : 'bottom';
                                            ctx.textAlign = 'center';
                                        } else {
                                            ctx.textAlign =  xAxis.direction > 0 ? 'left' : 'right';
                                        }
                                    } else {
                                        ctx.textAlign = angleRight < 3*Math.PI/2 ? xAxis.direction > 0 ? 'right' : 'left' : 'center';
                                    }
                                } else {
                                    ctx.textBaseline = yAxis.direction > 0 ? 'top' : 'bottom';
                                    if ( angleLeft < Math.PI/2 ) {
                                        if ( angleRight > 3*Math.PI/2 ) {
                                            angleText += Math.PI;
                                            ctx.textBaseline = yAxis.direction > 0 ? 'bottom' : 'top';
                                            ctx.textAlign = 'center';                                        
                                        } else {
                                            ctx.textAlign =  xAxis.direction > 0 ? 'left' : 'right';
                                        }
                                    } else {
                                        ctx.textAlign = angleRight > 3*Math.PI/2 ? xAxis.direction > 0 ? 'right' : 'left' : 'center';
                                    }
                                }
                            }
                            if ( !dataSize && series[i].line!='none' )
                                margin += series[i].strokeStyle.lineWidth;
                            if ( !dataSize && series[i].marker.type!='none' ) {
                                margin += series[i].marker.size/1.8;
                                if ( series[i].marker.strokeStyle )
                                    margin += series[i].marker.strokeStyle.lineWidth/2;
                            }
                            ctx.fillText( text, p.x - Math.sin(angleText) * margin * xAxis.direction,  p.y + Math.cos(angleText) * margin * yAxis.direction );
                        }
                    }
                }
            }
            jSignage.flushContext2D( ctx );
        });

        return layer;
    },

    horizontalGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'horizontalGauge', args, 'x', false, false, false, null, null );
    },

    horizontalBarGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'horizontalBarGauge', args, 'x', true, false, false, args.colors || ( args.color ? [args.color] : null ), 'label' in args ? [args.label] : null );
    },

    groupedHorizontalBarGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'groupedHorizontalBarGauge', args, 'x', true, true, false, args.colors, args.labels );
    },

    stackedHorizontalBarGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'stackedHorizontalBarGauge', args, 'x', true, false, true, args.colors, args.labels );
    },

    stackedAndGroupedHorizontalBarGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'stackedAndGroupedHorizontalBarGauge', true, args, 'x', true, true, args.colors, args.labels );
    },

    verticalGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'verticalGauge', args, 'y', false, false, false, null, null );
    },

    verticalBarGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'verticalBarGauge', args, 'y', true, false, false, args.colors || ( args.color ? [args.color] : null ), 'label' in args ? [args.label] : null  );
    },

    groupedVerticalBarGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'groupedVerticalBarGauge', args, 'y', true, true, false, args.colors, args.labels );
    },

    stackedVerticalBarGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'stackedVerticalBarGauge', args, 'y', true, false, true, args.colors, args.labels );
    },

    stackedAndGroupedVerticalBarGauge: function( args ) {
        return jSignage.Graph._linearBarGauge( 'stackedAndGroupedVerticalBarGauge', args, 'y', true, true, true, args.colors, args.labels );
    },

    _linearBarGauge: function ( name, args, orientation, withBar, isGrouped, isStacked, colors, labels ) {
        var layerWidth, layerHeight, layerMin, layerMax, layerParent, layerG2, canvas;

        var layer = jSignage.customLayer( name, args, null, function ( width, height, x, y, bbw, bbh, parent ) {
            layerWidth = width;
            layerHeight = height;
            layerMin = Math.min( width, height );
            layerMax = Math.max( width, height );
            layerG2 = this;
            layerParent = parent;
            canvas = jSignage.appendCanvasElement( this, 0, 0, width, height );
        } );

        layer.endEvent( function () {
            layerG2.textContent = '';
        } );

        layer.beginEvent( function () {
            var ctx = jSignage.getContext2D( canvas );
            var isPercentage = false;
            var defaultTextStyle = new TextStyle( args, layerMin * .15 );
            var text = args.text || null;
            if ( text && typeof(text)!='object' )
                text = {};
            if ( text && !jSignage.isArray( text ) )
                text = [ text ];
            var textStyle = args.textStyle ? new TextStyle( args.textStyle, defaultTextStyle) : defaultTextStyle;
            var gridlinesStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMax * .0015, 1 ), color: '#888' }, null );
            if ( args.gridlinesStrokeStyle )
                gridlinesStrokeStyle = new StrokeStyle( args.gridlinesStrokeStyle, gridlinesStrokeStyle );
            var axisStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMax * .005, 1 ), color: '#000' }, null );
            if ( args.axisStrokeStyle )
                axisStrokeStyle = new StrokeStyle( args.axisStrokeStyle, axisStrokeStyle );
            var baselineStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMax * .0015, 1 ), color: '#000' }, null );
            if ( args.baselineStrokeStyle )
                baselineStrokeStyle = new StrokeStyle( args.baselineStrokeStyle, baselineStrokeStyle );
            var animationDur = jSignage.durInSeconds( args.animationDur, 0 );
            var animateBars = args.animateBars || true;
            var animateIndicator = args.animateIndicator || true;
            var min = args.dataMin || 0, max = args.dataMax || 1;
            if ( min > max ) { var tmp=min; min=max; max=tmp; }
            var axis = args.axis || { }, altAxis = args.altAxis || null;
            var pos = axis.position && ( axis.position=='right' || axis.position=='top' ) ? -1 : 1;
            var locale = args.locale || null;
            axis = autoSetContinuousAxis( axis, min, max, locale, layerMin, defaultTextStyle, axisStrokeStyle, gridlinesStrokeStyle, baselineStrokeStyle, false, false, false, isPercentage, true );
            axis.position = pos;
            if ( altAxis ) {
                var pos = altAxis.position && ( altAxis.position=='left' || altAxis.position=='bottom' ) ? 1 : -1;
                altAxis = autoSetContinuousAxis( altAxis, min, max, locale, layerMin, defaultTextStyle, axisStrokeStyle, null, null, false, false, false, isPercentage, true );
                altAxis.position = pos;
            }
            var thresholds = args.thresholds || null;
            thresholds = thresholds ? jSignage.isArray( thresholds ) ? thresholds : [ thresholds ] : [ ];
            for ( var i=0; i < thresholds.length; i++ ) {
                var pos = 0;
                if ( thresholds[i].position )
                    pos = thresholds[i].position=='left' || thresholds[i].position=='bottom' ? 1 : thresholds[i].position=='right' || thresholds[i].position=='top' ? -1 : 0;
                if ( pos || !withBar ) {
                    var tAxis = axis.position==pos ? axis : altAxis && altAxis.position==pos ? altAxis : axis;
                    thresholds[i] = autoSetThresholds( thresholds[i], layerMin * .1, tAxis.minValue, tAxis.maxValue, layerMin );
                    thresholds[i].position = pos;
                    thresholds[i].axis = tAxis;
                } else {
                    thresholds[i].position = 0;
                }
            }
            var customBar = withBar && args.bar || null;
            var data = args.data || ( isGrouped || isStacked ? [] : 0 );
            colors = autoSetColors( colors || jSignage.Graph.baseColors );
            labels = labels || [];
            var legend = false, defaultLegendPosition = orientation=='x' ? 'bottom' : 'right';
            if ( withBar && args.legend ) {
                var lg = typeof(args.legend)=='object' ? args.legend : {};
                legend = {
                    position: lg.position || defaultLegendPosition,
                    alignment: lg.alignment,
                    orientation: lg.orientation,
                    textStyle: lg.textStyle ? new TextStyle( lg.textStyle, defaultTextStyle ) : defaultTextStyle,
                    strokeStyle: lg.strokeStyle ? new StrokeStyle( lg.strokeStyle, null ) : null,
                    frameStyle: lg.frameStyle ? new StrokeStyle( lg.frameStyle, null ) : null
                }
            }

            var backColor = args.backColor || null;
            var chartArea = addLegend( layerWidth, layerHeight, labels, legend, ctx, colors );
            var axisMargin = measureAxis( ctx, axis, orientation, orientation=='y' ? axis.position : axis.direction, orientation=='y' ? axis.direction : axis.position );
            var altMargin = altAxis ? measureAxis( ctx, altAxis, orientation, orientation=='y' ? altAxis.position : altAxis.direction, orientation=='y' ? altAxis.direction : altAxis.position ) : null;
            var marginLeft = 0, marginRight = 0, marginTop = 0, marginBottom = 0;
            if ( orientation=='y' ) {
                if ( axis.position==-1 )
                    marginRight = Math.max( marginRight, axisMargin.right+axis.offset );
                else
                    marginLeft = Math.max( marginLeft, axisMargin.left+axis.offset );
                marginTop = Math.max( marginTop, axisMargin.top );
                marginBottom = Math.max( marginBottom, axisMargin.bottom );
                if ( altAxis ) {
                    if ( altAxis.position==-1 )
                        marginRight = Math.max( marginRight, altMargin.right+altAxis.offset );
                    else
                        marginLeft = Math.max( marginLeft, altMargin.left+altAxis.offset );
                    marginTop = Math.max( marginTop, altMargin.top );
                    marginBottom = Math.max( marginBottom, altMargin.bottom );
                }
            } else {
                if ( axis.position==-1 )
                    marginTop = Math.max( marginTop, axisMargin.top+axis.offset );
                else
                    marginBottom = Math.max( marginBottom, axisMargin.bottom+axis.offset );
                marginLeft = Math.max( marginLeft, axisMargin.left );
                marginRight = Math.max( marginRight, axisMargin.right );
                if ( altAxis ) {
                    if ( altAxis.position==-1 )
                        marginTop = Math.max( marginTop, altMargin.top+altAxis.offset );
                    else
                        marginBottom = Math.max( marginBottom, altMargin.bottom+altAxis.offset );
                    marginLeft = Math.max( marginLeft, altMargin.left );
                    marginRight = Math.max( marginRight, altMargin.right );
                }
            }
            for ( var i=0; i < thresholds.length; i++ ) {
                if ( thresholds[i].position ) {
                    if ( orientation=='y' ) {
                        if ( thresholds[i].position==-1 )
                            marginRight = Math.max( marginRight, thresholds[i].maxSize );
                        else
                            marginLeft = Math.max( marginLeft, thresholds[i].maxSize );
                    } else {
                        if ( thresholds[i].position==-1 )
                            marginTop = Math.max( marginTop, thresholds[i].maxSize );
                        else
                            marginBottom = Math.max( marginBottom, thresholds[i].maxSize );
                    }
                }
            }
            var indicator = null;
            if ( args.indicator && typeof(args.indicator)=='object' && ( orientation=='y' ? args.indicator.position=='left' || args.indicator.position=='right' : args.indicator.position=='top' || args.indicator.position=='bottom' ) )
                indicator = autoSetLinearIndicator( args.indicator, orientation, layerMax, layerMin, layerMin*.15 );
            if ( indicator && indicator.placement=='outside' ) {
                if ( orientation=='y' ) {
                    if ( indicator.position=='left' )
                        marginLeft = Math.max( marginLeft, indicator.length+indicator.offset );
                    else
                        marginRight = Math.max( marginRight, indicator.length+indicator.offset );
                    marginTop = Math.max( marginTop, indicator.width/2 );
                    marginBottom = Math.max( marginBottom, indicator.width/2 );
                } else {
                    if ( indicator.position=='top' )
                        marginTop = Math.max( marginTop, indicator.length+indicator.offset );
                    else
                        marginBottom = Math.max( marginBottom, indicator.length+indicator.offset );
                    marginLeft = Math.max( marginLeft, indicator.width/2 );
                    marginRight = Math.max( marginRight, indicator.width/2 );
                }
            }
            if ( text ) {
                var ot = [];
                for ( var j=0; j < text.length; j++ ) {
                    var itxt = text[j];
                    var txt = {
                        position: itxt.position || 'center',
                        offset: jSignage.relAbs( itxt.offset, layerMin, 0 ),
                        style: itxt.style ? new TextStyle( itxt.style, textStyle ) : textStyle,
                        where: 'where' in itxt ? itxt.where : 'center'
                    };
                    if ( itxt.align )
                        txt.align = itxt.align;
                    else if ( txt.position=='left' )
                        txt.align = 'right';
                    else if ( txt.position=='right' )
                        txt.align = 'left';
                    else if ( txt.position=='top' || txt.position=='bottom' || txt.where=='center' || orientation=='y' )
                        txt.align = 'center';
                    else
                        txt.align = axis.dir < 0 ? 'right' : 'left';
                    if ( itxt.baseline )
                        txt.baseline = itxt.baseline;
                    else if ( txt.position=='top' )
                        txt.baseline = 'bottom';
                    else if ( txt.position=='bottom' )
                        txt.baseline = 'top';
                    else if ( txt.position=='left' || txt.position=='right' || txt.where=='center' || orientation=='x' )
                        txt.baseline = 'middle';
                    else
                        txt.baseline = axis.dir < 0 ? 'top' : 'bottom';
                    if ( 'text' in itxt ) {
                        txt.text = itxt.text;
                    } else {
                        txt.format = getFormatter( 'format' in itxt ? itxt.format : isPercentage ? '%' : '', locale );
                        txt.text = '';
                        var numChars = itxt.numChars || 5;
                        for ( var i=0; i < numChars; i++ )
                            txt.text += '5';
                    }
                    var m = measureLabels( ctx, txt.style, 'center', 'middle', [ txt.text ] );
                    if ( txt.position=='top' || txt.position=='bottom' ) {
                        if ( txt.position=='top' )
                            marginTop = Math.max( marginTop, txt.offset + ( txt.baseline=='bottom' ? m.ascent + m.descent : txt.baseline=='middle' ? m.ascent : 0 ) );
                        else
                            marginBottom = Math.max( marginBottom, txt.offset + ( txt.baseline=='top' ? m.ascent + m.descent : txt.baseline=='middle' ? m.descent : 0 ) );
                        if ( txt.where!='center' ) {
                            marginLeft = Math.max( marginLeft, txt.align=='right' ? m.left + m.right : txt.align=='left' ? 0 : m.left );
                            marginRight = Math.max( marginRight, txt.align=='left' ? m.left + m.right : txt.align=='right' ? 0 : m.right );
                        }
                    } else if ( txt.position=='left' || txt.position=='right' ) {
                        if ( txt.position=='left' )
                            marginLeft = Math.max( marginLeft, txt.offset + ( txt.align=='right' ? m.left + m.right : txt.align=='left' ? 0 : m.left ) );
                        else
                            marginRight = Math.max( marginRight, txt.offset + ( txt.align=='left' ? m.left + m.right : txt.align=='right' ? 0 : m.right ) );
                        if ( txt.where!='center' ) {
                            marginTop = Math.max( marginTop, txt.baseline=='bottom' ? m.ascent + m.descent : txt.baseline=='middle' ? m.ascent : 0 );
                            marginBottom = Math.max( marginBottom, txt.baseline=='top' ? m.ascent + m.descent : txt.baseline=='middle' ? m.descent : 0 );
                        }
                    } else if ( txt.where!='center' ) {
                        if ( orientation=='y' ) {
                            marginTop = Math.max( marginTop, txt.baseline=='bottom' ? m.ascent + m.descent : txt.baseline=='middle' ? m.ascent : 0 );
                            marginBottom = Math.max( marginBottom, txt.baseline=='top' ? m.ascent + m.descent : txt.baseline=='middle' ? m.descent : 0 );
                        } else {
                            marginLeft = Math.max( marginLeft, txt.align=='right' ? m.left + m.right : txt.align=='left' ? 0 : m.left );
                            marginRight = Math.max( marginRight, txt.align=='left' ? m.left + m.right : txt.align=='right' ? 0 : m.right );
                        }
                    }
                    ot.push( txt );
                }
                text = ot;
            }
            chartArea.x += marginLeft;
            chartArea.width -= marginLeft + marginRight;
            chartArea.y += marginTop;
            chartArea.height -= marginTop + marginBottom;
            if ( args.chartArea ) {
                chartArea.x = jSignage.relAbs( args.chartArea.left, layerWidth, chartArea.x );
                chartArea.y = jSignage.relAbs( args.chartArea.top, layerHeight, chartArea.y );
                chartArea.width = jSignage.relAbs( args.chartArea.width, layerWidth, chartArea.width );
                chartArea.height = jSignage.relAbs( args.chartArea.height, layerHeight, chartArea.height );
            }
            if ( !withBar ) {
                if ( orientation=='y' ) {
                    chartArea.x += chartArea.width / 2;
                    chartArea.width = 0;
                } else {
                    chartArea.y += chartArea.height / 2;
                    chartArea.height = 0;
                }
            }
            var chartWidth = orientation=='y' ? chartArea.width : chartArea.height;
            var groupSize = isGrouped ? data.length : 1;
            var barAlloc = chartWidth / ( groupSize || 1 );
            var barWidth = jSignage.relAbs( args.barWidth, barAlloc, barAlloc );
            var barRange = orientation=='y' ? chartArea.height : chartArea.width;
            for ( var i=0; i < thresholds.length; i++ ) {
                if ( !thresholds[i].position ) {
                    thresholds[i] = autoSetThresholds( thresholds[i], chartWidth, axis.minValue, axis.maxValue, chartWidth );
                    thresholds[i].position = 0;
                    thresholds[i].axis = axis;
                }
            }
            if ( customBar )
                customBar = autoSetThresholds( customBar, barWidth, axis.minValue, axis.maxValue, barWidth );
            if ( !indicator && ( args.indicator || !withBar ) )
                indicator = autoSetLinearIndicator( args.indicator || true, orientation, layerMax, chartWidth, chartWidth );
            if ( withBar && args.backColor ) {
                ctx.fillStyle = args.backColor;
                ctx.fillRect( chartArea.x, chartArea.y, chartArea.width, chartArea.height );
            }
            function hd( x ) { return chartArea.x + x; }
            function hr( x ) { return chartArea.x + chartArea.width - x; }
            function vd( y ) { return chartArea.y + chartArea.height - y; }
            function vr( y ) { return chartArea.y + y }
            function getHV( pos, dir, offset ) {
                var h, v;
                if ( pos==-1 ) {
                    h = orientation=='x' ? dir < 0 ? hr : hd : hr;
                    v = orientation=='y' ? dir < 0 ? vr : vd : vr;
                } else {
                    h = orientation=='x' && dir < 0 ? hr : hd;
                    v = orientation=='y' && dir < 0 ? vr : vd;
                }
                if( offset ) {
                    return [
                        orientation=='y' ? function( x ) { return h( x-offset ); } : h,
                        orientation=='x' ? function( y ) { return v( y-offset ); } : v
                    ];
                } else {
                    return [ h, v ];
                }
            }
            for ( var i=0; i < thresholds.length; i++ ) {
                var hv = getHV( thresholds[i].position, thresholds[i].axis.direction );
                drawThresholds( ctx, thresholds[i], orientation, hv[0], hv[1], barWidth, barRange, thresholds[i].axis.minValue, thresholds[i].axis.maxValue );
            }
            if ( axis.gridlinesStrokeStyle || axis.baselineStrokeStyle ) {
                var hv = getHV( axis.position, axis.direction, 0 );
                drawGridlines( ctx, axis, {}, orientation, hv[0], hv[1], chartArea.width, chartArea.height );
            }
            var hv = getHV( axis.position, axis.direction, axis.offset );
            drawAxis( ctx, axis, orientation, hv[0], hv[1], orientation=='y' ? axis.position : axis.direction, orientation=='y' ? axis.direction : axis.position, barRange );
            if ( altAxis ) {
                hv = getHV( altAxis.position, altAxis.direction, altAxis.offset );
                drawAxis( ctx, altAxis, orientation, hv[0], hv[1], orientation=='y' ? altAxis.position : altAxis.direction, orientation=='y' ? altAxis.direction : altAxis.position, barRange );
            }
            hv = getHV( 0, axis.direction );
            function getBarArea( i, start, end, barStep ) {
                var a1 = i * barAlloc + ( barAlloc - barWidth ) / 2;
                var a2 = i * barAlloc + ( barAlloc + barWidth ) / 2;
                var b1 = ( start - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                var b2 = ( end - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                if ( barStep ) {
                    b1 = Math.floor( ( b1 + barStep/2 ) / barStep ) * barStep;
                    b2 = Math.ceil( ( b2 - barStep/2 ) / barStep ) * barStep;
                }
                var x1 = hv[0]( orientation=='y' ? a1 : b1 );
                var x2 = hv[0]( orientation=='y' ? a2 : b2 );
                var y1 = hv[1]( orientation=='y' ? b1 : a1 );
                var y2 = hv[1]( orientation=='y' ? b2 : a2 );
                if ( x1 > x2 ) { var tmp = x1; x1 = x2; x2 = tmp; }
                if ( y1 > y2 ) { var tmp = y1; y1 = y2; y2 = tmp; }
                return { x: x1, y: y1, width: x2-x1, height: y2-y1 };
            }
            if ( customBar ) {
                if ( args.dynamic ) {
                    var myBars = [], myA = [], previousBarsData = isGrouped || isStacked ? [] : min;
                    for ( var i=0; i < groupSize; i++ ) {
                        if ( isGrouped )
                            previousBarsData[i] = isStacked ? [] : min;
                        var stack = isGrouped ? ( data[i] || 0 ) : data;
                        if ( isStacked )
                            for ( var j=0; j < stack.length; j++ )
                                previousBarsData[i][j] = 0;
                        var barArea = getBarArea( i, axis.minValue, axis.maxValue ), B = {};
                        B.clipPath = jSignage._createElement( 'clipPath' );
                        B.clipPath.id = jSignage.guuid();
                        B.rect = jSignage._createElement( 'rect' );
                        B.clipPath.appendChild( B.rect );
                        layerG2.appendChild( B.clipPath );
                        var g = jSignage._createElement( 'g', { 'clip-path': 'url(#'+B.clipPath.id+')' } );
                        layerG2.appendChild( g );
                        B.ctx = jSignage.getContext2D( jSignage.appendCanvasElement( g, barArea.x, barArea.y, barArea.width, barArea.height ) );
                        customBar.offset = i * barAlloc + ( barAlloc - barWidth ) / 2;
                        drawThresholds( B.ctx, customBar, orientation, hv[0], hv[1], barWidth, barRange, axis.minValue, axis.maxValue );
                        jSignage.flushContext2D( B.ctx );
                        myBars.push( B );
                    }
                    var barStep = 0;
                    if ( customBar.step ) {
                        var n = Math.ceil( ( axis.maxValue - axis.minValue ) / customBar.step );
                        barStep = barRange / n;
                    }
                    function getValues( start, end ) {
                        var v;
                        if ( barStep ) {
                            v = [];
                            var n = Math.round( Math.abs( end - start ) / barStep );
                            for ( var i=0; i <= n; i++ )
                                v.push( ( start * ( n - i ) + end * i ) / n );
                        } else {
                            v = [ start, end ];
                        }
                        return v.join( ';' );
                    }
                    layer.update = function( data ) {
                        for ( var i=0; i < myA.length; i++ )
                            jSignage.removeAnimation( myA[i] );
                        myA = [];
                        for ( var i=0; i < groupSize; i++ ) {
                            var stack = isGrouped ? ( data[i] || 0 ) : data;
                            var previousStack = isGrouped ? ( previousBarsData[i] || 0 ) : previousBarsData;
                            var start = isStacked ? 0 : axis.baseline, previousStart = start, end = start, previousEnd = previousStart;
                            for ( var j=0; j < ( isStacked ? stack.length : 1 ); j++ ) {
                                var value = isStacked ? ( stack[j] || 0 ) : stack;
                                var previousValue = isStacked ? ( previousStack[j] || 0 ) : previousStack;
                                end = isStacked ? end+value : value;
                                previousEnd = isStacked ? previousEnd+previousValue : previousValue;
                            }
                            var B = myBars[i];
                            var barEnd = getBarArea( i, start, end, barStep );
                            var barStart = animateBars && animationDur > 0 ? getBarArea( i, previousStart, previousEnd, barStep ) : barEnd;
                            if ( barStart.x==barEnd.x )
                                B.rect.setAttribute( 'x', barEnd.x );
                            else
                                myA.push( jSignage.svgAnimation( B.rect, 'animate', {
                                    attributeName: 'x',
                                    calcMode: barStep ? 'discrete' : 'linear',
                                    values: getValues( barStart.x, barEnd.x ),
                                    dur: animationDur,
                                    fill: 'freeze'
                                } ) );
                            if ( barStart.y==barEnd.y )
                                B.rect.setAttribute( 'y', barEnd.y );
                            else
                                myA.push( jSignage.svgAnimation( B.rect, 'animate', {
                                    attributeName: 'y',
                                    calcMode: barStep ? 'discrete' : 'linear',
                                    values: getValues( barStart.y, barEnd.y ),
                                    dur: animationDur,
                                    fill: 'freeze'
                                } ) );
                            if ( barStart.width==barEnd.width )
                                B.rect.setAttribute( 'width', barEnd.width );
                            else
                                myA.push( jSignage.svgAnimation( B.rect, 'animate', {
                                    attributeName: 'width',
                                    calcMode: barStep ? 'discrete' : 'linear',
                                    values: getValues( barStart.width, barEnd.width ),
                                    dur: animationDur,
                                    fill: 'freeze'
                                } ) );
                            if ( barStart.height==barEnd.height )
                                B.rect.setAttribute( 'height', barEnd.height );
                            else
                                myA.push( jSignage.svgAnimation( B.rect, 'animate', {
                                    attributeName: 'height',
                                    calcMode: barStep ? 'discrete' : 'linear',
                                    values: getValues( barStart.height, barEnd.height ),
                                    dur: animationDur,
                                    fill: 'freeze'
                                } ) );
                            start = end;
                            previousStart = previousEnd;
                        }
                        for ( var i=0; i < myA.length; i++ )
                            jSignage.beginAnimation( myA[i] );
                        previousBarsData = data;
                    };
                } else {
                    var barIdx = 0;
                    for ( var i=0; i < groupSize; i++ ) {
                        var stack = isGrouped ? ( data[i] || 0 ) : data;
                        var start = isStacked ? 0 : axis.baseline, end = start;
                        for ( var j=0; j < ( isStacked ? stack.length : 1 ); j++ ) {
                            var value = isStacked ? ( stack[j] || 0 ) : stack;
                            end = isStacked ? end+value : value;
                        }
                        customBar.offset = i * barAlloc + ( barAlloc - barWidth ) / 2;
                        drawThresholds( ctx, customBar, orientation, hv[0], hv[1], barWidth, barRange, axis.minValue, axis.maxValue, start, end, layerWidth, layerHeight );
                    }
                }
            } else if ( withBar ) {
                if ( args.dynamic ) {
                    var myBars = [], myA = [], barIdx = 0, previousBarsData = isGrouped || isStacked ? [] : min;
                    for ( var i=0; i < groupSize; i++ ) {
                        if ( isGrouped )
                            previousBarsData[i] = isStacked ? [] : min;
                        var stack = isGrouped ? ( data[i] || 0 ) : data;
                        var previousStack = isGrouped ? ( previousBarsData[i] || 0 ) : previousBarsData;
                        for ( var j=0; j < ( isStacked ? stack.length : 1 ); j++ ) {
                            if ( isStacked )
                                previousStack[j] = 0;
                            var bar = jSignage._createElement( 'rect', { stroke: 'none', fill: getColor( colors, barIdx++ ) } );
                            myBars.push( bar );
                            layerG2.appendChild( bar );
                        }
                    }
                    layer.update = function( data ) {
                        var barIdx = 0;
                        for ( var i=0; i < myA.length; i++ )
                            jSignage.removeAnimation( myA[i] );
                        myA = [];
                        for ( var i=0; i < groupSize; i++ ) {
                            var stack = isGrouped ? ( data[i] || 0 ) : data;
                            var previousStack = isGrouped ? ( previousBarsData[i] || 0 ) : previousBarsData;
                            var start = isStacked ? 0 : axis.baseline, previousStart = start;
                            for ( var j=0; j < ( isStacked ? stack.length : 1 ); j++ ) {
                                var value = isStacked ? ( stack[j] || 0 ) : stack;
                                var previousValue = isStacked ? ( previousStack[j] || 0 ) : previousStack;
                                var end = isStacked ? start+value : value;
                                var previousEnd = isStacked ? previousStart+previousValue : previousValue;
                                var bar = myBars[barIdx];
                                var barEnd = getBarArea( i, start, end );
                                var barStart = animateBars && animationDur > 0 ? getBarArea( i, previousStart, previousEnd ) : barEnd;
                                var colorEnd = getColor( colors, barIdx, end );
                                var colorStart = animateBars && animationDur > 0 ? getColor( colors, barIdx, previousEnd ) : colorEnd;
                                if ( colorStart==colorEnd )
                                    bar.setAttribute( 'fill', colorEnd );
                                else
                                    myA.push( jSignage.svgAnimation( bar, 'animateColor', {
                                        attributeName: 'fill',
                                        from: colorStart,
                                        to: colorEnd,
                                        dur: animationDur,
                                        fill: 'freeze'
                                    } ) );
                                if ( barStart.x==barEnd.x )
                                    bar.setAttribute( 'x', barEnd.x );
                                else
                                    myA.push( jSignage.svgAnimation( bar, 'animate', {
                                        attributeName: 'x',
                                        from: barStart.x,
                                        to: barEnd.x,
                                        dur: animationDur,
                                        fill: 'freeze'
                                    } ) );
                                if ( barStart.y==barEnd.y )
                                    bar.setAttribute( 'y', barEnd.y );
                                else
                                    myA.push( jSignage.svgAnimation( bar, 'animate', {
                                        attributeName: 'y',
                                        from: barStart.y,
                                        to: barEnd.y,
                                        dur: animationDur,
                                        fill: 'freeze'
                                    } ) );
                                if ( barStart.width==barEnd.width )
                                    bar.setAttribute( 'width', barEnd.width );
                                else
                                    myA.push( jSignage.svgAnimation( bar, 'animate', {
                                        attributeName: 'width',
                                        from: barStart.width,
                                        to: barEnd.width,
                                        dur: animationDur,
                                        fill: 'freeze'
                                    } ) );
                                if ( barStart.height==barEnd.height )
                                    bar.setAttribute( 'height', barEnd.height );
                                else
                                    myA.push( jSignage.svgAnimation( bar, 'animate', {
                                        attributeName: 'height',
                                        from: barStart.height,
                                        to: barEnd.height,
                                        dur: animationDur,
                                        fill: 'freeze'
                                    } ) );
                                start = end;
                                previousStart = previousEnd;
                                barIdx++;
                            }
                        }
                        for ( var i=0; i < myA.length; i++ )
                            jSignage.beginAnimation( myA[i] );
                        previousBarsData = data;
                    };
                } else {
                    var barIdx = 0;
                    for ( var i=0; i < groupSize; i++ ) {
                        var stack = isGrouped ? ( data[i] || 0 ) : data;
                        var start = isStacked ? 0 : axis.baseline;
                        for ( var j=0; j < ( isStacked ? stack.length : 1 ); j++ ) {
                            var value = isStacked ? ( stack[j] || 0 ) : stack;
                            var end = isStacked ? start+value : value;
                            ctx.fillStyle = getColor( colors, barIdx++, end );
                            var barArea = getBarArea( i, start, end );
                            ctx.fillRect( barArea.x, barArea.y, barArea.width, barArea.height );
                            start = end;
                        }
                    }
                }
            }
            if ( text ) {
                var myTexts = [];
                for ( var j=0; j < text.length; j++ ) if ( txt.where!='indicator' || ( !isGrouped && !isStacked ) ) {
                    var txt = text[j];
                    var t = txt.format ? txt.format( data ) : txt.text;
                    if ( txt.position=='left' )
                        txt.x = chartArea.x - txt.offset;
                    else if ( txt.position=='right' )
                        txt.x = chartArea.x + chartArea.width + txt.offset;
                    else
                        txt.x = chartArea.x + chartArea.width / 2;
                    if ( txt.position=='top' )
                        txt.y = chartArea.y - txt.offset;
                    else if ( txt.position=='bottom' )
                        txt.y = chartArea.y + chartArea.height + txt.offset;
                    else
                        txt.y = chartArea.y + chartArea.height / 2;
                    if ( txt.where!='center' ) {
                        var where = txt.where=='indicator' ? args.dynamic ? min : data : txt.where;
                        if ( orientation=='y' ) {
                            if ( txt.position!='top' && txt.position!='bottom' )
                                txt.y = hv[1]( ( where - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange );
                        } else {
                            if ( txt.position!='left' && txt.position!='right' )
                                txt.x = hv[0]( ( where - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange );
                        }
                    }
                    ctx.font = txt.style.font();
                    ctx.textBaseline = txt.baseline;
                    if ( !args.dynamic || ( !txt.format && txt.where!='indicator' ) ) {
                        ctx.fillStyle = txt.style.color;
                        ctx.textAlign = txt.align;
                        ctx.fillText( t, txt.x, txt.y );
                    } else {
                        txt.addY = -jSignage.measureText( ctx, '5' ).alphabeticBaseline;
                        txt.svgt = jSignage._createElement( 'text', {
                            stroke: 'none',
                            fill: txt.style.color,
                            'font-family': txt.style.fontFamily,
                            'font-style': txt.style.fontStyle,
                            'font-variant': txt.style.fontVariant,
                            'font-weight': txt.style.fontWeight,
                            'font-size': txt.style.fontSize,
                            'text-anchor': txt.align=='center' ? 'middle' : txt.align,
                            x: txt.where=='indicator' && animateIndicator && animationDur > 0 ? 0 : txt.x,
                            y: txt.where=='indicator' && animateIndicator && animationDur > 0 ? 0 : txt.y + txt.addY
                        });
                        txt.tspan = jSignage._createElement( 'tspan' );
                        txt.tspan.textContent = t;
                        txt.svgt.appendChild( txt.tspan );
                        myTexts.push( txt );
                        layerG2.appendChild( txt.svgt );
                    }
                }
                if ( myTexts.length > 0 ) {
                    var myA = [], previousData = min;
                    var textLink = layer.update;
                    layer.update = function( data ) {
                        if ( textLink )
                            textLink( data );
                        for ( var i=0; i < myTexts.length; i++ ) {
                            var txt = myTexts[i];
                            if ( txt.format )
                                txt.tspan.textContent = txt.format( data );
                            if ( txt.where=='indicator' ) {
                                var ax = txt.x, ay = txt.y;
                                if ( orientation=='y' ) {
                                    if ( txt.position!='top' && txt.position!='bottom' )
                                        txt.y = hv[1]( ( data - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange );
                                } else {
                                    if ( txt.position!='left' && txt.position!='right' )
                                        txt.x = hv[0]( ( data - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange );
                                }
                                if ( animateIndicator && animationDur > 0 ) {
                                    if ( txt.trA )
                                        jSignage.removeAnimation( txt.trA );
                                    txt.trA = jSignage.svgAnimation( txt.svgt, 'animateTransform', {
                                        attributeName: 'transform',
                                        type: 'translate',
                                        from: ax+','+(ay+txt.addY),
                                        to: txt.x+','+(txt.y+txt.addY),
                                        dur: animationDur,
                                        fill: 'freeze'
                                    });
                                    if ( txt.trA )
                                        jSignage.beginAnimation( txt.trA );
                                } else {
                                    if ( txt.x!=ax )
                                        txt.svgt.setAttribute( 'x', txt.x );
                                    if ( txt.y!=ay )
                                        txt.svgt.setAttribute( 'y', txt.y + txt.addY );
                                }
                            }
                        }
                        previousData = data;
                    }
                }
            }
            if ( indicator && !isGrouped && !isStacked ) {
                if ( args.dynamic ) {
                    var indicatorArea = measureLinearIndicator( indicator, orientation, chartArea ), indicatorAnim = null;
                    var g = jSignage._createElement( 'g' );
                    layerG2.appendChild( g );
                    var indicatorCtx = jSignage.getContext2D( jSignage.appendCanvasElement( g, indicatorArea.x, indicatorArea.y, indicatorArea.width, indicatorArea.height ) );
                    drawLinearIndicator( indicatorCtx, indicator, orientation, chartArea, 0 );
                    jSignage.flushContext2D( indicatorCtx );
                    var indicatorLink = layer.update;
                    function getT( data ) {
                        var where = ( data - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                        where = hv[orientation=='y' ? 1 : 0]( where );
                        return {
                            x: orientation=='y' ? 0 : where,
                            y: orientation=='y' ? where : 0
                        };
                    }
                    var previous = getT( min );
                    layer.update = function( data ) {
                        if ( indicatorLink )
                            indicatorLink( data );
                        var now = getT( data );
                        if ( animateIndicator && animationDur > 0 ) {
                            if ( indicatorAnim )
                                jSignage.removeAnimation( indicatorAnim );
                            indicatorAnim = jSignage.svgAnimation( g, 'animateTransform', {
                                attributeName: 'transform',
                                type: 'translate',
                                from: previous.x + ',' + previous.y,
                                to: now.x + ',' + now.y,
                                dur: animationDur,
                                fill: 'freeze'
                            });
                            jSignage.beginAnimation( indicatorAnim );
                        } else {
                            g.setAttribute( 'transform', 'translate('+now.x+','+now.y+')' );
                        }
                        previous = now;
                    };
                } else {
                    var where = ( data - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                    where = hv[orientation=='y' ? 1 : 0]( where );
                    drawLinearIndicator( ctx, indicator, orientation, chartArea, where );
                }
            }
            jSignage.flushContext2D( ctx );
            if ( args.dynamic && layer.update )
                layer.update( data );
        });

        return layer;
    },

    circularGauge: function( args ) {
        return jSignage.Graph._circularBarGauge( 'circularGauge', args );
    },

    circularBarGauge: function( args ) {
        return jSignage.Graph._circularBarGauge( 'circularBarGauge', args, true, false, false, args.colors || ( args.color ? [args.color] : null ), 'label' in args ? [args.label] : null  );
    },

    groupedCircularBarGauge: function( args ) {
        return jSignage.Graph._circularBarGauge( 'groupedCircularBarGauge', args, true, true, false, args.colors, args.labels );
    },

    stackedCircularBarGauge: function( args ) {
        return jSignage.Graph._circularBarGauge( 'stackedCircularBarGauge', args, true, false, true, args.colors, args.labels );
    },

    stackedAndGroupedCircularBarGauge: function( args ) {
        return jSignage.Graph._circularBarGauge( 'stackedAndGroupedCircularBarGauge', args, true, true, true, args.colors, args.labels );
    },

    _circularBarGauge: function ( name, args, withBar, isGrouped, isStacked, colors, labels ) {
        var layerWidth, layerHeight, layerMin, layerMax, layerParent, layerG2, canvas;

        var layer = jSignage.customLayer( name, args, null, function ( width, height, x, y, bbw, bbh, parent ) {
            layerWidth = width;
            layerHeight = height;
            layerMin = Math.min( width, height );
            layerMax = Math.max( width, height );
            layerG2 = this;
            layerParent = parent;
            canvas = jSignage.appendCanvasElement( this, 0, 0, width, height );
        } );

        layer.endEvent( function () {
            layerG2.textContent = '';
        } );

        layer.beginEvent( function () {
            var ctx = jSignage.getContext2D( canvas );
            var isPercentage = false;
            var defaultTextStyle = new TextStyle( args, layerMin * .05 );
            var text = args.text || null;
            if ( text && typeof(text)!='object' )
                text = {};
            if ( text && !jSignage.isArray( text ) )
                text = [ text ];
            var axisStrokeStyle = new StrokeStyle( { lineWidth: Math.max( layerMax * .005, 1 ), color: '#000' }, null );
            if ( args.axisStrokeStyle )
                axisStrokeStyle = new StrokeStyle( args.axisStrokeStyle, axisStrokeStyle );
            var animationDur = jSignage.durInSeconds( args.animationDur, 0 );
            var animateBars = args.animateBars || true;
            var animateIndicator = args.animateIndicator || true;
            var min = args.dataMin || 0, max = args.dataMax || 1;
            if ( min > max ) { var tmp=min; min=max; max=tmp; }
            var startAngle = ( 'startAngle' in args ? args.startAngle/180*Math.PI : -3*Math.PI/4 ) - Math.PI / 2;
            var endAngle = ( 'endAngle' in args ? args.endAngle/180*Math.PI : 3*Math.PI/4 ) - Math.PI / 2;
            var axis = args.axis || { }, altAxis = args.altAxis || null;
            var pos = axis.position && axis.position=='out' ? -1 : 1;
            var reverse = axis.reverse || false;
            var locale = args.locale || null;
            axis = autoSetContinuousAxis( axis, min, max, locale, layerMin, defaultTextStyle, axisStrokeStyle, null, null, !withBar, false, false, isPercentage, true );
            axis.position = pos;
            if ( altAxis ) {
                var pos = altAxis.position && altAxis.position=='in' ? 1 : -1;
                altAxis = autoSetContinuousAxis( altAxis, min, max, locale, layerMin, defaultTextStyle, axisStrokeStyle, null, null, !withBar, false, false, isPercentage, true );
                altAxis.position = pos;
            }
            var thresholds = args.thresholds || null;
            thresholds = thresholds ? jSignage.isArray( thresholds ) ? thresholds : [ thresholds ] : [ ];
            for ( var i=0; i < thresholds.length; i++ ) {
                var pos = 0;
                if ( thresholds[i].position )
                    pos = thresholds[i].position=='in' ? 1 : thresholds[i].position=='out' ? -1 : 0;
                if ( pos || !withBar ) {
                    var tAxis = axis.position==pos ? axis : altAxis && altAxis.position==pos ? altAxis : axis;
                    thresholds[i] = autoSetThresholds( thresholds[i], .1*layerMin, tAxis.minValue, tAxis.maxValue, layerMin );
                    thresholds[i].position = pos;
                    thresholds[i].axis = tAxis;
                } else {
                    thresholds[i].position = pos;
                }
            }
            var data = args.data || ( isGrouped || isStacked ? [] : 0 );
            colors = autoSetColors( colors || jSignage.Graph.baseColors );
            labels = labels || [];
            var legend = false;
            if ( withBar && args.legend ) {
                var lg = typeof(args.legend)=='object' ? args.legend : {};
                legend = {
                    position: lg.position || 'right',
                    alignment: lg.alignment,
                    orientation: lg.orientation,
                    textStyle: lg.textStyle ? new TextStyle( lg.textStyle, defaultTextStyle ) : defaultTextStyle,
                    strokeStyle: lg.strokeStyle ? new StrokeStyle( lg.strokeStyle, strokeStyle ) : null,
                    frameStyle: lg.frameStyle ? new StrokeStyle( lg.frameStyle, null ) : null
                }
            }
            var backColor = args.backColor || null;
            var chartArea = addLegend( layerWidth, layerHeight, labels, legend, ctx, colors );
            var axisMargin = measureCircularAxis( ctx, axis );
            var chartRadius = withBar ? jSignage.relAbs( args.gaugeRadius, layerMin, layerMin/6 ) : 0;
            var alpha1H = 0, alpha1V = 0, alpha2H = 0, alpha2V = 0;

            if ( axis.position < 0 ) {
                alpha1H = Math.max( alpha1H, axisMargin.alpha + axisMargin.maxTextWidth );
                alpha1V = Math.max( alpha1V, axisMargin.alpha + axisMargin.maxTextHeight );
            } else {
                alpha2H = Math.max( alpha2H, axisMargin.alpha + axisMargin.maxTextWidth );
                alpha2V = Math.max( alpha2V, axisMargin.alpha + axisMargin.maxTextHeight );
            }
            if ( altAxis ) {
                var altMargin = measureCircularAxis( ctx, altAxis );
                if ( altAxis.position < 0 ) {
                    alpha1H = Math.max( alpha1H, altMargin.alpha + altMargin.maxTextWidth );
                    alpha1V = Math.max( alpha1V, altMargin.alpha + altMargin.maxTextHeight );
                } else {
                    alpha2H = Math.max( alpha2H, altMargin.alpha + altMargin.maxTextWidth );
                    alpha2V = Math.max( alpha2V, altMargin.alpha + altMargin.maxTextHeight );
                }
            }
            for ( var i=0; i < thresholds.length; i++ ) {
                if ( thresholds[i].position < 0 ) {
                    alpha1H = Math.max( alpha1H, thresholds[i].maxSize );
                    alpha1V = Math.max( alpha1V, thresholds[i].maxSize );
                }
            }
            var indicator = null;
            if ( args.indicator && typeof(args.indicator)=='object' && ( args.indicator.position=='in' || args.indicator.position=='out' || args.indicator.position=='bar' ) )
                indicator = autoSetCircularIndicator( args.indicator, layerMax, layerMin, withBar && args.indicator.position=='bar' ? chartRadius : layerMin*.075 );
            if ( indicator ) {
                if ( indicator.position=='out' ) {
                    var il = indicator.length + indicator.offset + (indicator.back ? indicator.back.length : 0)
                    alpha1H = Math.max( alpha1H, il );
                    alpha1V = Math.max( alpha1V, il );
                } else if ( indicator.position=='in' ) {
                    alpha2H = Math.max( alpha2H, -indicator.offset );
                    alpha2V = Math.max( alpha2V, -indicator.offset );
                } else {
                    var il = indicator.length/2 + indicator.offset - chartRadius/2;
                    alpha1H = Math.max( alpha1H, il );
                    alpha1V = Math.max( alpha1V, il );
                }
            }
            var radius = Math.min( chartArea.width/2 - alpha1H, chartArea.height/2 - alpha1V, chartArea.width/2 - alpha2H + chartRadius,  chartArea.height/2 - alpha2V + chartRadius );
            radius = Math.max( radius, 0 );
            chartArea.x += chartArea.width/2 - radius;
            chartArea.y += chartArea.height/2 - radius;
            chartArea.width = radius * 2;
            chartArea.height = radius * 2;
            if ( args.chartArea ) {
                chartArea.x = jSignage.relAbs( args.chartArea.left, layerWidth, chartArea.x );
                chartArea.y = jSignage.relAbs( args.chartArea.top, layerHeight, chartArea.y );
                chartArea.width = jSignage.relAbs( args.chartArea.width, layerWidth, chartArea.width );
                chartArea.height = jSignage.relAbs( args.chartArea.height, layerHeight, chartArea.height );
            }
            radius = Math.min( chartArea.width, chartArea.height ) / 2;
            if ( !indicator && ( !withBar || args.indicator ) )
                indicator = autoSetCircularIndicator( args.indicator || true, layerMax, radius-chartRadius, radius-chartRadius );
            var cx = chartArea.x + chartArea.width / 2;
            var cy = chartArea.y + chartArea.height / 2;
            var startAngle = startAngle - Math.floor( startAngle / (2*Math.PI) )*2*Math.PI;
            var endAngle = endAngle - Math.floor( endAngle / (2*Math.PI) )*2*Math.PI;
            if ( endAngle==startAngle )
                endAngle = reverse ? startAngle - 2*Math.PI : startAngle + 2*Math.PI;
            var groupSize = isGrouped ? data.length : 1;
            var barAlloc = chartRadius / ( groupSize || 1 );
            var barWidth = jSignage.relAbs( args.barWidth, barAlloc, barAlloc );
            var barRange = endAngle - startAngle;
            if ( !reverse && barRange < 0 )
                barRange += 2*Math.PI;
            else if ( reverse && barRange > 0 )
                barRange -= 2*Math.PI;
            if ( withBar ) for ( var i=0; i < thresholds.length; i++ ) {
                if ( !thresholds[i].position ) {
                    thresholds[i] = autoSetThresholds( thresholds[i], chartRadius, axis.minValue, axis.maxValue, chartRadius );
                    thresholds[i].position = 0;
                    thresholds[i].axis = axis;
                }
            }
            if ( text ) {
                var ot = [], textAngle, textRadius, textPlacement;
                if ( barRange > Math.PI ) {
                    textAngle = endAngle + ( 2*Math.PI - barRange ) / 2;
                    textRadius = radius;
                    textPlacement = 'inside';
                } else {
                    textAngle = -Math.PI/2;
                    textRadius = 0;
                    textPlacement = 'center';
                }
                ctx.textAlign = 'center';
                ctx.textBaseline = 'middle';
                for ( var j=0; j < text.length; j++ ) {
                    var itxt = text[j];
                    var txt = {
                        angle: 'angle' in itxt ? itxt.angle/180*Math.PI-Math.PI/2 : textAngle,
                        radius: jSignage.relAbs( itxt.radius, radius, textRadius ),
                        placement: itxt.placement || textPlacement,
                        style: itxt.style ? new TextStyle( itxt.style, defaultTextStyle ) : defaultTextStyle
                    };
                    if ( textRadius==0 && !('radius' in itxt) )
                        txt.radius = txt.style.fontSize/2;
                    if ( 'text' in itxt ) {
                        txt.text = itxt.text;
                    } else {
                        txt.format = getFormatter( 'format' in itxt ? itxt.format : isPercentage ? '%' : '', locale );
                        if ( args.dynamic ) {
                            txt.text = '';
                            var numChars = itxt.numChars || 5;
                            for ( var i=0; i < numChars; i++ )
                                txt.text += '5';
                        } else {
                            txt.text = txt.format( data );
                        }
                    }
                    ctx.font = txt.style.font();
                    txt.area = getTextOnCircleArea( ctx, cx, cy, txt.radius, txt.angle, txt.placement, txt.text );
                    ot.push( txt );
                }
                text = ot;
            }
            if ( withBar && args.backColor ) {
                ctx.fillStyle = args.backColor;
                ctx.beginPath();
                ctx.arc( cx, cy, radius-chartRadius, startAngle, endAngle, reverse );
                ctx.arc( cx, cy, radius, endAngle, startAngle, !reverse );
                ctx.closePath();
                ctx.fill();
            }
            for ( var i=0; i < thresholds.length; i++ )
                drawCircularThresholds( ctx, thresholds[i], cx, cy, thresholds[i].position < 0 ? radius : thresholds[i].position > 0 ? radius-chartRadius : radius-chartRadius/2, startAngle, barRange, reverse, thresholds[i].axis.minValue, thresholds[i].axis.maxValue );
            drawCircularAxis( ctx, axis, cx, cy, axis.position < 0 ? radius : radius-chartRadius, axis.reverse ? endAngle : startAngle, barRange );
            if ( altAxis )
                drawCircularAxis( ctx, altAxis, cx, cy, altAxis.position < 0 ? radius : radius-chartRadius, altAxis.reverse ? endAngle : startAngle, barRange );
            if ( withBar ) {
                if ( args.dynamic ) {
                    var myBars = [], myA = [], barIdx = 0, previousBarsData = isGrouped || isStacked ? [] : min;
                    for ( var i=0; i < groupSize; i++ ) {
                        var r = radius - chartRadius + ( i + .5 ) * barAlloc;
                        if ( isGrouped )
                            previousBarsData[i] = isStacked ? [] : min;
                        var stack = isGrouped ? ( data[i] || 0 ) : data;
                        var previousStack = isGrouped ? ( previousBarsData[i] || 0 ) : previousBarsData;
                        for ( var j=0; j < ( isStacked ? stack.length : 1 ); j++ ) {
                            if ( isStacked )
                                previousStack[j] = 0;
                            var d = new jSignage.pathData();
                            d.moveTo( cx + r * Math.cos( startAngle ), cy + r * Math.sin( startAngle ) );
                            if ( Math.abs(barRange)>=6 ) {
                                d.arcTo( r, r, 0, 0, reverse ? 0 : 1, cx + r * Math.cos( startAngle+barRange/2 ), cy + r * Math.sin( startAngle+barRange/2 ) );
                                d.arcTo( r, r, 0, 0, reverse ? 0 : 1, cx + r * Math.cos( endAngle ), cy + r * Math.sin( endAngle ) );
                            } else {
                                d.arcTo( r, r, 0, endAngle-startAngle > Math.PI || endAngle < startAngle ? 1 : 0, reverse ? 0 : 1, cx + r * Math.cos( endAngle ), cy + r * Math.sin( endAngle ) );
                            }
                            var bar = jSignage._createElement( 'path', { d: d, stroke: getColor( colors, barIdx++ ), fill: 'none', 'stroke-width': barWidth, 'stroke-linecap': 'butt' } );
                            myBars.push( bar );
                            layerG2.appendChild( bar );
                        }
                    }
                    layer.update = function( data ) {
                        var barIdx = 0;
                        for ( var i=0; i < myA.length; i++ )
                            jSignage.removeAnimation( myA[i] );
                        myA = [];
                        for ( var i=0; i < groupSize; i++ ) {
                            var r = radius - chartRadius + ( i + .5 ) * barAlloc;
                            var stack = isGrouped ? ( data[i] || 0 ) : data;
                            var previousStack = isGrouped ? ( previousBarsData[i] || 0 ) : previousBarsData;
                            var start = isStacked ? 0 : axis.baseline, previousStart = start;
                            for ( var j=0; j < ( isStacked ? stack.length : 1 ); j++ ) {
                                var value = isStacked ? ( stack[j] || 0 ) : stack;
                                var previousValue = isStacked ? ( previousStack[j] || 0 ) : previousStack;
                                var end = isStacked ? start+value : value;
                                var previousEnd = isStacked ? previousStart+previousValue : previousValue;
                                var bar = myBars[barIdx];
                                var endA1 = startAngle + ( start - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                                var endA2 = startAngle + ( end - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                                var endC = getColor( colors, barIdx, end );
                                function getDashArray( a1, a2 ) {
                                    var D = [];
                                    if ( isStacked ) {
                                        D.push( 0 );
                                        D.push( Math.abs( a1 - startAngle ) *r );
                                    }
                                    D.push( Math.abs( a2 - a1 ) * r );
                                    D.push( 3 * Math.PI * r );
                                    return D.join( ',' );
                                }
                                if ( animateBars && animationDur > 0 ) {
                                    var startA1 = startAngle + ( previousStart - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                                    var startA2 = startAngle + ( previousEnd - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                                    var startC = getColor( colors, barIdx, previousEnd );
                                    myA.push( jSignage.svgAnimation( bar, 'animate', {
                                        attributeName: 'stroke-dasharray',
                                        from: getDashArray( startA1, startA2 ),
                                        to: getDashArray( endA1, endA2 ),
                                        dur: animationDur,
                                        fill: 'freeze'
                                    } ) );
                                    if ( endC==startC ) {
                                        bar.setAttribute( 'stroke', endC );
                                    } else {
                                        myA.push( jSignage.svgAnimation( bar, 'animateColor', {
                                            attributeName: 'stroke',
                                            from: startC,
                                            to: endC,
                                            dur: animationDur,
                                            fill: 'freeze'
                                        } ) );
                                    }
                                } else {
                                    bar.setAttribute( 'stroke-dasharray', getDashArray( endA1, endA2 ) );
                                    bar.setAttribute( 'stroke', endC );
                                }
                                start = end;
                                previousStart = previousEnd;
                                barIdx++;
                            }
                        }
                        for ( var i=0; i < myA.length; i++ )
                            jSignage.beginAnimation( myA[i] );
                        previousBarsData = data;
                    };
                } else {
                    var barIdx = 0;
                    for ( var i=0; i < groupSize; i++ ) {
                        var stack = isGrouped ? ( data[i] || 0 ) : data;
                        var start = isStacked ? 0 : axis.baseline;
                        for ( var j=0; j < ( isStacked ? stack.length : 1 ); j++ ) {
                            var value = isStacked ? ( stack[j] || 0 ) : stack;
                            var end = isStacked ? start+value : value;
                            var r1 = radius - chartRadius + i * barAlloc + ( barAlloc - barWidth ) / 2;
                            var r2 = radius - chartRadius + i * barAlloc + ( barAlloc + barWidth ) / 2;
                            var a1 = startAngle + ( start - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                            var a2 = startAngle + ( end - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                            ctx.fillStyle = getColor( colors, barIdx++, end );
                            ctx.beginPath();
                            ctx.arc( cx, cy, r1, a1, a2, reverse );
                            ctx.arc( cx, cy, r2, a2, a1, !reverse );
                            ctx.closePath();
                            ctx.fill();
                            start = end;
                        }
                    }
                }
            }
            if ( text ) {
                var myTexts = [];
                ctx.textAlign = 'center';
                ctx.textBaseline = 'middle';
                for ( var i=0; i < text.length; i++ ) {
                    var txt = text[i];
                    ctx.font = txt.style.font();
                    ctx.fillStyle = txt.style.color;
                    if ( args.dynamic && txt.format ) {
                        txt.svgt = jSignage._createElement( 'text', {
                            stroke: 'none',
                            fill: txt.style.color,
                            'font-family': txt.style.fontFamily,
                            'font-style': txt.style.fontStyle,
                            'font-variant': txt.style.fontVariant,
                            'font-weight': txt.style.fontWeight,
                            'font-size': txt.style.fontSize,
                            'text-anchor': 'middle',
                            x: txt.area.textX,
                            y: txt.area.textY + txt.area.addY
                        } );
                        myTexts.push( txt );
                        layerG2.appendChild( txt.svgt );
                    } else {
                        ctx.fillText( txt.text, txt.area.textX, txt.area.textY );
                    }
                }
                if ( myTexts.length > 0 ) {
                    var previousTextData = min;
                    var textLink = layer.update;
                    layer.update = function( data ) {
                        if ( textLink )
                            textLink( data );
                        for ( var i=0; i < myTexts.length; i++ ) {
                            var txt = myTexts[i];
                            txt.svgt.textContent = txt.format( data );
                        }
                    }
                }
            }
            if ( indicator && !isGrouped && !isStacked ) {
                var indicatorRadius = indicator.position=='in' ? radius-chartRadius : indicator.position=='out' ? radius : indicator.position=='bar' ? radius-chartRadius/2 : 0;
                if ( args.dynamic ) {
                    var indicatorArea = measureCircularIndicator( indicator, indicatorRadius );
                    var g = jSignage._createElement( 'g' );
                    layerG2.appendChild( g );
                    var indicatorCtx = jSignage.getContext2D( jSignage.appendCanvasElement( g, cx+indicatorArea.x, cy+indicatorArea.y, indicatorArea.width, indicatorArea.height ) );
                    drawCircularIndicator( indicatorCtx, indicator, cx, cy, 0, indicatorRadius );
                    jSignage.flushContext2D( indicatorCtx );
                    var indicatorA = null;
                    var previousAngle = startAngle;
                    var indicatorLink = layer.update;
                    layer.update = function( data ) {
                        if ( indicatorLink )
                            indicatorLink( data );
                        var indicatorAngle = startAngle + ( data - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                        if ( animateIndicator && animationDur > 0 ) {
                            if ( indicatorA )
                                jSignage.removeAnimation( indicatorA );
                            indicatorA = jSignage.svgAnimation( g, 'animateTransform', {
                                attributeName: 'transform',
                                type: 'rotate',
                                from: (previousAngle/Math.PI*180)+','+cx+','+cy,
                                to: (indicatorAngle/Math.PI*180)+','+cx+','+cy,
                                dur: animationDur,
                                fill: 'freeze'
                            });
                            jSignage.beginAnimation( indicatorA );
                        } else {
                            g.setAttribute( 'transform', 'rotate('+(indicatorAngle/Math.PI*180)+','+cx+','+cy+')' );
                        }
                        previousAngle = indicatorAngle;
                    };
                } else {
                    var indicatorAngle = startAngle + ( data - axis.minValue ) / ( axis.maxValue - axis.minValue ) * barRange;
                    drawCircularIndicator( ctx, indicator, cx, cy, indicatorAngle, indicatorRadius );
                }
            }
            jSignage.flushContext2D( ctx );
            if ( args.dynamic && layer.update )
                layer.update( data );
        });

        return layer;
    },

    richText: function( text ) {
        var r = { style: null };
        if ( jSignage.isArray( text ) ) {
            alert( 'This text must have uniform formatting' );
            r.text = 'error';
        } else if ( typeof(text)=='object' && text!==null ) {
            r.text = 'text' in text ? text.text : '';
            r.style = {};
            jSignage.copyProps( text, r.style, richTextToStyle );
        } else {
            r.text = text===null || text===undefined ? '' : text;
        }
        return r;
    },
    
    mergeStyle: function( A, B ) {
        for ( x in B ) {
            if ( !(x in A) )
                A[x] = B[x];
            else if ( jSignage.isArray( A[x] ) )
                A[x] = A[x].concat( B[x] );
            else if ( typeof(A[x])=='object' && A[x] && typeof(B[x])=='object' && B[x] )
                jSignage.Graph.mergeStyle( A[x], B[x] );
            else
                A[x] = B[x];
        }
        return A;
    }
};
} ) ( );
