#!/bin/bash
#
#  Copyright 2006 by Texas Instruments Incorporated.
#
#  Modifications (C) 2010 SpinetiX S.A.
#  
#  All rights reserved. Property of Texas Instruments Incorporated.
#  Restricted rights to use, duplicate or disclose this code are
#  granted through contract.
#  
#

# WARNING: this script is referred also from the recovery system when
# installing a new firmware image, so do not change the name of the
# argument or its meaning, nor the location of this script. It must be
# executable from the recovery system (i.e. no funky dependencies).

# Modules are loaded from the current directory if present (using insmod),
# otherwise they are loaded from the collection of system modules by modprobe.
#
# A failure in loading a module or creating the device node immediately fails
# this script.
#
# If a module is already loaded a warning is printed but the module is not
# reloaded.

function loadmodule
{
    local modname="$1"
    shift
    grep -q "^$modname" /proc/modules && \
        echo "WARNING: $modname already loaded" >&2 && return
    if [ -f "$modname".ko ]; then
        insmod "$modname".ko "$@"
    else
        modprobe "$modname" "$@"
    fi
}

#
# Function to load the hardware type into HARDWARE variable
#
function load_hardware
{
    local tag sep val
    HARDWARE=
    while read tag sep val; do
	if [ "$tag" = "Hardware" ]; then
	    HARDWARE="$val"
	    return 0
	fi
    done < /proc/cpuinfo
    echo "Could not find hardware type" >&2
    return 1
}

#
# Function to load the parameters based on HARDWARE variable
#
function load_params
{
    # The CMEM memory should be placed right after Linux memory, hence
    # CMEM_START and CMEM_END must be adjusted if KERNEL_MEM is changed

    # this is to be set to non-empty if the fw_setenv and fw_printenv
    # can be safely used
    FWENV=

    case "$HARDWARE" in
	"Bonsai"|"DaVinci EVM")
	    [ "$HARDWARE" == "Bonsai" ] && FWENV=1
	    KERNEL_MEM="176128k"
	    CMEM_START="0x8AC00000"
	    CMEM_END="0x8b000000"
	    CMEM_POOLS="1x2097152,2x1048576"
	    CMEM_MAX_SHARED="20971520"
	    ;;
	"DaVinci DM6467 EVM"|"DaVinci DM6467T EVM")
	    KERNEL_MEM="143360k"
	    CMEM_START="0x88C00000"
	    CMEM_END="0x89000000"
	    CMEM_POOLS="1x2097152,2x1048576"
	    CMEM_MAX_SHARED="20971520"
	    ;;
	"Sakura")
	    FWENV=1
	    KERNEL_MEM="354304k"
	    CMEM_START="0x95A00000"
	    CMEM_END="0x95E00000"
	    CMEM_POOLS="1x2097152,2x1048576"
	    CMEM_MAX_SHARED="31457280"
	    ;;
	"ti8148evm")
	    HDVPSS_IMG="ti814x_hdvpss_1G.xem3"
	    KERNEL_MEM="262144k"
	    CMEM_START="0x90000000"
	    CMEM_END="0x90400000"
	    CMEM_POOLS="2x2093056,2x4096"
	    CMEM_MAX_SHARED="31457280"
	    ;;
	"ikebana")
	    HDVPSS_IMG="ikebana_hdvpss_1G.xem3"
	    KERNEL_MEM="262144k"
	    CMEM_START="0x90000000"
	    CMEM_END="0x90500000"
	    CMEM_POOLS="2x2093056,2x4096,1x1048576"
	    CMEM_MAX_SHARED="31457280"
	    ;;
	*)
	    echo "unknown hardware '$HARDWARE'" >&2
	    return 1
	    ;;
    esac
    return 0
}

#
# Functions to check and set the amount of maximum kernel memory
#

function check_kmem
{
    [ -n "FWENV" ] || return 0 # nothing to check

    local mem="$(fw_printenv -n mem 2>/dev/null)"
    if [ "$mem" != "$KERNEL_MEM" ]; then
	if [ "$1" != "-s" ]; then
	    echo "incorrect kernel memory: '$mem' instead of '$KERNEL_MEM'" >&2
	fi
	return 1
    else
	return 0
    fi
}

function set_kmem
{
    local ret

    [ -n "FWENV" ] || return 1 # not supported

    fw_setenv mem "$KERNEL_MEM"
    ret=$?
    if [ "$ret" -eq 0 ]; then
	echo "Successfully set kernel memory to '$KERNEL_MEM'"
    else
	echo "Failed setting kernel memory to '$KERNEL_MEM'" >&2
    fi
    return $ret
}

function upgrade_kmem
{
    check_kmem -s || set_kmem
}

function upgrade_kargs
{
    [ -n "$args" ] || return 0 # no value, nothing to do

    [ -n "FWENV" ] || return 1 # not supported

    local args="$(fw_printenv -n mainexargs 2>/dev/null)"
    local nargs
    local ret

    # Extract the known kernel arguments, currently only
    # musb_hdrc.sched_intr_ep=
    nargs="$(echo "$args" | awk 'BEGIN { nargs="" } { for (i = 1; i <= NF; i = i + 1) { na=""; if ($i ~ /^musb_hdrc\.sched_intr_ep=/) na=$i; if (na != "") { if (nargs != "") nargs = nargs " "; nargs = nargs na } } } END { print nargs; } ')"

    [ "$nargs" = "$args" ] && return 0 # no change

    # Set the resulting new argument list
    if [ -n "$nargs" ]; then
	fw_setenv mainexargs "$nargs"
    else
	fw_setenv mainexargs
    fi
    ret=$?
    if [ "$ret" -eq 0 ]; then
	echo "Successfully cleaned kernel extra arguments"
    else
	echo "Failed cleaning kernel extra arguments" >&2
    fi
    return $ret
}

# Load hardware type and parameters
load_hardware && load_params || exit 1

# Handle request for upgrading kernel memory and extra arguments

if [ "$1" == "upgrade" ]; then
    upgrade_kmem && upgrade_kargs
    exit
fi

# Check that the correct kernel memory is set, if not we modify it so
# that next boot takes it.
if ! check_kmem; then
    echo "setting correct value for future boots" >&2
    set_kmem >&2
fi

# insert cmemk using the hardware dependent parameters
loadmodule cmemk \
    phys_start="$CMEM_START" phys_end="$CMEM_END" \
    pools="$CMEM_POOLS" max_shared="$CMEM_MAX_SHARED" \
    || exit

if [ -n "$HDVPSS_IMG" ]; then

    loadmodule syslink || exit
    # Wait some time to correctly load syslink. 1 sec seems to be enough.
    sleep 1
    (./slaveloader list | grep -q "VPSS-M3 \[Unknown\]") && (./slaveloader startup VPSS-M3 "$HDVPSS_IMG" > /dev/null || exit)
    loadmodule vpss sbufaddr=0xA0200000 || exit
    loadmodule ti81xxvo video1_numbuffers=11 video2_numbuffers=0 video3_numbuffers=0 || exit
    loadmodule ti81xxhdmi || exit

else

    # insert dsplinkk
    loadmodule dsplinkk || exit

    # make /dev/dsplink
    rm -f /dev/dsplink
    mknod -m 660 /dev/dsplink c `awk "\\$2==\"dsplink\" {print \\$1}" /proc/devices` 0 && \
	chgrp dsp /dev/dsplink

fi

#
#  @(#) codec_engine_1_02 1,0,0,147 7-14-2006 ce-d14
#

