<?php
class IPConfig extends CFormModel {
	
    const NET_ETHERNET  = "ethernet";
	const NET_MODEM     = "modem";
    
    protected $loaded=false;
    var $netiface=self::NET_ETHERNET;
    var $dhcp=true;
    var $address="";
	var $netmask="";
	var $gateway="";
    
    var $nameserver1="";
    var $nameserver2="";
    var $nameserver3="";
    var $domain="";
	
    var $fnameDNS="/etc/network/resolv.conf.static";
    
    
    protected $startTag="# START-spxsysconfig DO NOT REMOVE THE START AND END TAGS";
	protected $endTag="# END-spxsysconfig";
	protected $fname="/etc/network/interfaces";
    
    var $configType = array ( "DHCP"=>1, "Static"=>0 );
    var $netifaceSelect = array ( IPConfig::NET_ETHERNET=>"Ethernet", IPConfig::NET_MODEM=>"3G Modem" );
    
    /**
	 * @return array validation rules for model attributes.
	 */
	public function rules() {
		return array(
			array('address', 'required', 'on'=>'fixip'),
            array('address, gateway, nameserver1, nameserver2, nameserver3', 'checkIP', 'on'=>'fixip'),
            array('netmask', 'checkNetmask', 'on'=>'fixip'),
            array('domain', 'safe', 'on'=>'fixip'),
            array('netiface', 'safe', 'on'=>'modem3g, dhcp, fixip'),
            array('dhcp', 'safe'),
		);
	}
    
    public function attributeLabels()
	{
		return array(
            'netiface'=>'Use this interface for the network: ',
            'address'=>'Address: ',
            'netmask'=>'Netmask: ',
            'gateway'=>'Gateway: ',
            'nameserver1'=>'Server 1: ',
            'nameserver2'=>'Server 2: ',
            'nameserver3'=>'Server 3: ',
            'domain'=>'DNS suffix: ',
    	);
	}
    
    function mask($adr,$msk){
		$a=ip2long($adr);
		$m=ip2long($msk);
		return long2ip($a & $m);
	}
	function broad($adr,$msk){
		$a=ip2long($adr);
		$m=ip2long($msk);
		
		return long2ip(($a & $m)+(~$m));
	}
    public function beforeValidate() {
        // insure that the DHCP is 0 or 1, and not "0" or "1"
        $this->dhcp = intval( $this->dhcp );
        return parent::beforeValidate();
    }
    public function changedInterface( $array ) {
        if ( !isset($array['netiface']) ) return false;
        if ( $array['netiface']==$this->netiface ) return false;
        return true;
    }
    
    public function loadIP() {
        if ( $this->loaded ) 
            return true;
            
        if ( !file_exists($this->fname) ) {
        
            $this->addError('file','Failed loading IP config. '.'Please re-instal the firmware.');
            return false;
        }
        
        $interface=file_get_contents($this->fname);
        if ($interface=="") {
            $this->addError('file','Configuration file is incomplete. '.'Please re-instal the firmware.');
            return false;
        }
        
        $start=strpos($interface,$this->startTag);
		$end=strpos($interface,$this->endTag);
		if ($start===false || $end===false) return false;
        
        // should look for first auto interface to be
		// consistent with ifwatchdog
		$str1="auto ";
		$iface=strpos($interface,$str1, $start);
		$iface+=strlen($str1);
		$line_end=strpos($interface,"\n",$iface);
		$ifname = substr($interface,$iface,$line_end-$iface);

		if ($ifname == "ppp0") 
			$this->netiface = self::NET_MODEM;
		else
			$this->netiface = self::NET_ETHERNET;
        
        $str1="iface $ifname inet ";
		$iface=strpos($interface,$str1, $start);
		$iface+=strlen($str1);
		$conf=substr($interface,$iface,$end-$iface);
		$vals=explode ("\n",$conf);	
		
		if (trim($vals[0])=="dhcp"){
			$this->dhcp = 1;
		}elseif (trim($vals[0])=="static"){
			$this->dhcp = 0;
			foreach ($vals as $name){
				if (!(($n=strpos($name,"address "))===false)){
					$this->address=trim(substr($name, strlen("address")+$n));
				}elseif (!(($n=strpos($name,"netmask "))===false)){
					$this->netmask=trim(substr($name, strlen("netmask")+$n));
				}elseif (!(($n=strpos($name,"gateway "))===false)){
					$this->gateway=trim(substr($name, strlen("gateway")+$n));
				}
			}
		}elseif (trim($vals[0])=="ppp"){
			// nothing else to read, we always use the same ppp config
		} else
            $this->addError('dhcp', "Warning: Unknown inet state: '".$vals[0]."'");
        
        $this->loaded=true;
        return true;
    }
    
    public function saveIP() {
        if ($this->netiface == self::NET_MODEM)
			$iface="ppp0";
		else
			$iface="eth0";
		$conf="auto $iface\niface $iface inet ";
		
		if ($this->netiface == self::NET_MODEM) {
			$conf.="ppp\n";
			$conf.="\tprovider modem\n";
			$conf.="\tpre-up /usr/libexec/spxdevs/modem-check\n";
		} else if (!$this->dhcp){
		    $conf.="static\n";
			$conf.="\taddress $this->address\n";
			$conf.="\tnetwork ".$this->mask($this->address,$this->netmask)."\n";
			$conf.="\tnetmask $this->netmask\n";
			if ($this->gateway!="") $conf.="\tgateway $this->gateway\n";
			$conf.="\tbroadcast ".$this->broad($this->address,$this->netmask)."\n";
			$conf.="\tpre-up [ ! -x /etc/network/eth-inet-dad ] || /etc/network/eth-inet-dad\n";
			
		} else {
			$conf.="dhcp\n";
		}
			

        if ( !file_exists($this->fname) ) {
            $this->addError('address','Failed opening network configuration, please try again.');
            return false;
        }
        
        $interface=file_get_contents($this->fname);
        if ($interface=="") {
            $this->addError('address','Network configuration is empty, please try again.');
            return false;
        }
        
        $start=strpos($interface,$this->startTag);
		$end=strpos($interface,$this->endTag);
		
		if ($start===false) $start=strlen($interface);	
		if ($end===false) {
			$end=$start;	
			$conf.=$this->endTag."\n";
		}
		if (!Yii::app()->user->tools->save_file($this->fname,substr_replace($interface,$this->startTag."\n".$conf,$start,$end-$start))) {
            $this->addError('file','Failed saving network configuration, please try again.');
			return false;
        }

		return true;
	}

    function loadDNS(){
		if ( !file_exists($this->fnameDNS) ) {
            $this->addError('file','Failed loading DNS config. '.'Please re-instal the firmware.');
            return false;
        }
        $dns=file_get_contents($this->fnameDNS);
		$tmp=$dns;
		$i=1;
		$pos=strpos($tmp,"nameserver");
		while (!($pos===false)){
			$pos+=strlen("nameserver");
			$name="nameserver".$i; 
			$i++;
			$this->$name=trim(substr($tmp,$pos,strpos($tmp,"\n",$pos)-$pos));
			$tmp=substr($tmp,strpos($tmp,"\n",$pos));
			$pos=strpos($tmp,"nameserver");
		}
			
		$pos=strpos($dns,"domain ");
		if (!($pos===false)) {
            $pos+=strlen("domain "); 
            $this->domain=trim(substr($dns,$pos,strpos($dns,"\n",$pos)-$pos));
        }
	}
	function saveDNS(){
		$conf="";
		if ($this->nameserver1!="" ) 
			$conf.="nameserver $this->nameserver1\n"; 
		if ($this->nameserver2!="" ) 
			$conf.="nameserver $this->nameserver2\n"; 
		if ($this->nameserver3!="" ) 
			$conf.="nameserver $this->nameserver3\n"; 
			
		if ($this->domain!="") $conf.="domain $this->domain\n";
		if ( $conf == "" ) $conf="\n"; 
		return Yii::app()->user->tools->save_file($this->fnameDNS,$conf);
	}
    
    function saveSafeModeConf(){
		// copy settings to failsafe-data
		if ($this->netiface == self::NET_MODEM) {
			$netconf = "auto\n";
		} elseif ($this->dhcp) {
		    $netconf = "dhcp\n";
		} else {
		    $netconf = "static=" . $this->address . ";" .
			$this->netmask . ";" .
			$this->broad($this->address,$this->netmask ) . ";";
		    if ($this->gateway!="") 
			    $netconf .= $this->gateway;
            $netconf .= ";";
            if ($this->nameserver1 != "")
                $netconf .= $this->nameserver1;
            $netconf .= ";";
            if ($this->nameserver2 != "")
                $netconf .= $this->nameserver2;
            $netconf .= ";";
            if ($this->nameserver3 != "")
                $netconf .= $this->nameserver3;
            $netconf .= ";";
            if ($this->domain != "")
                $netconf .= $this->domain;
		    $netconf .= "\n";
		}

		if ( ! Yii::app()->user->tools->setSafeModeData("network", $netconf, FALSE) ){
            $this->addError('file','Failed saving network configuration for recovery console, please try again.');
            return false;
        }

        return true;
    }
    
    
    function load() {
        $this->loadIP();
        $this->loadDNS();
    }
	
    function save() {
        $this->saveIP();
        $this->saveDNS();
        $this->saveSafeModeConf();
        
        if ($this->netiface == ipconfig::NET_MODEM) {
             Yii::app()->user->tools->messages[] = "<h3>3G modem will be used (automatic IP address)</h3>" ;
         } elseif ($this->dhcp) {
             $message = "<h3>DHCP will be used on Ethernet.</h3>";
             $hostname = trim(file_get_contents("/etc/hostname"));
             $message .= "The unit is always reachable at <a href='http://$hostname.local/'>http://$hostname.local</a> from Zeroconf enabled computers on the local network.";
             Yii::app()->user->tools->messages[] = $message;
         } else  {
             Yii::app()->user->tools->messages[] = "<h3>New ip will be <a href='http://{$this->address}/'>{$this->address}</a> on Ethernet</h3>";
         }
     
        Yii::app()->user->tools->addReason( "network config change (" . $this->netiface . ", " .
                                            ($this->netiface == IPConfig::NET_MODEM ? "automatic IP" :
                                            ($this->dhcp ? "DHCP" : $this->address)) . ")" );
            
    }
	
    
    public function checkConfig(  ){
        if ( $this->dhcp ) return true;
        
        $a=ip2long($this->address);
        $m=ip2long($this->netmask);
        if ( $a===false || $a==-1 ) {
            $this->addError('address','Valid format xxx.xxx.xxx.xxx, where xxx is between 0-255');
            return false;
        }
        if ( $m===false || $m==-1 ) {
            $this->addError('netmask','Valid format xxx.xxx.xxx.xxx, where xxx is between 0-255');
            return false;
        }
        if ( ($a & ip2long("255.0.0.0")) == ip2long("127.0.0.0") ){
            $this->addError('address','127.0.0.0/8 is reserved for loopback addresses');
            return false;
        }
       
        if ( ($a & $m) == $a ) {
            $this->addError('address','IP address is incompatible with netmask');
            $this->addError('netmask',"");
            return false;
        }
        if ( ($a & ~$m) == ~$m ) {
            $x=~$m;
            $this->addError('address','Address and netmask combination is not allowed');
            $this->addError('netmask',"");
            return false;
        }
        
        if ( $this->gateway!="" ) {
            
            $g=ip2long($this->gateway);
            if ( ($a & $m) != ($g & $m) ) {
                $this->addError('address','Gateway not in the same subnet as the address.');
                $this->addError('netmask',"");
                $this->addError('gateway',"");
                
                return false;
            }
            if ( $a == $g ) {
                $this->addError('address','Gateway cannot be equal to the address.');
                $this->addError('gateway',"");
                return false;
            }
            if ( ($g & $m) == $g || ($g & ~$m) == ~$m) {
                $this->addError('gateway','Gateway value not correct');
                return false;
            }
        }
        for ( $i=1; $i<4; $i++ ){
            $name = 'nameserver'.$i;
            $d=ip2long($this->$name);
            if ( $this->$name!="" ) {
                if ( $a == $d ) {
                    $this->addError('address','DNS cannot be equal to the address.');
                    $this->addError($name,"");
                    return false;
                }
                if ( ($a & $m) == ($d & $m) ) {
                    if ( ($d & $m) == $d ) {
                        $this->addError($name,'Reserved addresses');
                        return false;
                    }
                    if ( ($d & ~$m) == ~$m) {
                        $this->addError($name,'Reserved addresses');
                        return false;
                    }
                }
            }
        }
        return true;
    }
    
    
    
}