<?php

class Modem3G extends CFormModel {
	
    protected $_active=null;
    var $statusstring="";
    
    var $signal=false;
    var $device=false;
    var $serial=false;
    var $subscriber=false;
    
    var $simple=1;
    
    var $ipconf=null;
    
    public function rules() {
        return array(
			array('simple', 'safe'),
		);
    }
    
    public function attributeLabels()
	{
		$labels =  array(
            'statusstring'=>'Modem status:',
            'simple'=>'Configuration type:'
    	);
        foreach ($this->getFormNames() as $fname) {
            $labels[$fname] = ucfirst($this->getFieldDesc($fname));
        }
        return $labels;
	}
    
    public function getStatusString() {
        if ($this->_active===null){
            $this->updateStatus();
        }
        return $this->statusstring;
    }
    public function updateStatus(  ) {
        $modempresent = (file_exists(self::$devnode) && file_exists(self::$status_dir . '/status'));
        
        $this->_active = $modempresent;
        if ( !$this->_active ) {
            if ( $this->ipconf===null )
                $this->ipconf = new IPConfig;
            $this->ipconf->loadIP();
            $this->_active = ($this->ipconf->netiface == IPConfig::NET_MODEM);
            if ( !$this->_active ) return;
        }
        // update $this->status        
        $linktext = 'connect log';
        if ($modempresent) {
            switch ( $this->getStatus() ) {
            case 'READY':
                switch ( $this->getConnectStatus() ) {
                    case "":
                        $linktext = 'previous connect log';
                        $this->statusstring="ready"; break;
                    case 'ONLINE':
                        $this->statusstring="online"; break;
                    case 'CONNECTING':
                        $linktext = 'previous connect log';
                        $this->statusstring="connecting"; break;
                    case 'CONNECTED':
                        $this->status="connected, initializing link"; break;
                    case 'DISCONNECTED':
                        $this->statusstring="disconnected"; break;
                    case 'IO ERROR':
                        $this->statusstring="I/O error while connecting"; break;
                    case 'TIMEOUT':
                        $this->statusstring="bad command or timeout"; break;
                    case 'NETWORK ERROR':
                        $this->statusstring="phone network error"; break;
                    default:
                        $this->statusstring="ready, unknown connection status"; break;
                    }
                $connectlog = realpath('../log/modem-connect.log');
                if ($connectlog)
                    $this->statusstring.=" (<a href='log/modem-connect.log'>" . $linktext . "</a>)";
                break;
            case 'PIN INCORRECT':
                $this->statusstring="locked, PIN is incorrect"; break;
            case 'PIN':
                $this->statusstring="locked, PIN required"; break;
            case 'PUK INCORRECT':
                $this->statusstring="locked, PUK is incorrect"; break;
            case 'PUK':
                $this->statusstring="locked, PUK required"; break;
            case 'PIN2':
                $this->statusstring="locked, PIN2 required"; break;
            case 'PUK2':
                $this->statusstring="locked, PUK2 required"; break;
            case 'ERROR':
                $this->statusstring="initialization failed"; break;
            default:
                $this->statusstring="unknown state"; break;
            }
        } else {
            $this->statusstring="not found";
        }
        // update $this->signal
        $sq = self::getSignalQuality();
        if ($sq === false)
            $this->signal = false;
        else
            $this->signal = htmlspecialchars($sq). "&nbsp;%";
        
        // update $this->device
        $this->device="";
        if ($modempresent && file_exists(self::$status_dir . '/manufacturer'))
            $this->device.= " " . htmlspecialchars(trim(file_get_contents(self::$status_dir . '/manufacturer')));
        if ($modempresent && file_exists(self::$status_dir . '/model'))
            $this->device.= " " . htmlspecialchars(trim(file_get_contents(self::$status_dir . '/model')));
        if ($modempresent && file_exists(self::$status_dir . '/revision'))
            $this->device.= " (rev. " . htmlspecialchars(trim(file_get_contents(self::$status_dir . '/revision'))) . ")";
        if (!$modempresent)
            $this->device = false;
            
        // update $this->serial
        if ($modempresent && file_exists(self::$status_dir . '/serial'))
            $this->serial = htmlspecialchars(file_get_contents(self::$status_dir . '/serial'));
        else
            $this->serial = false;
            
        // update $this->subscriber
        if ($modempresent && file_exists(self::$status_dir . '/imsi'))
            $this->subscriber = htmlspecialchars(file_get_contents(self::$status_dir . '/imsi'));
        else
            $this->subscriber = false;
    }
    
    public function getActive() {
        if ($this->_active===null){
            $this->updateStatus();
        }
        return $this->_active;
    }
    
    
    
    
    // NOTE: The 3GPP AT commands are defined in 3GPP TS 27.007

    const NUM_INIT = 6;
    private static $form_prefix = 'm_';
    private static $fdesc = array(
        "pin" => "PIN",
        "puk" => "PUK",
        "dial" => "dial number",
        "apn" => "APN",
        "pdptype" => "PDP type",
        "username" => "username",
        "password" => "password",
        "authproto" => "auth protocol");
    private static $defaults = array(
        // init1 to init3 are the defaults used in simple mode
        "init1" => "AT Z", // after ATZ there can be no trailing commands
        "init2" => "AT Q0 V1 E1 &C1 &D2 +CMEE=2",
        "init3" => "AT +CR=1",
        "dial" => "*99**1#",
        "pdptype" => "IP",
        "authproto" => "any");
    private static $choice = array(
        "pdptype" => array('IP'=>"IP", 'PPP'=>"PPP"),
        "authproto" => array('any'=>"any", 'none'=>"none", 'pap'=>'pap', 'chap'=>'chap')
        );
    private static $type = array( // if not in the table, this is advanced
        "pin" => "all",
        "puk" => "all",
		"pdptype" => "simple",
        "apn" => "simple",
		"username" => "all",
        "password" => "all",
        "authproto" => "all"
        );
    private static $help = array(
        'puk' => "required to reset the PIN to the above value",
        'init1' => "e.g., '&nbsp;AT Z&nbsp;'",
        "init2" => "e.g., '&nbsp;AT Q0 V1 E1 &amp;C1 &amp;D2 +CMEE=2&nbsp;'",
        "init3" => "e.g., '&nbsp;AT +CGREG=2; +CR=1; +CGREG?; +CGEREP=1&nbsp;'",
        "init4" => "e.g., '&nbsp;AT +CGDCONT=<i>&lt;cid&gt;</i>,\"IP\",\"<i>&lt;apn&gt;</i>\"&nbsp;'",
        "dial" => "e.g., '&nbsp;*99**1#&nbsp;' or '&nbsp;*99**1*<i>&lt;cid&gt;</i>#&nbsp;'",
        "username" => "use [serial] as placeholder for the serial number" );
    private static $peer_file = '/etc/ppp/peers/modem';
    private static $chat_file = '/etc/chatscripts/modem';
    private static $conf_file = '/etc/network/modem.conf';
    private static $pass_file = '/etc/ppp/modem-password';
    private static $pin_file = '/etc/chatscripts/modem-pin';
    private static $modem_ctl = '/usr/libexec/spxdevs/modem-control';
    private static $status_dir = '/dev/.modem-data';
    private static $devnode = '/dev/modem';
    private static $devnodectl = '/dev/modemctl';

    private $fields = array();
    private $changed;
    private $status = '';
    private $connect_status = '';
    

    static function getFormNames() {
        static $fnames;
        if (isset($fnames))
            return $fnames;
        $fnames[] = self::$form_prefix . 'pin';
        $fnames[] = self::$form_prefix . 'puk';
        for ($i = 1; $i <= self::NUM_INIT; $i++) 
            $fnames[] = self::$form_prefix . "init$i";
        foreach (self::$fdesc as $key => $val) {
            if ($key != 'pin' && $key != 'puk')
                $fnames[] = self::$form_prefix . "$key";
        }
        return $fnames;
    }

    static function getFieldDesc($name) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        if (strncmp($name, 'init', 4) == 0)
            return 'init string #' . substr($name, 4);
        else
            return self::$fdesc[$name];
    }

    static function getFieldHelp($name) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        if (array_key_exists($name, self::$help))
            return self::$help[$name];
        else
            return null;
    }
    
    static function isSimpleOnly($name) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        return array_key_exists($name, self::$type) &&  self::$type[$name]=="simple";
    }
    static function isAdvancedOnly($name) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        return !array_key_exists($name, self::$type);
    }

    static function isChoiceField($name) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        return array_key_exists($name, self::$choice);
    }

    static function getFieldChoices($name) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        return self::$choice[$name];
    }

    /**
     * Returns the signal quality. This method is slow, so do not use
     * int time critical places.
     * @return The signal quality as a percentage or FALSE
     * on error
     */
    static function getSignalQuality() {
        if ( ! file_exists(self::$devnodectl) )
            return false;
        $cmd = self::$modem_ctl .
            " -c " . escapeshellarg('+CSQ') .
            " " . self::$devnodectl;
        exec($cmd, $out, $retval);
        if ($retval != 0)
            return false;
        foreach ($out as $line) {
            if (preg_match("@^\+CSQ:\s+(\d+)@", $line, $matches)) {
                $rssi = $matches[1];
                if ($rssi < 0 || $rssi >= 99)
                    return false; // unknown or not detectable
                $dbm = -113 + 2 * $rssi;
                if ($dbm >= -65)
                    return 1;
                return round(100 * ($dbm+113) / 48);
            }
        }
        return false;
    }

    static function quote($str, $enclose = true) {
        static $from = array("\\", "\"");
        static $to = array("\\\\", "\\\"");
        $str = str_replace($from, $to, $str);
        if ( $enclose )
            return '"' . $str . '"';
        else
            return str_replace(' ', '\s', $str); // escape spaces
    }

    static function atstring($str) {
        static $from = array("\\", "\"");
        static $to = array("\\5C", "\\22");
        return '"' . str_replace($from, $to, $str) . '"';
    }

    function isFieldActive($name) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        if ($name == 'puk')
            return $this->isPINBlocked();
        return true;
    }
    /*
    public function __set($name,$value) {
        echo "->Set $name = $value\n";
        if ( in_array( $name, self::getFormNames() ) ){
            $this->setField($name, $value);            
            $ret = $value;
        } else {
            $ret = parent::__set( $name, $value );
        }                 
        return $ret;
    }
     */
    public function __get( $name ) {        
        if ( in_array( $name, self::getFormNames() ) ){
            return $this->getField( $name );
        } else {            
            return parent::__get( $name );            
        }
    }
    public function beforeValidate() {
        // insure that the simple is 0 or 1, and not "0" or "1"
        $this->simple = intval( $this->simple );
        return parent::beforeValidate();
    }
    
    function setAttributes( $array ) {
        
        $wassimple = $this->simple;
        parent::setAttributes( $array );
        
        if ( $wassimple!=$this->simple )
            $this->changed = true;        
        
        $vals=self::getFormNames();
        foreach ($vals as $name){
            $this->setField($name, isset($array[$name]) ? $array[$name] : '');
        }
    }
    
    function setField($name, $val) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        $val = isset($val) ? trim($val) : $val='';
        if ($this->simple) {
            if (!array_key_exists($name, self::$type))
                $val=''; // advanced only field
        } else {
            if (array_key_exists($name, self::$type) && self::$type[$name]=="simple")
                $val=''; // simple only field
        }
        if (array_key_exists($name, $this->fields)) {
            if ($this->fields[$name] == $val)
                return;
        } else {
            if (strlen($val) == 0)
                return;
        }
        if (strlen($val) != 0)
            $this->fields[$name] = $val;
        else
            unset($this->fields[$name]);
        $this->changed = true;
    }

    function getField($name) {
        $name = preg_replace("/^" . self::$form_prefix . "/", '', $name);
        if (array_key_exists($name, $this->fields))
            return $this->fields[$name];
        else
            return '';
    }

    function isPINBlocked() {
        return ($this->getStatus() == 'PUK' || $this->getStatus() == 'PUK INCORRECT');
    }

    function isChanged() {
        return $this->changed;
    }

	function init() {
		$this->clear();
	}

	function load() {
        if ( !file_exists(self::$conf_file) ) 
            return ;
            
        $xml = new DOMDocument();
        if ( ! $xml->load(self::$conf_file) )
            return;
        $ml = $xml->getElementsByTagName('modem');
        if ($ml->length == 0)
            return;
        $this->fields = array();
        $modem = $ml->item(0);
        $n_init = 0;
        foreach ($ml->item(0)->childNodes as $item) {
            if ($item->nodeType != XML_ELEMENT_NODE)
                continue; // skip all comments, whitespace, etc.
            $name = $item->nodeName;
            if (strncmp($name, 'init', 4) == 0) {
                $n_init++;
                $name = $name . $n_init;
            }
            $this->fields[$name] = trim($item->textContent);
        }
        $pin = file_get_contents(self::$pin_file);
        if ($pin)
            $this->fields['pin'] = $pin;
        $pass = file_get_contents(self::$pass_file);
        if ($pass)
            $this->fields['password'] = $pass;
        $this->changed = false;

        // The prescence of pdptype means a simple configuration
        $this->simple = array_key_exists('pdptype', $this->fields);
	}

	function save() {

        if ( $this->hasErrors() ) return false;
        
        if ( !$this->changed ) return true;
        
        
		if ( $this->simple ){ // simple config, set implicit parameters
			if ( !isset($this->fields["apn"]) )
				$this->fields["apn"]="";
        	$this->fields['init1'] = self::$defaults['init1'];
        	$this->fields['init2'] = self::$defaults['init2'];
        	$this->fields['init3'] = self::$defaults['init3'];
        	$this->fields['init4'] =
                'AT +CGDCONT=1,' . $this->atstring($this->fields["pdptype"]) .
                ',' . $this->atstring($this->fields["apn"]);
        	$this->fields['dial'] = '*99**1*1#';
        }
        
        // create the xml modem config
        $xml = new DOMDocument();
        $xml->formatOutput = true;
        $root = $xml->createElement('modems');
        $xml->appendChild($root);
        $modem = $xml->createElement('modem');
        $root->appendChild($modem);
        foreach ($this->fields as $name => $field) {
            if (strncmp($name, 'init', 4) == 0) {
                $xname = 'init';
            } elseif ($name == 'pin') {
                continue;
            } elseif ($name == 'puk') {
                continue;
            } elseif ($name == 'password') {
                continue;
            } else {
                $xname = $name;
            }
            $node = $xml->createElement($xname);
            $modem->appendChild($node);
            $node->appendChild($xml->createTextNode($field));
        }
        

        // create chat script
        $chat = '';
        $chat .= "# Use a small timeout while we set everything up\n";
        $chat .= "TIMEOUT 3\n";
        $chat .= "# The network failue strings on which we abort\n";
        $chat .= "# we timeout instead of abort on error messages\n";
        $chat .= "# so as to get the full message in the logs\n";
        $chat .= "ABORT BUSY\n";
        $chat .= "ABORT 'NO CARRIER'\n";
        $chat .= "ABORT VOICE\n";
        $chat .= "ABORT 'NO DIALTONE'\n";
        $chat .= "ABORT 'NO DIAL TONE'\n";
        $chat .= "ABORT 'NO ANSWER'\n";
        $chat .= "ABORT DELAYED\n";
        $noexpect = true;
        $chat .= "# --- Initialization\n";
        $chat .= "SAY \"--- passing init strings ---\\n\"\n";
        for ($i = 1; $i <= self::NUM_INIT; $i++) {
            $name = "init$i";
            if (array_key_exists($name, $this->fields)) {
                $chat .= $noexpect ? "'' " : "OK\\r\\n ";
                $chat .= $this->quote($this->fields[$name]) . "\n";
                $noexpect = false;
            }
        }
        if ( ! $noexpect ) {
            $chat .= "OK\\r\\n \\c\n";
            $noexpect = true;
        }
        if (array_key_exists('dial', $this->fields)) {
            $chat .= "# --- Dial\n";
            $chat .= $noexpect ? "'' " : "OK\\r\\n ";
            $chat .= $this->quote("ATD{$this->fields['dial']}") . "\n";
            $noexpect = false;
        }
        $chat .= "# --- Done, expect data state confirmation and transition to data state\n";
        $chat .= "# Session establishment may take a long time\n";
        $chat .= "TIMEOUT 20\n";
        $chat .= "SAY \"--- waiting for connection ---\\n\"\n";
        $chat .= "CONNECT \c\n";
        $chat .= "\\n \\c\n";
        $chat .= "SAY \"--- entered data state ---\\n\"\n";

        // create ppp peer definition
        $peer = '';
        $peer .= "# Enable debugging\n";
        $peer .= "debug\n";
        $peer .= "# Do not detach until the protocol is up but discard logs on stdout\n";
        $peer .= "updetach\n";
        $peer .= "logfile /dev/null\n";
        $peer .= "# Be sure to use ppp0 as the interface\n";
        $peer .= "unit 0\n";
        $peer .= "# The modem device file\n";
        $peer .= self::$devnode . "\n";
        $peer .= "# Create the lock file for the device\n";
        $peer .= "lock\n";
        $peer .= "# Use hardware flow control\n";
        $peer .= "crtscts\n";
        $peer .= "# Be sure ppp makes use of control lines\n";
        $peer .= "modem\n";
        $peer .= "# Use pid file /var/run/ppp-modem.pid\n";
        $peer .= "linkname modem\n";
        $peer .= "# Do not require peer to authenticate\n";
        $peer .= "noauth\n";
        $peer .= "# Do not attempt to determine local IP address from hostname\n";
        $peer .= "noipdefault\n";
        $peer .= "# Do not let ppp enter a default route; it is handled elsewhere\n";
        $peer .= "nodefaultroute\n";
        $peer .= "# Get DNS servers from peer\n";
        $peer .= "usepeerdns\n";
        $peer .= "# Do not do proxy ARP\n";
        $peer .= "noproxyarp\n";
        $peer .= "# Do not retry after connection goes down, this is handled elsewhere\n";
        $peer .= "nopersist\n";
        if ( $this->simple && strncmp($this->fields["pdptype"], 'IP', 2) == 0 ) {
            $peer .= "# Modem using native IP connection, disable PPP data compression\n";
            $peer .= "nobsdcomp\n";
            $peer .= "nodeflate\n";
            $peer .= "nopredictor1\n";
            $peer .= "# Modem using native IP connection, disable IP header compression\n";
            $peer .= "novj\n";
        }
        $peer .= "# The script to initialize the modem connection\n";
        $peer .= "connect /usr/libexec/spxdevs/modem-connect\n";
        $peer .= "# The script to bring down the modem connection\n";
        $peer .= "disconnect \"/usr/libexec/spxdevs/modem-connect -d\"\n";
        if (array_key_exists('username', $this->fields)) {
            global $serial;
            $username = str_replace('[serial]', $serial, $this->fields['username']);
            $peer .= "# The user to use in authentication\n";
            $peer .= "user " . $this->quote($username) . "\n";
        }
        if (array_key_exists('authproto', $this->fields)) {
            switch ($this->fields['authproto']) {
            case 'chap':
                $peer .= "# Use 'modem-chap' as remote name for CHAP only auth\n";
                $peer .= "remotename modem-chap\n";
                break;
            case 'pap':
                $peer .= "# Use 'modem-pap' as remote name for PAP only auth\n";
                $peer .= "remotename modem-pap\n";
                break;
            case 'none':
                $peer .= "# Use 'modem-none' as remote name for no authentication\n";
                $peer .= "remotename modem-none\n";
                break;
            case 'any':
                $peer .= "# Use 'modem' as remote name for auth with any protocol\n";
                $peer .= "remotename modem\n";
                break;
            }
        }

        // The pin password values
        $pin = array_key_exists('pin', $this->fields) ? $this->fields['pin'] : '';
        $pass = array_key_exists('password', $this->fields) ? $this->fields['password'] : '';

        $allOK = Yii::app()->user->tools->save_file(self::$peer_file, $peer);
        if ($allOK)
            $allOK = Yii::app()->user->tools->save_file(self::$chat_file, $chat);
        if ($allOK)
            $allOK = Yii::app()->user->tools->save_file(self::$conf_file, $xml->saveXML());
        if ($allOK)
            $allOK = Yii::app()->user->tools->save_file(self::$pin_file, $pin, true, null, false);
        if ($allOK)
            $allOK = Yii::app()->user->tools->save_file(self::$pass_file, $pass, true, null, false);
        
        if ($allOK) {
            Yii::app()->user->tools->addReason( "modem configuration change" );
        }            
		return $allOK;
	}

	function validate() {
		$isOK=true;
        $n_modem_cmds = 0;

        // check for valid string (only ASCII non-control characters)
        foreach ($this->fields as $name => $field) {
            if (strlen($field) != 0 && preg_match("/[^\x20-\x7E]/", $field)) {
                $this->addError(self::$form_prefix .$field, htmlspecialchars($field) .
                    " is not a valid " . htmlspecialchars(self::getFieldDesc($name)) .
                    ". Configuration not saved.");
                $isOK=false;
            }
        }

        // check that dial string has no ';' except as last character
        if (array_key_exists('dial', $this->fields)) {
            $n_modem_cmds++;
            $dial = $this->fields['dial'];
            $p = strpos($dial, ';');
            if ($p !== FALSE && ($p != strlen($dial)-1 || strlen($dial) == 1)) {
                $this->addError(self::$form_prefix .'dial', "The " . htmlspecialchars(self::getFieldDesc('dial')) .
                                        " can only have a ';' at the end. "."Configuration not saved.");
                $isOK = false;
            }
        }

        // check that PDP type is present in simple mode
        if ($this->simple && !array_key_exists('pdptype', $this->fields)) {
            $this->addError(self::$form_prefix ."pdptype",  "The " . htmlspecialchars(self::getFieldDesc('pdptype')) .
                                        " is required in simple mode. "."Configuration not saved.");
            $isOK = false;
        }

        // check that PDP type, if any, is valid
        if (array_key_exists('pdptype', $this->fields)) {
            $n_modem_cmds++; // a PDP type generates a modem command
            $pdptype = $this->fields['pdptype'];
            // Other valid types would be IPV6 and IPV4V6 but we have not
            // verified IPv6 support, so we do not accept them yet
            if ($pdptype != 'IP' && $pdptype != 'PPP') {
                $this->addError(self::$form_prefix ."pdptype",  "The " . htmlspecialchars(self::getFieldDesc('pdptype')) .
                                            " must be IP or PPP. "."Configuration not saved.");
                $isOK = false;
            }
        }

		// check that each init string starts with AT
        for ($i = 1; $i <= self::NUM_INIT; $i++) {
            $name = "init$i";
            if (array_key_exists($name, $this->fields)) {
                $n_modem_cmds++;
                $field = $this->fields[$name];
                if (substr($field, 0, 2) != 'AT' && substr($field, 0, 2) != 'at' ) {
                    $this->addError(self::$form_prefix .$name,  "Init strings must start with 'AT' or 'at'. " .
                                            "Configuration not saved.");
                    $isOK = false;
                }
            }
        }

        // check that there is at least one modem command (besides PIN)
        if ($n_modem_cmds <= 0) {
            $this->addError(self::$form_prefix .'dial', "At least one init string or dial number must be entered. ".
                                "Configuration not saved.");
            $isOK = false;
        }

        // check for valid authentication protocols
        if (array_key_exists('authproto', $this->fields)) {
            if ( ! in_array($this->fields['authproto'], self::$choice['authproto'])) {
                $this->addError(self::$form_prefix .'authproto',    "Invalid authentication protocol. " .
                                                "Configuration not saved.");
                    $isOK = false;
            }
        }
		return $isOK;
	}

	function clear() {
        $this->fields = self::$defaults;
        $this->changed = true;
        $this->simple = true;
	}
    
    function getStatus($refresh = false)
    {
        if ( $refresh || ! $this->status ) {
            if ( ! file_exists(self::$devnode) ||
                 ! file_exists(self::$status_dir . '/status') )
            {
                $this->status = 'NOT FOUND';
            } else {
                $this->status = trim(file_get_contents(self::$status_dir . '/status'));
            }
        }
        return $this->status;
    }

    function getConnectStatus($refresh = false)
    {
        if ( $refresh || ! $this->connect_status ) {
            if ( ! file_exists(self::$status_dir . '/connect-status') )
            {
                $this->connect_status = '';
            } else {
                $this->connect_status =
                    trim(file_get_contents(self::$status_dir . '/connect-status'));
            }
        }
        return $this->connect_status;
    }

    function resetPIN()
    {
        if ( $this->hasErrors() ) return false;
        if ( ! $this->isPINBlocked() ) {
            $this->addError(self::$form_prefix .'puk',  "Attempted to reset PIN when not blocked. " .
                                                        "Configuration not saved.");
            return false;
        }
        $puk = $this->getField('puk');
        $newpin = $this->getField('pin');
        if ( ! $puk ) {
            $this->addError(self::$form_prefix .'puk',  "The PUK is required to reset the PIN. " .
                                                        "Configuration not saved.");
            return false;
        }
        if ( ! file_exists(self::$devnodectl) ) {
            $this->addError(self::$form_prefix .'puk',  "No modem control device found to reset the PIN. " .
                                                        "Configuration not saved.");
            return false;
        }
        $cmd = self::$modem_ctl .
            " -p " . escapeshellarg($newpin) .
            " -P " . escapeshellarg($puk) .
            " " . self::$devnodectl;
        exec($cmd, $out, $retval);
        $status = $this->getStatus(true); // refreshes the status value
        if ($retval != 0) {
            $str = "Failed resetting the PIN";
            
            if ($status == 'PUK INCORRECT')
                $str .=" (PUK was incorrect)";
            $str .=". ";
            $this->addError(self::$form_prefix .'puk', $str . "Configuration not saved.");
            unset($this->fields['puk']); // so that we do not show previous PUK in form
            return false;
        }
        
        Yii::app()->user->tools->messages[] = "<h3>Modem PIN has been successfully reset</h3>";
        Yii::app()->user->tools->addReason( "modem PIN reset" );
        
        return true;
    }
    
}