#!/bin/bash
#
# License: Copyright 2011 SpinetiX S.A. This file is licensed
#          under the terms of the GNU General Public License version 2.
#          This program is licensed "as is" without any warranty of any
#          kind, whether express or implied.
#
# Copyright 1999-2003 MontaVista Software, Inc.
# Copyright 2002, 2003, 2004 Sony Corporation
# Copyright 2002, 2003, 2004 Matsushita Electric Industrial Co., Ltd.
#
### BEGIN INIT INFO
# Required-Start:
# Required-Stop:
# Should-Start:
# Should-Stop:
# Default-Start: S
# Default-Stop: 0 1 2 3 4 5 6
# Short-Description: Storage partitioning and formatting.
# Description: Storage partitioning and formatting.
### END INIT INFO
# chkconfig: S 20 0

# Init script information
INIT_NAME=config
DESC=""

# The file that permanently stores configuration changes so as to be
# robust against reboots while resetting.
BOOTCONFIG=/boot-config

# Minimum sector for partition start, we make it 4 MB to keep
# alignment with allocation zones on SD cards and such
MINPSTART=8192

# Minimum size for partition
MINPSIZE=524288

# The partition type to use
PTYPE=83

# function to load configuration according to hardware type
function load_config
{
    local tag sep val
    local hw
    while read tag sep val; do
	if [ "$tag" = "Hardware" ]; then
	    hw="$val"
	    break
	fi
    done < /proc/cpuinfo

    if [ -z "$hw" ]; then
	echo "ERROR: could not find hardware type" >&2
	return 1
    fi

    #
    # MAINDEV = whole block device
    # PARTPREFIX = prefix name to form partitions
    # PARTNUM = storage partition number
    # FWPARTNUM = firmware package partition number
    # PARTDEV = storage partition block device
    # FWPARTDEV = firmware package partition block device
    #
    case "$hw" in
	"Bonsai")
	    MAINDEV=/dev/mmcblk0
	    PARTPREFIX="$MAINDEV"p
	    PARTNUM=2
	    FWPARTNUM=4
	    ;;
	"Sakura")
	    MAINDEV=/dev/hda
	    PARTPREFIX="$MAINDEV"
	    PARTNUM=1
	    FWPARTNUM=4
	    ;;
	*)
	    echo "ERROR: unknown hardware type '$hw'" >&2
	    return 1
    esac

    PARTDEV="$PARTPREFIX""$PARTNUM"
    FWPARTDEV="$PARTPREFIX""$FWPARTNUM"

    return 0
}
function check_bdev() {
    local dev="$1"
    local n
    local r=1
    for n in 1 2 3 4 5 6; do
	if [ ! -b "$dev" ]; then
	    sleep 5
	else
	    sleep 1
	    if [ -b "$dev" ]; then
		r=0
		break
	    fi
	fi
    done
    return $r
}

function format_ext3() {
    local bdev="$1"
    if [ ! -b "$bdev" ]; then
	echo "ERROR: partition block device not present" >&2
	return 1
    fi
    mke2fs -q -j -O dir_index,^resize_inode -L sys-data "$bdev" && \
	tune2fs -c 0 -i 0 -r 9765 "$bdev" > /dev/null
    return
}

function kill_fs() {
    local bdev="$1"
    if [ ! -b "$bdev" ]; then
	echo "ERROR: partition block device not present" >&2
	return 1
    fi
    dd if=/dev/zero bs=4096 count=1 of="$bdev" 2>/dev/null
}

function check_fs_present() {
    # For now we just test that the filesystem is ext3
    [ "$(blkid -c /dev/null -w /dev/null -o value -s TYPE "$PARTDEV")" = ext3 ]
}

function check_partition() {
    sfdisk -d $MAINDEV | \
	awk -F '[ =,:]+' -v p=$PARTDEV -v minst=$MINPSTART \
	-v minsz=$MINPSIZE -v id=$PTYPE \
	'BEGIN { r = 1 } END { exit r } $1 == p { if ($3 >= minst && $5 >= minsz && $7 == id) {r = 0; exit} }'
}

function make_partition() {
    local nrblocks
    local pdata
    local pstart psize pend pid
    local minsect maxsect
    local start size
    local fwstart fwsize
    local i

    # Check that we can change the partition table
    if ! blockdev --rereadpt "$MAINDEV" 2>/dev/null; then
	echo "ERROR: internal storage device in use, cannot repartition" >&2
	return 1
    fi

    # Get the size of the device
    nrblocks="$(sfdisk -s "$MAINDEV")"
    if [ -z "$nrblocks" ]; then
	echo "ERROR: failed getting size of internal storage" >&2
	return 1
    fi

    # Get the partition data into pdata array for partitions 1 to 4,
    # for each partition we get the start sector, size (in sectors) and id
    # unassigned partitions get all zero fields
    pdata=($(sfdisk -d "$MAINDEV" | awk -F '[ =,:]+' -v d="$PARTPREFIX"  '$1 ~ "^"d"[1-4]" && $2 == "start" { print $3, $5, $7 }'))

    # Extract the start, size, end and id for each partition (one based arrays)
    for i in 1 2 3 4; do pstart[$i]=${pdata[$(((i-1)*3))]}; done
    for i in 1 2 3 4; do psize[$i]=${pdata[$(((i-1)*3+1))]}; done
    for i in 1 2 3 4; do pend[$i]=$(( ${pstart[$i]} + ${psize[$i]} )); done
    for i in 1 2 3 4; do pid[$i]=${pdata[$(((i-1)*3+2))]}; done

    # Get the overall start and end of partitions
    minsect=-1
    maxsect=0
    for i in 1 2 3 4; do
	[ "${psize[$i]}" -eq 0 ] && continue
	[ "$minsect" -eq -1 -o "$minsect" -gt "${pstart[$i]}" ] && minsect="${pstart[$i]}"
	[ "$maxsect" -lt "${pend[$i]}" ] && maxsect="${pend[$i]}"
    done

    # Fill in the main partition definition
    start="$MINPSTART"
    [ "$start" -lt "$minsect" ] && start="$minsect"
    [ "$maxsect" -eq 0 ] && maxsect=$(( nrblocks * 2 ))

    if [ "${pid[$FWPARTNUM]}" = "$PTYPE" ]; then
	# firmware package partition to preserve
	size=$(( pstart[$FWPARTNUM] - start ))
	fwstart="${pstart[$FWPARTNUM]}"
	fwsize="${psize[$FWPARTNUM]}"
    else
	# no partitions to preserve, just use till end of device
	size=$(( maxsect - start ))
	fwstart=0
	fwsize=0
    fi

    # Fill in all partition definitions
    for (( i = 1; i <= 4; i++ )) ; do
	if [ "$i" -eq "$PARTNUM" ]; then
	    # main partition
	    pdef[$i]="$start,$size,$PTYPE,*"
	elif [ "$i" -eq "$FWPARTNUM" -a "$fwsize" -ne 0 ]; then
	    # firmware package partition
	    pdef[$i]="$fwstart,$fwsize,$PTYPE"
	else
	    # unallocated partition
	    pdef[$i]=",0"
	fi
    done

    # Now create the partitions
    echo -e "${pdef[1]}\n${pdef[2]}\n${pdef[3]}\n${pdef[4]}\n" | \
	sfdisk -f -L -uS "$MAINDEV" > /dev/null

    # And finally wait until the device node appears
    if ! check_bdev "$PARTDEV"; then
	echo "ERROR: timed out waiting for block device $MAINDEV" >&2
	return 1
    fi

}

# Does the internal storage format, if the first argument is -f the format
# is forced, otherwise it is done only if not yet formatted
function do_format() {
    local isforced

    [ "$1" = "-f" ] && isforced=yes && shift

    # Skip this on NFS root systems, unless forced, as we can otherwise
    # break havoc on special configurations and such
    if [ -z "$isforced" ] && grep -q '\bnfsroot=' /proc/cmdline; then
	echo "NFS root system, skipping internal storage formatting"
	return 0
    fi

    if ! load_config; then
	echo "ERROR: failed to load internal storage configuration" >&2
	return 1
    fi

    if [ ! -b $MAINDEV ]; then
	echo "ERROR: internal storage device not present" >&2
	return 1
    fi

    if [ -n "$isforced" ]; then
	echo -n "Internal storage format forced: "
    elif ! check_fs_present; then
	echo -n "Internal storage needs format: "
    else
	return 0
    fi

    # we need to format, if not already there write the config order
    # to stable storage (atomically) or a power cut during the format
    # will leave us with a half initialized filesystem; if we are in
    # forced mode then the config order is already on stable storage

    if [ -z "$isforced" ]; then
	if [ -f "$BOOTCONFIG" ]; then
	    cat "$BOOTCONFIG" > "$BOOTCONFIG".new
	else
	    : > "$BOOTCONFIG".new
	fi
	[ $? -eq 0 ] &&	echo "format" >> "$BOOTCONFIG".new && \
	    mv -f "$BOOTCONFIG".new "$BOOTCONFIG" && sync
	if [ $? -ne 0 ]; then
	    echo "ERROR: failed to save format config order" >&2
	    return 1
	fi
    fi

    if ! check_partition; then
	# partition is not there or not right, attempt to make it
	echo -n "making partition... "
	make_partition
	if [ $? -ne 0 ]; then
	    echo "ERROR: main partition not present and failed creating it" >&2
	    return 1
	fi
    fi

    echo -n "formatting... "
    format_ext3 "$PARTDEV"
    if [ $? -ne 0 ]; then
	echo "ERROR: failed to format internal storage" >&2
	kill_fs "$PARTDEV"
	return 1
    fi

    # succeeded, remove the format config order atomically
    sed -e '/^format\($\|=\)/d' "$BOOTCONFIG" > "$BOOTCONFIG".new
    if [ $? -eq 0 ]; then
	if [ -s "$BOOTCONFIG".new ]; then
	    mv -f "$BOOTCONFIG".new "$BOOTCONFIG" && sync
	else
	    rm -f "$BOOTCONFIG" "$BOOTCONFIG".new && sync
	fi
    else
	rm -f "$BOOTCONFIG".new
	[ -n "" ] # just make a non-zero exit code to trigger message below
    fi
    if [ $? -ne 0 ]; then
	echo "ERROR: failed to remove format config order" >&2
    fi

    echo "done"
    return 0
}

case "$1" in
    start)
	if [ -f "$BOOTCONFIG" ] && grep -q '^format\($\|=\)' "$BOOTCONFIG"; then
	    do_format -f # do a forced format
	else
	    do_format # do a format if required only
	fi
	;;
    stop)
	;;
esac
