#!/bin/bash
#
# License: Copyright 2011 SpinetiX S.A. This file is licensed
#          under the terms of the GNU General Public License version 2.
#          This program is licensed "as is" without any warranty of any
#          kind, whether express or implied.
#
### BEGIN INIT INFO
# Required-Start:
# Required-Stop:
# Should-Start:
# Should-Stop:
# Default-Start: S
# Default-Stop: 0 1 2 3 4 5 6
# Short-Description: Do system specific filesystem changes and setup
# Description: Do system specific filesystem changes and setup
### END INIT INFO
# chkconfig: S 35 0

# Init script information
INIT_NAME=spxfschange.sh
DESC=""

# The destination root for the various /var bind mounts
DESTVAR="/srv/.var"

# The maximum number of tries for /etc/fstab locking, as this is a boot script
# there should not be much concurrency, so this can be ratehr short
MAX_LOCK_TRIES=7

# The directory where to find the tmpfile.d spec
TMPFILESCONFDIR=/etc/tmpfiles.d

# Source spxsysconf functions
. /etc/spinetix/init-functions

#
# /etc/fstab locking uses /etc/fstab.lock
#
# This also serves for us to wait for the filesystem on which /etc/fstab
# resides to be mounted read-write.
#
# WARNING: the lock file is also used from other scripts, so DO NOT CHANGE
# the lockfile path.

fstab_lock() {
	for (( i=1; i <= $MAX_LOCK_TRIES; i++ )) ; do
		dotlockfile -p -r 0 -l /etc/fstab.lock && return 0
		sleep 1
	done
	echo "ERROR: failed acquiring fstab lock"
	return 1
}

fstab_unlock() {
	dotlockfile -p -u /etc/fstab.lock
}

# Adds mount at the end of /etc/fstab. If an entry for the same mount
# point exists in /etc/fstab but it does not match the arguments it is
# replaced (at the same position)
# arg1 = device, bind source directory or "none" for device-less
# arg2 = mount point
# arg3 = type
# arg4 = options
# For a bind mount use "none" as arg3 and "bind" as arg4
addmount()
{
    local mdev="${1%/}"
    local mmpt="${2%/}"
    local mfstype="$3"
    local mopts="$4"
    local line=0
    local r

    # NOTE: although this is a boot script which runs very early there may be
    # concurrent mounts of USB media, so we need to do fstab locking
    fstab_lock || return

    # First check if it is already in fstab
    local dev mpt fstype opts dump pass junk
    while read dev mpt fstype opts dump pass junk; do
	line=$(( line + 1 ))

	[ "${dev:0:1}" = "#" ] && continue # comment line 
	[ "$mmpt" = "${mpt%/}" ] || continue # not the same mount point

	if [ "$mdev" = "${dev%/}" -a "$mfstype" = "$fstype" -a "$mopts" = "$opts" ]; then
	    # already in and with same definition, nothing to do
	    fstab_unlock || return
	    return 0
	fi

	# we need to change the mount point definition at the same place
	cp -a /etc/fstab /etc/fstab.new && \
	    sed -ne "$line"',$ !p' /etc/fstab > /etc/fstab.new && \
	    echo -e "$mdev\t$mmpt\t$mfstype\t$mopts\t0 0" >> /etc/fstab.new && \
	    sed -ne "1,$line"' !p' /etc/fstab >> /etc/fstab.new && \
	    mv /etc/fstab.new /etc/fstab
	r=$?
	[ -f /etc/fstab.new ] && rm -f /etc/fstab.new
	fstab_unlock || return
	return $r

    done < /etc/fstab

    # It was not found in /etc/fstab, so add it at end
    cp -a /etc/fstab /etc/fstab.new && \
	echo -e "$mdev\t$mmpt\t$mfstype\t$mopts\t0 0" >> /etc/fstab.new && \
	mv /etc/fstab.new /etc/fstab
    r=$?
    [ -f /etc/fstab.new ] && rm -f /etc/fstab.new
    fstab_unlock || return
    return $r
}

# Copies all content from arg1 to arg2, preserving all
# attributes. The destination (arg2) is cleaned up before copying
copympt()
{
    local from="$1"
    local to="$2"
    # If target exists clean it up completely
    if [ -e "$to" ]; then
	rm -rf "$to"
    fi
    # Make sure parent directory exists but not target
    mkdir -p "$to" && rmdir "$to" || return
    # Copy preserving all attributes
    cp -a "$from" "$to"
}

# Moves selected var dirs to tmpfs and $DESTVAR.
# Returns 0 if all dirs were processed successfully, non-zero otherwise.
movevardirs()
{
    local RET=0

    # First do the ones mounted as tmpfs

    for m in run lock; do
	src=/var/"$m"
	ismounted "$src" tmpfs && continue
	ismounted "$src" && umount "$src"
	if ! addmount none "$src" tmpfs defaults; then
	    echo "ERROR: failed adding tmpfs mount at '$src'" >&2
	    RET=1
	    continue
	fi
	sync # be sure everything is on stable storage before proceeding
	if ! mount "$src"; then
	    echo "ERROR: failed doing tmpfs mount at '$src'" >&2
	    RET=1
	fi
    done

    # Now do the bind mounts

    for m in log cache tmp; do
	src=/var/"$m"
	dst="$DESTVAR"/"$m"
	ismounted "$src" && continue
	if ! copympt "$src" "$dst"; then
	    echo "ERROR: failed copying '$src' to '$dst'" >&2
	    rm -rf -- "$dst"
	    RET=1
	    continue
	fi
	if ! addmount "$dst" "$src" none bind; then
	    echo "ERROR: failed adding bind mount for '$dst' at '$src'" >&2
	    rm -rf -- "$dst"
	    RET=1
	    continue
	fi
	sync # be sure everything is on stable storage before proceeding
	if ! cleanvarpath "$src"; then
	    echo "WARNING: could not clean source path '$src'" >&2
	    RET=1
	fi
	if ! mount "$src"; then
	    echo "ERROR: failed doing bind mount for '$dst' at '$src'" >&2
	    RET=1
	fi
    done

    return $RET
}

set_path_attrs() {
    local path="$1"
    local mode="$2"
    local uid="$3"
    local gid="$4"

    [ -n "$mode" -a "$mode" != "-" ] && chmod -- "$mode" "$path"
    [ -n "$uid" -a "$uid" != "-" ] && chown -- "$uid" "$path"
    [ -n "$gid" -a "$gid" != "-" ] && chgrp -- "$gid" "$path"
}

# Parses the tmpfiles specification (compatible with systemd) given as the
# first arg and performs all the required actions
do_tmpfiles_conf() {
    local conf="$1"
    local type path mode uid gid junk
    while read type path mode uid gid junk; do
	[ -z "$type" -o "$type" = "#" ] && continue # empty line or comment
	if [ -z "$path" ]; then
	    echo "empty path in tmpfiles spec" >&2
	    continue
	fi
	if [ "${path:0:1}" != "/" ]; then
	    echo "relative path ('$path') in tmpfiles spec not allowed" >&2
	    continue
	fi
	case "$type" in
	    f) # create a file if it does not exist yet
		if [ ! -e "$path" ]; then
		    : > "$path"
		    set_path_attrs "$path" "$mode" "$uid" "$gid"
		fi
		;;
	    F) # create or truncate a file
		: > "$path"
		set_path_attrs "$path" "$mode" "$uid" "$gid"
		;;
	    d) # create a directory if it does not exist yet
		if [ ! -e "$path" ]; then
		    mkdir -- "$path"
		    set_path_attrs "$path" "$mode" "$uid" "$gid"
		fi
		;;
	    D) # create or empty a directory
		if [ ! -e "$path" ]; then
		    mkdir -- "$path"
		    set_path_attrs "$path" "$mode" "$uid" "$gid"
		else
		    find -- "$path" -mindepth 1 -print0 | xargs -0r rm -rf --
		fi		    
		;;
	    x) # ignore a path
		;;
	    r) # remove a path
		if [ -d "$path" ]; then
		    rmdir -- "$path"
		elif [ -e "$path" ]; then
		    rm -f -- "$path"
		fi
		;;
	    R) # recursively remove a path
		[ -e "$path" ] && rm -rf -- "$path"
		;;
	    *) # unkown specification
		echo "invalid tmpfiles spec '$type'" >&2
		;;
	esac
    done < "$conf"
}

# Do all the tmpfiles specifications found
do_tmpfiles_all() {
    for f in "$TMPFILESCONFDIR"/*.conf ; do
	[ -f "$f" ] && do_tmpfiles_conf "$f"
    done
}

#
# Main
#

case "$1" in
    start)
	echo -n "Adapting filesystems"

	movevardirs

	echo -n ", "

	echo -n "creating volatile paths"

	do_tmpfiles_all

	# This file better be a symlink to an always writable directory
	# or else ifup/ifdown will not work
	if [ ! -L /etc/network/ifstate ]; then
	    ln -sf /var/run/network/ifstate /etc/network/ifstate
	fi

	echo "."

	;;
    stop)
	;;
esac
