<?php

/**
 * This is the model class for table "Resource".
 *
 * The followings are the available columns in table 'Resource':
 * @property integer $id
 * @property string $type
 * @property string $keywordsStr
 */
class Resource extends AssetsActiveRecord
{
    private $_backup = null;
    private $wasNewRecord = false;
    var $delayPreview = false;
	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Resource';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('type, name', 'required'),
			array('type', 'in', 'range'=>array('media', 'playlist', 'playout', 'template', 'apps', 'schedule', 'external', 'project' ) ),
            array('name', 'length', 'max'=>128),
            array('modified', 'default', 'value'=> date("Y-m-d H:i:s", time()), 'setOnEmpty'=>false ),
            array('keywords', 'safe'),
            array('keywordsStr', 'length', 'max'=>256),     
            array('id', 'numerical', 'integerOnly'=>true, 'on'=>'update' ),
            array('uuid, version', 'safe', 'on'=>'insert' ),
			// The following rule is used by search().
			// @todo Please remove those attributes that should not be searched.
			array('id, type, name', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
        
        $relations = array(
            'parents' => array(self::MANY_MANY, 'Resource',
                'Owner(childId, parentId)'),
            'children' => array(self::MANY_MANY, 'Resource',
                'Owner(parentId, childId)'),
		);
        return $relations;
	}
    
    public function behaviors(){
        return array( 
            'CAdvancedArBehavior' => array( 'class' => 'application.components.CAdvancedArBehavior' ),            
        );
    }

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'id',
			'type' => 'type',
            'name' => 'name',
            'keywordsStr' => 'keywords',
		);
	}
    public function getMainResource( $forDelete = false ) {
        
        if ( $this->type == 'media' )
            return Media::model()->findByPk($this->id);
        if ( $forDelete ) 
            return null; 
        // for delete, the other resource do not have any specific sql data associated
        // so there is no need to fetch them
        if ( $this->type == 'playlist' )
            return Playlist::model()->findByPk($this->id);
        if ( $this->type == 'playout' )
            return Playout::model()->findByPk($this->id);
        if ( $this->type == 'template' )
            return Template::model()->findByPk($this->id);
        if ( $this->type == 'apps' )
            return Apps::model()->findByPk($this->id);
        if ( $this->type == 'schedule' )
            return Schedule::model()->findByPk($this->id);
        if ( $this->type == 'external' )
            return External::model()->findByPk($this->id);
        
        return null;        
    }
    static function newModel( $type )
	{
        switch( $type )
        {
            case 'media':
                return new Media;
            case 'playlist':
                return new Playlist;
            case 'playout':
                return new Playout;
            case 'template':                
                return new Template;
            case 'apps':
                return new Apps;
            case 'external':
                return new External;
            case 'project':
                return new Project;
            default:
                return null;
        }
    }
    static function loadModel( $type, $id, $loadResource=true )
	{
        $model = null;
        switch( $type )
        {
            case 'media':
                $model = Media::model()->with('resource')->findByPk($id);
                break;
            case 'playlist':
                $model = Playlist::model()->findByPk($id);
                break;            
            case 'playout':
                $model = Playout::model()->findByPk($id);
                break;
            case 'template':
                $model = Template::model()->findByPk($id);
                break;
            case 'apps':
                $model = Apps::model()->findByPk($id);
                break;            
            case 'external':
                $model = External::model()->findByPk($id);
                break; 
            case 'schedule':
                $model = Schedule::model()->findByPk($id);
                break;
            case 'project':
                $model = Project::model()->findByPk($id);
                break;
        }
        if( $loadResource && $model===null ){
            $model = Resource::model()->findByPk($id);
        }		
		return $model;
	}
    
    static function loadAll( $type, $criteria = "" ){
        $models = null;
        switch( $type )
        {
            case 'resource':
                $models = Resource::model()->findAll( $criteria );
                break;           
            case 'media':
                $models = Media::model()->with('resource')->findAll( $criteria );
                break;              
            case 'playlist':
                $models = Playlist::model()->findAll( $criteria );
                break;     
            case 'playout':
                $models = Playout::model()->findAll( $criteria );
                break; 
            case 'template':
                $models = Template::model()->findAll( $criteria );
                break; 
            case 'apps':
                $models = Apps::model()->findAll( $criteria );
                break;             
            case 'external':
                $models = External::model()->findAll( $criteria );
                break; 
            case 'schedule':
                $models = Schedule::model()->findAll( $criteria );
                break;
            case 'project':
                $models = Project::model()->findAll( $criteria );
                break;
        }
        return $models;
    }
    public function removeChild( $childId, $delayPreview = false ) {
        // media cannot have children so this function should never been called 
        // for media
        $model = $this->getMainResource();         
        if ( $model ) {
            if (!$model->removeChild( $childId, $delayPreview ) ){
                $this->addErrors( $model->getErrors() );
                return false;
            }
            $this->_backup = $model;
        }
        return true;
    }
    static function resourceLink( $id, $type) {
        
        if ( $id === null )
            return false;
        return $type . sprintf( "/%u/", $id );        
    }
    static function saveAtomic($filepath, $content){        
        if ( !@file_put_contents($filepath . ".tmp", $content) )
            return false;
        if ( !file_exists($filepath) ){
            touch( $filepath );
        }
        if ( !@rename($filepath . ".tmp", $filepath) ) {
            @unlink( $filepath );
            if ( !@rename($filepath . ".tmp", $filepath) ) {
                @unlink($filepath . ".tmp");
                return false;
            }
        }
        return true;
    }
    public function getLink( ) {
        return Resource::resourceLink( $this->id, $this->type );        
    }    
    public function getFolder( ) {
        $folder = $this->getLink( );        
        if ( !$folder )
            return $folder;
        return GlobalSettings::$fileBase. $folder;
    }
    public function getManage() {
        $folder = $this->getLink( );        
        if ( !$folder )
            return $folder;
        return GlobalSettings::$manageBase . $folder;
    }
    public function getUri() {
        $folder = $this->getLink( );        
        if ( !$folder )
            return $folder;
        return GlobalSettings::$uriBase . $folder;
    }
    public function getHref() {
        $folder = $this->getLink( );        
        if ( !$folder )
            return $folder;
        return GlobalSettings::$hrefBase . $folder;
    }
    public function getPreviews() {
        $previews = array();
        $dir = $this->getFolder( ) . "previews/";
        $uriBase = $this->getUri() . "previews/";
        if (!file_exists( $dir ))
            return $previews;
        
        $objects = scandir( $dir );
        foreach ($objects as $filename) {
            if ( substr( $filename,0, 7 )=="preview" ){
                $name = explode( ".", $filename );
                $val = explode( "_", $name[0] );
                $preview = array( 'uri' => $uriBase.$filename ."?_=".strtotime($this->modified) );
                if ( count($val)>1 ) 
                    $preview['size'] = $val[1];
                if ( count($val)==4 ) {
                    $preview['time'] = $val[2];
                    $preview['type'] = $val[3];
                }else if ( count($val)==3 ) {
                    $preview['type'] = $val[2];
                    $preview['time'] = 0;
                } else {
                   $preview['type'] = 'generated';
                   $preview['time'] = 0;
                }
                $previews[] = $preview;
            }
        }
        return $previews;
    }
    public function getResource() {        
        return $this;
    }
    public function getDur() {        
        return 0;
    }
    
    public function getKeywords() {
        if ( empty( $this->keywordsStr ) )
            return array();
        return explode(" ", $this->keywordsStr );
    }
    public function setKeywords( $val ) {
        return $this->keywordsStr = implode(" ", $val );
    }
    public function getListDesc( $preview=true ) {
        $description = array(
            'id' => $this->id,
            'name' => $this->name,            
            'type' => $this->type,
            'manage' => $this->manage,
            'modified' => $this->modified,
            'keywords' => $this->keywords
        );
        if (  $preview ) {
            $previews = $this->previews;
            if ( count( $previews ) )
                $description['previews'] = $previews;
        }
        return $description; 
    }
    public function getDesc( $parents = true, $short = false ) {
        $description = $this->getListDesc( $parents );
        if ( $parents ) {
            foreach ( $this->parents as $parent ){
                $description['parents'][] = $parent->getListDesc( false );
            }
        }
        return $description; 
    }

    public function previewsImage( $dimension, $path, $outpath, $file ) {
        
        $exe = "cd /srv/raperca && previewer";
        if( PHP_OS == "WINNT" ){
            $exe = "C:\\SpinetiX\\sources\\genesis\\trunk\\device\\Release\\previewer.exe";
            if ( !file_exists( $exe ) ){
                $exe = "\"C:\\Program Files (x86)\\SpinetiX\\Elementi\\Elementi\\bin\\previewer.exe\"";
            }
        }
        $cmd = $exe . " -f png -s " . escapeshellarg($dimension) . " -i " . escapeshellarg($path) . " -o " . escapeshellarg($outpath) . " " . escapeshellarg($file) . " 2>&1";
        Yii::log($cmd, 'info', 'spx.preview');
        
        $output = array();
        $ret = 10;
        exec( $cmd, $output, $ret );
        $results = json_decode(implode( "\n", $output ), true);
        if ( $ret == 0 ) {            
            Yii::log(print_r($results, true), 'info', 'spx.preview');
            foreach( $results as $result ){
                if ( $result["success"] ){
                    $dim = $result['previewWidth'] ."x". $result["previewHeight"];
                    //rename($result.dst, $outpath."preview_$dim.png");
                    rename($outpath.".preview.".$file.".png", $outpath."preview_$dim.png");
                } else {
                    $this->addError('file', $result['error'] );
                    return false;
                }
            }
        } else {
            if ( PHP_OS == "WINNT" ){
                list($width, $height, $type) = getimagesize( $path.$file );
                if ( image_type_to_mime_type($type)=="image/jpeg"  ){
                    $img = imagecreatefromjpeg($path.$file);
                } else if ( image_type_to_mime_type($type)=="image/png"  ){
                    $img = imagecreatefrompng($path.$file);
                } else 
                    $img = null;
                if ( $img ) {
                    $size = explode( "x", $dimension );
                    $thumb = imagecreatetruecolor($size[0], $size[1]);
                    // Resize
                    imagecopyresized($thumb, $img, 0, 0, 0, 0, $size[0], $size[1], $width, $height);

                    imagepng( $thumb, $outpath."preview_$dimension.png");
                } else {
                    return false;
                }
                return true;
            }
            $this->addError('file', 'An error occured while resizing the image');
            return false;
        }
        
        return true;
    }
    public function deleteRecover( ){
        if ( $this->version ){
            $this->version = "0.0";
            if ( !$this->save() ) {
                throw new Exception( "recover failed" );
            }
        } else {
            if ( !$this->delete() ) {
                throw new Exception( "recover failed" );
            }
        }
    }
    public function copyNew( $src, $dst ){
        if ( $this->isNewRecord || !file_exists($dst) || filesize($dst) == 0 || filemtime($src) > filemtime($dst) ){
             return @copy( $src, $dst );
        }
        return true;
    }
    public function computeEtag( $filename = null ) {
        if ( !$filename ){
            return "";
        }
        if (file_exists($filename)){
            $info = stat( $filename );
            return sprintf("%08X-%08X",$info['ino'], $info['size']);
        } else {
            return "";
        }
    }    
    public function recover( $file = null ){
        $this->delayPreview = true;
        
        $folder = $this->getFolder();
        if ( $folder && !file_exists($folder) ) {
            $this->deleteRecover();
            return true;
        }
        if ( $file ){
            if ( !file_exists($file) || filesize($file) == 0 ){
                $this->deleteRecover();
                return true;
            }
            if ( file_exists( $file .".tmp" ) ) {
                // something was not saved as expected
                @unlink( $file .".tmp" );
                GlobalSettings::rrmdir( $folder . "previews/" );
            } 
            $etag = $this->computeEtag( $file );
            if ( $this->etag !== null && $this->etag != $etag ){
                GlobalSettings::rrmdir( $folder . "previews/" );
                $this->etag = $etag;
                $this->save();
            }
        }
        return false;
    }
    public function afterSave(){
        $this->wasNewRecord = $this->isNewRecord;
        if ( $this->isNewRecord ){
            $folder = $this->getFolder();
            if ( $folder && !file_exists($folder) ) {
                if ( !@mkdir( $folder ) ){
                    return false;
                }
            }
        }
        parent::afterSave();
    }
    public function beforeDelete() {
        /*
        $target = $this->getResource();
        if ( $target!==null ) {
            if ( !$target->delete() )
                return false;
        }         
         */
        
        foreach ($this->parents as $parent) {            
            if( !$parent->removeChild( $this->id, $this->delayPreview ) ){
                foreach ( $parent->getErrors() as $errors ) {
                    $this->addErrors( array( 'parent-'.$parent->id => $errors ) );
                }
                return false;
            }
        }
        return parent::beforeDelete();
    }
    
    public function afterDelete(){
        
        parent::afterDelete();
        
        $folder = $this->getFolder();
        if ( $folder ) {
            if ( !GlobalSettings::rrmdir( $folder ) )
                throw new Exception( "folder removal failed" );
        }
        
    }
    
    public function rollback( )            
    {
        if ( $this->isNewRecord ){
            $folder = $this->getFolder();
            if ( $folder ) {
                GlobalSettings::rrmdir( $folder );
            }            
        } else if ( $this->wasNewRecord ){
            $this->delete();
        } else {
            if ( $this->_backup ){
                $this->_backup->rollback();
            }
        }
    }
    
	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('id',$this->id);
		$criteria->compare('type',$this->type, true);
        $criteria->compare('name',$this->name, true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return Resource the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}
}
