/*!
 * jSignage.Multiscreen
 * http://www.spinetix.com/
 * Copyright SpinetiX S.A.
 * Released under the GPL Version 2 license.
 *
 * $Date: 2024-09-30 10:53:48 +0000 (Mon, 30 Sep 2024) $
 * $Revision: 39117 $
 */
 
( function () {
    var version = new String( "1.1.0" );
    version.major = 1;
    version.minor = 1;
    version.revision = 0;

    var reLengthWithUnit = /^\s*([+-]?(?:[0-9]+(?:\.[0-9]*)?)|(?:\.[0-9]+))\s*([a-zA-Z."]*)\s*$/;
    var reXbyY = /^\s*([1-9][0-9]*)\s*x\s*([1-9][0-9]*)\s*$/;
    var reCustom = /^\s*(-?[0-9]+(?:\.[0-9]*)?)\s+(-?[0-9]+(?:\.[0-9]*)?)\s+([0-9]+(?:\.[0-9]*)?)\s+([0-9]+(?:\.[0-9]*)?)\s*$/;
    var reXMLId = /^\s*([A-Z_a-z\xC0-\xD6\xD8-\xF6\xF8-\u02FF\u0370-\u037D\u037F-\u1FFF\u200C-\u200D\u2070-\u218F\u2C00-\u2FEF\u3001-\uFFFF][.0-9A-Z_a-z\xB7\xC0-\xD6\xD8-\xF6\xF8-\u037D\u037F-\u1FFF\u200C-\u200D\u203D-\u2040\u2070-\u218F\u2C00-\u2FEF\u3001-\uFFFF-]*)\s*$/;

function isVertical( orientation ) {
    return orientation === "rotateRight" || orientation === "rotateLeft";
}

jSignage.Multiscreen = {
    
    version: version,
    
    isRectVisible: function( ctm, bbox, x, y, width, height, weak ) {
        var x1 = ctm.a * x + ctm.c * y + ctm.e;
        var y1 = ctm.b * x + ctm.d * y + ctm.f;
        var x2 = ctm.a * ( x + width ) + ctm.c * y + ctm.e;
        var y2 = ctm.b * ( x + width ) + ctm.d * y + ctm.f;
        var x3 = ctm.a * ( x + width ) + ctm.c * ( y + height ) + ctm.e;
        var y3 = ctm.b * ( x + width ) + ctm.d * ( y + height ) + ctm.f;
        var x4 = ctm.a * x + ctm.c * ( y + height ) + ctm.e;
        var y4 = ctm.b * x + ctm.d * ( y + height ) + ctm.f;
        var left = Math.min( x1, x2, x3, x4 );
        var top = Math.min( y1, y2, y3, y4 );
        var right = Math.max( x1, x2, x3, x4 );
        var bottom = Math.max( y1, y2, y3, y4 );
        if ( weak )
            return left < bbox.x2 - 8 && bbox.x1 + 8 < right && top < bbox.y2 - 8 && bbox.y1 + 8 < bottom;
        else
            return left < bbox.x2 && bbox.x1 < right && top < bbox.y2 && bbox.y1 < bottom;
    },

    megaImage: function ( args ) {
        var layerG2, layerG3;
        var layer = jSignage.customLayer( 'megaImage', args, null, function ( width, height ) {
            layerG2 = this;
        });

        layer.endEvent( function () {
            layerG3.textContent = '';
            layer.tiles = null;
        } );

        layer.beginEvent( function () {
            layerG3 = jSignage._createElement( 'g' );
            layerG2.appendChild( layerG3 );
            layer.tiles = [];
            layer.scale = 0;
            realize();
        } );

        layer.ontransform = function () {
            if ( layer.tiles!==null )
                realize();
        }

        var idx = 1;
        layer.tiles = null;

        function realize() {
            var tiles = [];
            var ctm = jSignage.getCTM( layerG2 );
            var scaleX = Math.sqrt( ctm.a * ctm.a + ctm.c * ctm.c );
            var scaleY = Math.sqrt( ctm.b * ctm.b + ctm.d * ctm.d );
            var scale = Math.max( scaleX, scaleY );
            if ( args.width * scale >= 3 && args.height * scale >= 3 ) {
                scale = 1 / scale;
                var level = null, cnt = 0;
                for ( var i = 0; i < args.levels.length; i++ ) {
                    level = args.levels[i];
                    if ( scale * 0.85 < args.levels[i].scale )
                        break;
                }
                if ( level ) {
                    if ( layer.scale != level.scale ) {
                        layerG3.setAttribute( 'transform', 'scale(' + level.scale + ')' );
                        layer.scale = level.scale;
                    }
                    if ( level.tiles ) {
                        ctm = jSignage.getCTM( layerG3 );
                        var bbox = {
                            x1: window.screen.availX || 0,
                            y1: window.screen.availY || 0,
                            x2: ( window.screen.availX || 0 ) + window.screen.availWidth,
                            y2: ( window.screen.availY || 0 ) + window.screen.availHeight
                        };
                        function addVisibleTiles( list ) {
                            for ( var i = 0; i < list.length; i++ ) {
                                var tile = list[i];
                                ++cnt;
                                if ( jSignage.Multiscreen.isRectVisible( ctm, bbox, tile.x, tile.y, tile.width, tile.height ) ) {
                                    if ( tile.tiles )
                                        addVisibleTiles( tile.tiles );
                                    else
                                        tiles.push( tile );
                                }
                            }
                        }
                        addVisibleTiles( level.tiles );
                    } else {
                        tiles.push( level );
                    }
                }
            }

            if ( layer.tiles.length == tiles.length ) {
                for ( var i = 0; i < tiles.length; i++ )
                    if ( layer.tiles[i] !== tiles[i] )
                        break;
                if ( i == tiles.length )
                    return;
            }

            {
                var zlst = [];
                for ( var i = 0; i < tiles.length; i++ )
                    zlst.push( tiles[i].z );
                //alert( 'new tile selection after ' + cnt + ' checks: [' + zlst.join( ',' ) + ']' );
            }

            var i = 0 , j = 0, img = layerG3.firstElementChild;
            while ( img && img.localName != 'image' )
                img = img.nextElementSibling;
            while ( i < layer.tiles.length || j < tiles.length ) {
                var oldTile = i < layer.tiles.length ? layer.tiles[i] : null;
                var newTile = j < tiles.length ? tiles[j] : null;
                if ( oldTile && ( !newTile || oldTile.z < newTile.z ) ) {
                    var next = img.nextElementSibling;
                    layerG3.removeChild( img );
                    img = next;
                    ++i;
                    // alert( '  removed ' + oldTile.z );
                } else if ( oldTile === newTile ) {
                    ++i;
                    ++j;
                    img = img.nextElementSibling;
                } else if ( newTile ) {
                    ++j;
                    layerG3.insertBefore( jSignage._createElement( 'image', {
                        x: newTile.x || 0,
                        y: newTile.y || 0,
                        width: newTile.width,
                        height: newTile.height,
                        href: newTile.href
                    } ), img );
                    // alert( '  added ' + newTile.z );
                }
            }

            layer.tiles = tiles;
        }

        return layer;
    },

    megaVideo: function ( args ) {
        var layerG2=null, video=null;
        var layer = jSignage.customLayer( 'megaVideo', args, null, function ( width, height ) {
            layerG2 = this;
        } );

        layer.endEvent( function () {
            if ( video ) {
                layerG2.removeChild( video );
                video = null;
            }
            layer.tile = -1;
        } );

        layer.beginEvent( function () {
            layer.tile = -1;
            realize();
        } );

        layer.ontransform = function () {
            if ( layer.tile >= 0 )
                realize();
        }

        layer.tile = -1;

        function realize() {
            var ctm = jSignage.getCTM( layerG2 );
            var bbox = {
                x1: window.screen.availX || 0,
                y1: window.screen.availY || 0,
                x2: ( window.screen.availX || 0 ) + window.screen.availWidth,
                y2: ( window.screen.availY || 0 ) + window.screen.availHeight
            };
            var v = -1, nv = 0;

            if ( window.MULTI_SCREEN_ID != 'fullscreen' && jSignage.isArray( args.tiles ) ) {
                for ( var i = 0; i < args.tiles.length; i++ ) {
                    var tile = args.tiles[i];
                    if ( jSignage.Multiscreen.isRectVisible( ctm, bbox, tile.x, tile.y, tile.width, tile.height, true ) ) {
                        if ( v == -1 ) {
                            v = i + 1;
                            nv = 1;
                        } else {
                            nv++;
                        }
                    }
                }
                if ( nv > 1 ) {
                    if ( nv != args.tiles.length && layer.tile==-1 )
                        alert( 'Video tiles are not aligned on screen boundaries, video resolution will not be optimized' );
                    v = -1;
                    nv = 0;
                }
            }

            if ( v == -1 && jSignage.Multiscreen.isRectVisible( ctm, bbox, 0, 0, args.width, args.height ) )
                v = 0;

            //alert( 'new video tile selection: ' + v );

            if ( layer.tile != v ) {
                if ( layer.tile >= 0 ) {
                    layerG2.removeChild( video );
                    video = null;
                }
                video = jSignage._createElement( 'video', {
                    begin: 0,
                    dur: 'media',
                    fill: 'freeze',
                    x: v == 0 ? 0 : args.tiles[v - 1].x,
                    y: v == 0 ? 0 : args.tiles[v - 1].y,
                    width: v == 0 ? args.width : args.tiles[v - 1].width,
                    height: v == 0 ? args.height : args.tiles[v - 1].height,
                    href: v == 0 ? args.href : args.tiles[v - 1].href
                } );
                layerG2.appendChild( video );
            }

            layer.tile = v;

        }

        return layer;
    },

    /**
    * Parses a length from string into a numeric value or NaN in case of errors.
    * The default unit is assumed to be cm.
    * @param {string} raw The length. 
    * @return {number} The numeric value corresponding to the input length. 
    */
    parseLength: function( raw ) {
        var m = reLengthWithUnit.exec( raw );
        if ( !m )
            return NaN;
        var len = parseFloat( m[1] );
        if ( !isFinite( len ) )
            return NaN;
        var unit = m[2].toLowerCase();
        if ( unit == "in" || unit == "in." || unit == "inch" || unit == "inches" || unit == "\"" )
            len *= 2.54;
        else if ( unit == "mm" )
            len *= 0.1;
        else if ( unit == "m" )
            len *= 100;
        else if ( unit != "cm" && unit != "" )
            len = NaN;
        return len;
    },

    /**
     * Parses an aspect ratio string into a numeric value or NaN in case of errors.
     * @param {string} raw The aspect ratio. 
     * @return {number} The division result of the two values included in the aspect ratio string.
     */
    parseAspectRatio : function( raw ) {
        var ar = raw.split( ':' );
        if ( ar.length == 1 )
            ar.push( 1 );
        if ( ar.length == 2 ) {
            var numerator = parseFloat( ar[0] );
            var denominator = parseFloat( ar[1] );
            if ( isFinite( numerator ) && numerator > 0 && isFinite( denominator ) && denominator > 0 )
                return numerator / denominator;
        }
        return NaN;
    },

    /**
     * Creates a rectangle filled with a linear black gradient.
     * @param {boolean} vertical Whether the gradient is vertical.
     * @param {boolean} reverse Whether the gradient is reversed.
     * @param {number} start The start value for the gradient.
     * @param {number} end The end value for the gradient.
     * @param {Array.<number>} viewbox Array with the view box coordinates.
     * @return {Object} A jSignage rectangle object. 
     */
    createBorder: function ( vertical, reverse, x1, y1, x2, y2 ) {
        var grad = {
            id: $.guuid(),
            stops: [
                { offset: 0, opacity: 0, color: '#000000' },
                { offset: 1, opacity: 1, color: '#000000' }
            ]
        };
        if ( vertical ) {
            grad.x1 = grad.x2 = x1;
            if ( reverse ) {
                grad.y1 = y2;
                grad.y2 = y1;
            } else {
                grad.y1 = y1;
                grad.y2 = y2;
            }
        } else {
            grad.y1 = grad.y2 = y1;
            if ( reverse ) {
                grad.x1 = x2;
                grad.x2 = x1;
            } else {
                grad.x1 = x1;
                grad.x2 = x2;
            }
        }
        return jSignage.rect( {
            x: x1,
            y: y1,
            width: x2 - x1,
            height: y2 - y1,
            fill: "url(#" + grad.id + ")"
        } ).add( $.linearGradient( grad ) );
    },

    setupArray: function ( config ) {
        var screenHeight = parseInt( config.height );
        if ( !isFinite( screenHeight ) || screenHeight <= 0 ) {
            alert( "Invalid screen height, must be a positive number of lines" );
            return null;
        }

        var mine = null, others = [];

        var diagonal = jSignage.Multiscreen.parseLength( config.diagonal );
        if ( !isFinite( diagonal ) || diagonal <= 0 ) {
            alert( "Invalid diagonal, must be a positive number with units" );
            return null;
        }

        var frameH = jSignage.Multiscreen.parseLength( config.frameH );
        if ( !isFinite( frameH ) ) {
            alert( "Invalid horizontal frame width, must be a number with units or 0 if frame width is not to be accounted for" );
            return null;
        }

        var frameV = jSignage.Multiscreen.parseLength( config.frameV );
        if ( !isFinite( frameV ) ) {
            alert( "Invalid vertical frame width, must be a number with units or 0 if frame width is not to be accounted for" );
            return null;
        }

        var aspect = jSignage.Multiscreen.parseAspectRatio( config.aspect );
        if ( !isFinite( aspect ) ) {
            alert( "Invalid aspect ratio, must be a pair of positive numbers separated by a colon" );
            return null;
        }

        var idToPlayers = {};

        if ( config.screens && config.multiOutput !== true )
            for ( var i = 0; i < config.screens.length; i++ )
                idToPlayers[config.screens[i].id] = config.screens[i].players;

        var outputs = [];
        if ( jSignage.isArray( config.outputs ) )
            outputs = config.outputs;

        var screenWidth = Math.round( screenHeight * aspect );
        var pixelSize = screenHeight / ( diagonal / Math.sqrt( 1 + aspect * aspect ) );
        var framePixelsHeight = Math.round( 2 * frameH * pixelSize );
        var framePixelsWidth = Math.round( 2 * frameV * pixelSize );

        if ( isVertical( config.orientation ) ) {
            var tmp = screenWidth;
            screenWidth = screenHeight;
            screenHeight = tmp;
            tmp = framePixelsWidth;
            framePixelsWidth = framePixelsHeight;
            framePixelsHeight = tmp;
        }

        var custom = {
            multiOutput: config.multiOutput === undefined ? false : config.multiOutput,
            width: screenWidth * config.columns + framePixelsWidth * ( config.columns - 1 ),
            height: screenHeight * config.rows + framePixelsHeight * ( config.rows - 1 ),
            screens: [],
            blending: config.blending || false
        };
        for ( var row = 0; row < config.rows; row++ ) {
            for ( var col = 0; col < config.columns; col++ ) {
                var id, orientationOverride = false;
                if ( config.multiOutput === true ) {
                    var idx = row * config.columns + col;
                    if ( idx < outputs.length ) {
                        id = outputs[ idx ];
                        var idxColon = id.indexOf( ":" );
                        if ( idxColon > 0 ) {
                            orientationOverride = id.substring( idxColon + 1 );
                            id = id.substring( 0, idxColon );
                        }
                    } else if ( outputs.length === 0 ) {
                        id = "HDMI" + ( idx + 1 );
                    } else {
                        jSignage.warn( "Number of outputs exceeds provided list in single player multiple output project." );
                        row = config.rows;
                        break;
                    }
                } else {
                    id = "screen-" + ( row + 1 ) + "-" + ( col + 1 );
                }
                var screen = {
                    id: id,
                    x: col * ( screenWidth + framePixelsWidth ),
                    y: row * ( screenHeight + framePixelsHeight ),
                    width: screenWidth,
                    height: screenHeight
                };
                if ( id in idToPlayers )
                    screen.players = idToPlayers[ id ];
                if ( config.orientation ) {
                    if ( orientationOverride && isVertical( config.orientation ) === isVertical( orientationOverride ) )
                        screen.orientation = orientationOverride;
                    else
                        screen.orientation = config.orientation;
                }
                custom.screens.push( screen );
            }
        }

        return jSignage.Multiscreen.setupCustom( custom );
    },
    
    setupCustom: function ( config ) {
        // build the multiScreen element
        var R = {
            viewBox: [0, 0, config.width, config.height],
            byScreen: {},
            blending: null
        }
        window.localStorage.spxMultiscreenConfig = JSON.stringify( config );
        jSignage.Multiscreen.setAlternativeViewboxes( config.screens, true, config.blending, R, config.multiOutput );
        return R;
    },

    setupLayout: function ( viewBox ) {
        var config;
        try {
            config = JSON.parse( window.localStorage.spxMultiscreenConfig );
        } catch ( e ) {
            config = null;
        }
        if ( !config || typeof ( config ) != 'object' ) {
            alert( 'This layout must be used in a multiscreen project' );
            return null;
        }
        var R = {
            viewBox: [0, 0, 1920, 1080],
            byScreen: {},
            overlap: null
        };
        var box = reCustom.exec( viewBox );
        if ( box ) {
            R.viewBox = [parseFloat( box[1] ), parseFloat( box[2] ), parseFloat( box[3] ), parseFloat( box[4] )];
        } else if ( viewBox == 'all' ) {
            R.viewBox = [0, 0, config.width, config.height];
            jSignage.Multiscreen.setAlternativeViewboxes( config.screens, window.top == window, config.blending, R, config.multiOutput );
        } else {
            var wh = reXbyY.exec( viewBox );
            if ( wh ) {
                var W = parseInt( wh[1] );
                var H = parseInt( wh[2] );
                var screenWidth = 1920, screenHeight = 1080, framePixelsWidth = null, framePixelsHeight = null, screens = [];
                if ( config.screens.length > 0 ) {
                    var A = config.screens[0];
                    screenWidth = A.width;
                    screenHeight = A.height;
                    for ( var i = 1; i < config.screens.length && ( framePixelsWidth === null || framePixelsHeight === null ) ; i++ ) {
                        var B = config.screens[i];
                        if ( framePixelsWidth === null && B.x > A.x )
                            framePixelsWidth = B.x - A.x - screenWidth;
                        if ( framePixelsHeight === null && B.y > A.y )
                            framePixelsHeight = B.y - A.y - screenHeight;
                    }
                }
                if ( framePixelsWidth === null )
                    framePixelsWidth = 0;
                if ( framePixelsHeight === null )
                    framePixelsHeight = 0;
                if ( W > 0 && H > 0 ) {
                    R.viewBox = [0, 0, screenWidth * W + framePixelsWidth * ( W - 1 ), screenHeight * H + framePixelsHeight * ( H - 1 )];
                    if ( W > 1 || H > 1 ) {
                        for ( var row = 0; row < H; row++ ) {
                            for ( var col = 0; col < W; col++ ) {
                                screens.push( {
                                    id: "screen-" + ( row + 1 ) + "-" + ( col + 1 ),
                                    x: col * ( screenWidth + framePixelsWidth ),
                                    y: row * ( screenHeight + framePixelsHeight ),
                                    width: screenWidth,
                                    height: screenHeight
                                } );
                            }
                        }
                        jSignage.Multiscreen.setAlternativeViewboxes( screens, window.top == window, config.blending, R, config.multiOutput );
                    }
                }
            } else {
                var bbox = null, selection = [], idlist = viewBox.split( ',' ), idset={};
                for ( var i = 0; i < idlist.length; i++ )
                    idset[jSignage.trim( idlist[i] )] = true;
                for ( var row=0; row < config.screens.length; row++ ) {
                    var S = config.screens[row];
                    if ( !idset[S.id] )
                        continue;
                    selection.push( S );
                    if ( !bbox ) {
                        bbox = { left: S.x, top: S.y, right: S.x + S.width, bottom: S.y + S.height };
                    } else {
                        if ( S.x < bbox.left )
                            bbox.left = S.x;
                        if ( S.y < bbox.top )
                            bbox.top = S.y;
                        if ( S.x + S.width > bbox.right )
                            bbox.right = S.x + S.width;
                        if ( S.y + S.height > bbox.bottom )
                            bbox.bottom = S.y+S.height;
                    }
                }
                if ( selection.length > 1 ) {
                    R.viewBox = [bbox.left, bbox.top, bbox.right - bbox.left, bbox.bottom - bbox.top];
                    jSignage.Multiscreen.setAlternativeViewboxes( selection, window.top == window, config.blending, R, config.multiOutput );
                } else if ( selection.length==1 ) {
                    R.viewBox = [0, 0, bbox.right - bbox.left, bbox.bottom - bbox.top];
                }
            }
        }
        var bg = jSignage( '#bgrect' );
        if ( bg.length ) {
            bg.attr( 'x', R.viewBox[0] );
            bg.attr( 'y', R.viewBox[1] );
            bg.attr( 'width', R.viewBox[2] );
            bg.attr( 'height', R.viewBox[3] );
        }
        return R;
    },

    setAlternativeViewboxes: function( screens, add_multiscreen, add_blending, R, is_multioutput ) {
        var multiScreen = add_multiscreen ? document.createElementNS( $.spxNS, "multiScreen" ) : null;
        var mine = null, others = [], msid = window.MULTI_SCREEN_ID;
        for ( var row = 0; row < screens.length; row++ ) {
            var S = screens[row];
            var vbox = [S.x, S.y, S.width, S.height];
            R.byScreen[S.id] = vbox;
            if ( multiScreen ) {
                var screen = document.createElementNS( $.spxNS, "screen" );
                screen.id = S.id;
                screen.setAttribute( "viewBox", vbox.join( " " ) );
                if ( 'players' in S )
                    screen.setAttribute( "players", jSignage.isArray( S.players ) ? S.players.join( ',' ) : S.players );
                if ( S.orientation )
                    screen.setAttribute( "orientation", S.orientation );
                multiScreen.appendChild( screen );
            }
            if ( add_blending ) {
                if ( S.id == msid )
                    mine = vbox;
                else
                    others.push( vbox );
            }
        }
        if ( multiScreen ) {
            if ( is_multioutput )
                multiScreen.setAttribute( "multiOutput", "" + is_multioutput );
            jSignage( 'svg' ).append( multiScreen );
        }
        if ( add_blending ) {
            R.blending = {
                mine: mine,
                others: others
            };
        }
    },

    anchorTransform: function ( setup, layer ) {
        var list = ['left', 'top', 'right', 'bottom'];
        if ( typeof ( layer.args ) != 'object' )
            return;
        for ( var i = 0; i < list.length; i++ ) {
            var key = list[i];
            if ( key in layer.args ) {
                var ap = reXMLId.exec( layer.args[key] );
                if ( ap ) {
                    var anchor = ap[1];
                    var dot = anchor.lastIndexOf( '.' );
                    if ( dot > 0 && dot < anchor.length - 1 ) {
                        var id = anchor.substring( 0, dot );
                        var align = anchor.substring( dot + 1 );
                        if ( id in setup.byScreen ) {
                            if ( align == 'left' )
                                layer.args[key] = setup.byScreen[id][0];
                            else if ( align == 'top' )
                                layer.args[key] = setup.byScreen[id][1];
                            else if ( align == 'right' )
                                layer.args[key] = setup.byScreen[id][0] + setup.byScreen[id][2];
                            else if ( align == 'bottom' )
                                layer.args[key] = setup.byScreen[id][1] + setup.byScreen[id][3];
                            else
                                alert( 'Reference to unknown anchor point in screen' );
                        } else {
                            alert( 'Reference to anchor point in non-existing screen' );
                        }
                    }
                    if ( key=='right' ) {
                        layer.args.width = layer.args.right - ( layer.args.left || 0 );
                        delete layer.args.right;
                    } else if ( key=='bottom' ) {
                        layer.args.height = layer.args.bottom - ( layer.args.top || 0 );
                        delete layer.args.bottom;
                    }
                }
            }
        }
    },

    edgeBlending: function( setup ) {
        // check for overlaps and add blending borders
        var r = [], mine = setup.mine, others = setup.others, i;
        if ( mine ) {
            var mx1 = mine[0], mx2 = mine[0] + mine[2], my1 = mine[1], my2 = mine[1] + mine[3];
            var left = [], right = [], top = [], bottom = [];
            for ( var i = 0; i < others.length; i++ ) {
                var theirs = others[i];
                var tx1 = theirs[0], tx2 = theirs[0] + theirs[2], ty1 = theirs[1], ty2 = theirs[1] + theirs[3];
                if ( tx1 < mx2 && ty1 < my2 && mx1 < tx2 && my1 < ty2 ) {
                    var x1 = Math.max( mx1, tx1 );
                    var x2 = Math.min( mx2, tx2 );
                    var y1 = Math.max( my1, ty1 );
                    var y2 = Math.min( my2, ty2 );
                    if ( tx1 < mx1 && tx2 <= mx2 && y2 - y1 >= x2 - x1 )
                        left.push( [x1, y1, x2, y2] );
                    if ( tx2 > mx2 && tx1 >= mx1 && y2 - y1 >= x2 - x1 )
                        right.push( [x1, y1, x2, y2] );
                    if ( ty1 < my1 && ty2 < my2 && x2 - x1 > y2 - y1 )
                        top.push( [x1, y1, x2, y2] );
                    if ( ty2 > my2 && ty1 >= my1 && x2 - x1 > y2 - y1 )
                        bottom.push( [x1, y1, x2, y2] );
                }
            }
            function remove_overlaps( list ) {
                for ( var i = 0; i < list.length ; i++ ) {
                    for ( var j = i + 1; j < list.length; j++ ) {
                        if ( list[j][0] < list[i][2] && list[j][1] < list[i][3] && list[i][0] < list[j][2] && list[i][1] < list[j][3] ) {
                            if ( ( list[i][2] - list[i][0] ) * ( list[i][3] - list[i][1] ) < ( list[j][2] - list[j][0] ) * ( list[j][3] - list[j][1] ) ) {
                                list = list.splice( i, 1 );
                                --i;
                                --j;
                            } else {
                                list = list.splice( j, 1 )
                                --j;
                            }
                        }
                    }
                }
            }
            remove_overlaps( left );
            for ( i = 0; i < left.length; i++ )
                r.push( jSignage.Multiscreen.createBorder( false, true, left[i][0], left[i][1], left[i][2], left[i][3] ) );
            remove_overlaps( right );
            for ( i = 0; i < right.length; i++ )
                r.push( jSignage.Multiscreen.createBorder( false, false, right[i][0], right[i][1], right[i][2], right[i][3] ) );
            remove_overlaps( top );
            for ( i = 0; i < top.length; i++ )
                r.push( jSignage.Multiscreen.createBorder( true, true, top[i][0], top[i][1], top[i][2], top[i][3] ) );
            remove_overlaps( bottom );
            for ( i = 0; i < bottom.length; i++ )
                r.push( jSignage.Multiscreen.createBorder( true, false, bottom[i][0], bottom[i][1], bottom[i][2], bottom[i][3] ) );
        }

        return r;
    },

    uncan: function ( setup, json, text_modifier, args_modifier, jsonTransform ) {
        return jSignage.uncan( json, text_modifier, args_modifier, function ( layer ) {
            if ( jsonTransform )
                jsonTransform( layer );
            jSignage.Multiscreen.anchorTransform( setup, layer );
        } );
    }

};

} ) ( );
