/*!
 * jSignage.Weather
 * http://www.spinetix.com/
 * Copyright SpinetiX S.A.
 * Released under the GPL Version 2 or later license.

 * $Date: 2021-02-01 09:38:38 +0000 (Mon, 01 Feb 2021) $
 * $Revision: 37509 $
 */
 ( function () {
var version = new String( "1.0.3" );
version.major = 1;
version.minor = 0;
version.revision = 3;

function getOptionsSrcFromUriArray( base, uri, data ) {
    var results = [];
    for ( var i=0; i<data.length; i++){
        results.push( $.extend( true, { src: uri.replace("[[i]]", encodeURIComponent(data[i].trim())) }, base ) );
    }
    return results;
}

function addParser( src, parser ){
    if ( jSignage.isArray( src ) ){
        for( var i=0; i<src.length; i++ ){
            src[i].parser.push( parser );
        }
        // parser is the same for all in the end
        //src[0].parser.push( parser );
    } else {
        src.parser.push( parser );
    }
    return src;
}

function compareVersion( v1, v2 ){
    v1 = v1.split(".");
    v2 = v2.split(".");
    for ( var i=0; i< 3; i++ ){
        if ( v1[i] < v2[i] ){
            return -1;
        }
        if ( v1[i] > v2[i] ){
            return 1;
        }
    }
    return 0;    
}

function getConfig( args, src ) {
    var config;
    if ( args.target === 'today' ){
        config = $.extend(true, {}, src.daily );
        args.cnt = 2;
        config.parser.push( {
            startDate: "this day",
            endDate: "this day",
            src: "date",
            test: "range",
            type: "filterDate"
        });            
    } else if ( args.target === 'tomorrow' ){
        config = $.extend(true, {}, src.daily );            
        args.cnt = 3;
        config.parser.push( {
            startDate: "+1 day",
            endDate: "+1 day",
            src: "date",
            test: "range",
            type: "filterDate"
        });
    } else {
        config = $.extend(true, {},src[args.target] );
    }
    return config;
}

/* jSignage Weather add-on implementation */
jSignage.Weather = { 
    version: version,
    iconsNames: [
        "sunny", "cloudy_rain1", "cloudy_rain2", "cloudy_rain3", "rain1", "rain2", "rain3", "mist", "cloudy1", "cloudy2", "cloudy3", "snow", "storm", "thunderstorm", "tornado", "windy"
    ],
    options : {
        iconSrc: "http://download.spinetix.com/content/widgets/icons/weather/",
        iconExt: "png"
    },
    extendOptionsFromUri: function ( options  ){
        var uri = document.documentURI || window.location && window.location.href || document.location && document.location.href;
        return $.extend( options, jSignage.decodeURIQueryString( uri ) );        
    },
    getAndParseFeedArray: function( feeds, callback, useGetURL ) {        
        var cnt;
        var results = [];
        var addResult = function( idx, data, rt ){
            if ( jSignage.isArray( data ) && data.length === 1){
                results[idx] = data[0];
            } else {
                results[idx] = data;
            }
            cnt--;
            if ( cnt === 0 ){
                callback( results, rt );
                cnt = results.length;
            }
        };
        var process = function( idx, feed ){
            jSignage.getAndParseFeed( feed, function( data, rt ) {
                addResult( idx, data, rt );
            }, useGetURL );
        };
        if ( jSignage.isArray( feeds ) ){
            cnt = feeds.length;
            for ( var i=0; i<feeds.length; i++ ){
                process( i, feeds[i]);               
            }
        } else {
            return jSignage.getAndParseFeed( feeds, callback, useGetURL );    
        }            
    },
    getFallbacks: function( i, args ){
        i = i || 0;
        args = args || {};
        var iconSrc = args.iconSrc || jSignage.Weather.options.iconSrc,
            iconExt = args.iconExt || jSignage.Weather.options.iconExt;
        var item = { 
            name : "Somewhere",
            country: "US",
            date: new Date( Date.now()+i*24*3600*1000 ),
            temp: Math.floor( Math.random()*60-20 ),
            icon: jSignage.Weather.iconsNames[ Math.floor( Math.random()*7 ) ]
        };
        item.icon_url = iconSrc + item.icon + '.' + iconExt;
        return item;
    },
    getRandomWeather: function( nbItems, args ){
        args = args || {};
        var data = []; 
        for ( var i=0; i<nbItems; i++ ){
            data.push( jSignage.Weather.getFallbacks( i, args ) );
        }
        return data;        
    },    
    getWeather: function( args, provider, callback ){
        var dataSource = provider.getOptions( args ), 
            iconSrc = args.iconSrc || jSignage.Weather.options.iconSrc,
            iconExt = args.iconExt || jSignage.Weather.options.iconExt;
        addParser( dataSource, {
            inline: "return { icon_url: \"" + iconSrc + "\" + text +\"." + iconExt + "\" };",
            src: "icon",
            type: "custom"
        } );
        if ( args.locale ){
            jSignage.setLocale( args.locale );
        }        
        jSignage.Weather.getAndParseFeedArray( dataSource, callback );      
    }
};

/* OpenWeatherMap provider implementation */
jSignage.Weather.openweather = {
    getOptions: function ( args ) {
        var openweatherAPI = "http://api.openweathermap.org/data/2.5";
        var openweatherAPIGateway = 'https://api-gw.services.spinetix.com/openweathermap_v1';
        
        var config = getConfig( args, jSignage.Weather.openweather );
        
        // fix for trhe case of rain icon with no rain
        addParser( config, {
            inline: function( id, rowData ) {
                if ( id < 300 ) return {icon: "thunderstorm"};
                if ( id < 500 ) return {icon: "rain2"};
                if ( id == 500 ) return {icon: "cloudy_rain1"};
                if ( id == 501 ) return {icon: "cloudy_rain2"};
                if ( id < 600 ) return {icon: "rain3"};
                if ( id < 700 ) return {icon: "snow"};
                if ( id < 781 ) return {icon: "mist"};
                if ( id == 781 ) return {icon: "tornado"};
                if ( id == 800 ) return {icon: "sunny"};
                if ( id == 801 ) return {icon: "cloudy1"};
                if ( id == 802 ) return {icon: "cloudy2"};
                if ( id < 900 ) return {icon: "cloudy3"};
                if ( id < 903 ) return {icon: "tornado"};
                return {icon: "cloudy_rain2"};
            },
            src: "id",
            type: "custom"
        } );
        /*
        if ( compareVersion( jSignage.fn.jsignage, "1.2.1") >= 0) {
            addParser( config, {
                inline: "alert(text + ' => ' + rowData.rain); if ( text == '10d' && rowData.rain < 1 ) { if ( rowData.clouds > 0 ) return '02d'; else return '01d'; } else { return text; };",
                src: "icon",
                type: "custom"
            } );
        }
        */
        var argsProps = ['refresh', 'syncVariable'];            
        var configProps = ['fallbacks', 'parser']; 
        var dataSource = {};
        jSignage.copyProps( args, dataSource, argsProps );
        jSignage.copyProps( config, dataSource, configProps );
        dataSource.type = 'uri';
        
        var src = config.baseSrc;
        
        if ( args.units && ( args.units === "imperial" || args.units === "F" || args.units === "f" || args.units === "Fahrenheit" ) ) {
            src += "units=imperial";
        } else {
            src += "units=metric";
        }
        if ( args.cnt ) {
            src += "&cnt=" + args.cnt;
        }
        if ( args.locale ) {
            src += "&lang=" + args.locale.trim();
        }       
        if ( args.apiid && args.apiid !== "" ) {
            src = openweatherAPI + src + "&APPID=" + args.apiid;            
        } else {
            src = openweatherAPIGateway + src; 
            dataSource.headers = {"x-api-key": "m7yvCCmkfe7G2csuc4wKU24dMhSjzVUw4XXBuPiI"};
        }
        var item;
        var srcArray = [];
        // smart detections  
        if ( args.locations ){
            if ( !jSignage.isArray( args.locations ) ){
                if ( args.locations.search(";") !== -1  ){
                    args.locations = args.locations.split(";");
                } else {
                    args.locations = [ args.locations ];  
                }
            }
            for ( var i=0; i<args.locations.length; i++ ){
                var loc = args.locations[i];
                if ( loc == parseInt(loc,10) ){
                    if ( ("" + loc).length == 7 ){
                        srcArray.push( src + "&id=" + loc );
                    } else if ( ("" + loc).length == 5 ){
                        // us zip code?
                        srcArray.push( src + "&zip=" + loc );
                    } else {
                        // fall back to location
                        srcArray.push( src + "&q=" + encodeURIComponent(loc.trim()) );
                    }
                } else if ( loc.search(",")!== -1 ) {
                    var data = loc.split(",");
                    if ( data.length == 2 ){
                        if ( data[0] == parseFloat(data[0]) && data[1] == parseFloat(data[1]) ){
                            srcArray.push( src + "&lat=" + data[0] + "&lon=" + data[1] ); 
                        } else if ( data[0] == parseInt(data[0],10) ) {
                            srcArray.push( src + "&zip=" + loc );
                        } else {
                            // fall back to location
                            srcArray.push( src + "&q=" + encodeURIComponent(loc.trim()) );    
                        }
                    } else {
                        // fall back to location
                        srcArray.push( src + "&q=" + encodeURIComponent(loc.trim()) );
                    }
                } else {
                    // fall back to location
                    srcArray.push( src + "&q=" + encodeURIComponent(loc.trim()) );    
                }               
            }            
            var results = [];
            for ( var i=0; i<srcArray.length; i++){
                results.push( $.extend( true, { src: srcArray[i] }, dataSource ) );
            }
            return results;             
        } else {
            if ( args.id ){
                src += "&id=[[i]]";
                item = args.id;
            } else if ( args.lat && args.lon ){
                src += "&lat=" + args.lat + "&lon=" + args.lon;
                dataSource.src = src + item.trim();
                return dataSource;
            } else if ( args.zip ){
                src += "&zip=[[i]]";
                item = args.zip;
            } else {
                throw Error('Must specify a location, using either "locations", "id", "lat, lon" or "zip"');
            }
            if ( !jSignage.isArray( item ) && item.search(";") !== -1 ){
                item = item.split(";");
            }
            if ( jSignage.isArray( item ) ){                
                return getOptionsSrcFromUriArray( dataSource, src, item );
            } else {
                dataSource.src = src.replace( "[[i]]", encodeURIComponent(item.trim())) ;
                return dataSource;
            }
        }
    },        
    current : {            
        fallbacks: {
            clouds: 0,
            rain: 0,
            snow: 0
        },
        parser: [
            {   
                columns: [
                        { name: "id", property: "weather.0.id" },
                        { name: "main", property: "weather.0.main" },
                        { name: "description", property: "weather.0.description" },
                        { name: "temp", property: "main.temp" },
                        { name: "pressure", property: "main.pressure" },
                        { name: "humidity", property: "main.humidity" },
                        { name: "temp_min", property: "main.temp_min" },
                        { name: "temp_max", property: "main.temp_max" },
                        { name: "wind_speed", property: "wind.speed" },
                        { name: "wind_deg", property: "wind.deg" },
                        { name: "clouds", property: "clouds.all" },
                        { name: "rain", property: "rain.3h" },
                        { name: "snow", property: "snow.3h" },
                        { name: "date", property: "dt" },
                        { name: "country", property: "sys.country" },
                        { name: "sunrise", property: "sys.sunrise" },
                        { name: "sunset", property: "sys.sunset" },                        
                        { name: "name", property: "name" }
                ],
                rows: "",
                type: "json"
            },
            { format: "unix", src: "date", type: "date" },
            { format: "unix", src: "sunrise", type: "date" },
            { format: "unix", src: "sunset", type: "date" }              
        ],
        baseSrc: "/weather?"
    }, 
    daily : {            
        fallbacks: {
            clouds: 0,
            rain: 0,
            snow: 0
        },
        parser: [
            {   
                columns: [
                        { name: "country", property: "city.country" },
                        { name: "name", property: "city.name" },
                        { name: "id", property: "list" }
                ],
                rows: "",
                type: "json"
            },
            {   
                columns: [
                        { name: "date", property: "dt" },
                        { name: "temp", property: "temp.day" },
                        { name: "temp_min", property: "temp.min" },
                        { name: "temp_max", property: "temp.max" },
                        { name: "temp_night", property: "temp.night" },
                        { name: "temp_day", property: "temp.day" },                        
                        { name: "pressure", property: "pressure" },
                        { name: "humidity", property: "humidity" },
                        { name: "id", property: "weather.0.id" },
                        { name: "main", property: "weather.0.main" },
                        { name: "description", property: "weather.0.description" },
                        { name: "wind_deg", property: "deg" },
                        { name: "wind_speed", property: "speed" },
                        { name: "clouds", property: "clouds" },                                                    
                        { name: "rain", property: "rain" },
                        { name: "snow", property: "snow" }                        
                ],
                src: "id",
                rows: "",
                type: "json"
            },
            { format: "unix", src: "date", type: "date" }		
        ],
        baseSrc: "/forecast/daily?"
    },
    hourly : {            
        fallbacks: {
            clouds: 0,
            rain: 0,
            snow: 0
        },
        parser: [
            {   
                columns: [
                        { name: "country", property: "city.country" },
                        { name: "name", property: "city.name" },
                        { name: "id", property: "list" }
                ],
                rows: "",
                type: "json"
            },
            {   
                columns: [
                        { name: "date", property: "dt" },
                        { name: "temp", property: "main.temp" },
                        { name: "temp_max", property: "main.temp_max" },
                        { name: "temp_min", property: "main.temp_min" },
                        { name: "pressure", property: "main.pressure" },
                        { name: "humidity", property: "main.humidity" },
                        { name: "id", property: "weather.0.id" },
                        { name: "main", property: "weather.0.main" },                        
                        { name: "description", property: "weather.0.description" },                        
                        { name: "clouds", property: "clouds.all" },                            
                        { name: "wind_speed", property: ".wind.speed" },
                        { name: "wind_deg", property: "wind.deg" },
                        { name: "rain", property: "rain.3h" },
                        { name: "snow", property: "snow.3h" }
                ],
                src: "id",
                rows: "",
                type: "json"
            },
            { format: "unix", src: "date", type: "date" }		
        ],
        baseSrc: "/forecast?"
    }
};   

/* WeatherUnderground provider implementation */
jSignage.Weather.wunderground = {
    getOptions: function ( args ) {
        var config = getConfig( args, jSignage.Weather.wunderground );
        var map = {
            "chanceflurries": "snow",
            "chancerain": "cloudy_rain1",
            "chancesleet": "snow",
            "chancesnow": "snow",
            "chancetstorms": "cloudy_rain3",
            "clear": "sunny",
            "cloudy": "cloudy2",
            "flurries": "snow",
            "fog": "mist",
            "hazy": "cloudy1",
            "mostlycloudy": "cloudy3",
            "mostlysunny": "cloudy1",
            "partlycloudy": "cloudy2",
            "partlysunny": "cloudy2",
            "sleet": "snow",
            "rain": "rain3",
            "snow": "snow",
            "sunny": "sunny",
            "tstorms": "thunderstorm",
            "unknown": "cloudy_rain2"
        };
        // fix for trhe case of rain icon with no rain
        addParser( config, {
            inline: function( id, rowData ) {
                if ( id in map ) return {icon: map[id]};                
                return {icon: "cloudy_rain2"};
            },
            src: "id",
            type: "custom"
        } );
        
        if ( args.units && ( args.units === "imperial" || args.units === "F" || args.units === "f" || args.units === "Fahrenheit" ) ) {            
            addParser( config, {
                inline: function( val ) { return {temp: val}; },
                src: "temp_f",
                type: "custom"
            } );
        } else {
            addParser( config, {
                inline: function( val ) { return {temp: val}; },
                src: "temp_c",
                type: "custom"
            } );
        }
        var argsProps = ['refresh', 'syncVariable'];            
        var configProps = ['fallbacks', 'parser']; 
        var dataSource = {};
        jSignage.copyProps( args, dataSource, argsProps );
        jSignage.copyProps( config, dataSource, configProps );
        dataSource.type = 'uri';
        
        var src = "http://api.wunderground.com/api/";
        if ( args.apiid && args.apiid !== "" ) {
            src += args.apiid + "/";
        } else {
            throw Error('Must specify an API Key');
        }
        src += config.features + "/";
        if ( args.locale ) {
            src += "lang:" + args.locale.trim() +"/";
        }
        var item;
        if ( args.locations && args.locations!="" && ( !jSignage.isArray( args.locations ) || args.locations.length != 0) ){
            src += "/q/[[i]]";
            item = args.locations;
        } else if ( args.id ){
            src += "/q/pws:[[i]]";
            item = args.id;
        } else if ( args.lat && args.lon ){
            src += "/q/"+args.lat+","+args.lon;
            item = "";            
        } else if ( args.zip ){
            src += "/q/[[i]]";
            item = args.zip;
        } else {
            src += "/q/autoip";
            item = "";
        }
        src += ".json";
        if ( !jSignage.isArray( item ) && item.search(";") !== -1 ){
            item = item.split(";");
        }
        if ( jSignage.isArray( item ) ){                
            return getOptionsSrcFromUriArray( dataSource, src, item );
        } else {
            dataSource.src = src.replace( "[[i]]", encodeURIComponent(item.trim())) ;
            return dataSource;
        }
    },        
    current : {            
        parser: [
            {   
                columns: [
                        { name: "name", property: "display_location.city" },
                        { name: "country", property: "display_location.country_iso3166" },
                        { name: "fullname", property: "display_location.full" },
                        
                        { name: "date", property: "observation_time_rfc822" },
                        
                        { name: "id", property: "icon" },
                        { name: "main", property: "weather" },
                        { name: "description", property: "weather" },
                        
                        { name: "temp_f", property: "temp_f" },
                        { name: "temp_c", property: "temp_c" },
                        { name: "pressure", property: "pressure_mb" },
                        { name: "humidity", property: "relative_humidit" },
                        { name: "wind_speed", property: "wind_kph" },
                        { name: "wind_deg", property: "wind_degrees" },
                        { name: "rain", property: "precip_today_metric" }
                ],
                rows: "current_observation",
                type: "json"
            },
            { format: "rfc822", src: "date", type: "date" }  
        ],
        features : "conditions"
    },
    daily : {            
        parser: [
            {   
                columns: [
                        { name: "date", property: "date.epoch" },
                        
                        { name: "id", property: "icon" },
                        { name: "main", property: "conditions" },
                        { name: "description", property: "conditions" },
                        
                        { name: "temp_f", property: "high.fahrenheit" },
                        { name: "temp_c", property: "high.celsius" },
                        { name: "temp_max_f", property: "high.fahrenheit" },
                        { name: "temp_max_c", property: "high.celsius" },
                        { name: "temp_min_f", property: "low.fahrenheit" },
                        { name: "temp_min_c", property: "low.celsius" },
                        
                        { name: "humidity", property: "avehumidity" },
                        { name: "wind_speed", property: "avewind.kph" },
                        { name: "wind_deg", property: "avewind.degrees" },
                        { name: "rain", property: "qpf_allday.mm" },
                        { name: "snow", property: "snow_allday.mm" }
                ],
                rows: "forecast.simpleforecast.forecastday",
                type: "json"
            },
            { format: "unix", src: "date", type: "date" }  
        ],
        features : "forecast10day" // "forecast"
    }
}; 

/* WorldWeatherOnline provider implementation */
jSignage.Weather.worldweatheronline = {
    getOptions: function ( args ) {
        var config = getConfig( args, jSignage.Weather.worldweatheronline );
        var map = {
                395: "snow",
                392: "thunderstorm",
                389: "thunderstorm",
                386: "thunderstorm",
                377: "snow",
                374: "snow",
                371: "snow",
                368: "snow",
                365: "snow",
                362: "snow",
                359: "rain3",
                356: "rain3",
                353: "rain1",
                350: "snow",
                338: "snow",
                335: "snow",
                332: "snow",
                329: "snow",
                326: "snow",
                323: "snow",
                320: "snow",
                317: "snow",
                314: "snow",
                311: "snow",
                308: "rain3",
                305: "rain3",
                302: "rain3",
                299: "rain3",
                296: "cloudy_rain3",
                293: "cloudy_rain3",
                284: "snow",
                281: "snow",
                266: "cloudy_rain2",
                263: "cloudy_rain2",
                260: "mist",
                248: "mist",
                230: "snow",
                227: "snow",
                200: "thunderstorm",
                185: "snow",
                182: "snow",
                179: "snow",
                176: "cloudy_rain1",
                143: "cloudy_rain1",
                122: "cloudy3",
                119: "cloudy2",
                116: "cloudy1",
                113: "sunny"
            };
        // fix for trhe case of rain icon with no rain
        addParser( config, {
            inline: function( id, rowData ) {
                if ( id in map ) return {icon: map[id]};                
                return {icon: "cloudy_rain2"};
            },
            src: "id",
            type: "custom"
        } );
        
        if ( args.units && ( args.units === "imperial" || args.units === "F" || args.units === "f" || args.units === "Fahrenheit" ) ) {            
            addParser( config, {
                inline: function( val ) { return {temp: val}; },
                src: "temp_f",
                type: "custom"
            } );
        } else {
            addParser( config, {
                inline: function( val ) { return {temp: val}; },
                src: "temp_c",
                type: "custom"
            } );
        }
        var argsProps = ['refresh', 'syncVariable'];            
        var configProps = ['fallbacks', 'parser']; 
        var dataSource = {};
        jSignage.copyProps( args, dataSource, argsProps );
        jSignage.copyProps( config, dataSource, configProps );
        dataSource.type = 'uri';
        
        var src = "http://api.worldweatheronline.com/premium/v1/weather.ashx?includelocation=yes&extra=localObsTime";
        if ( args.apiid && args.apiid !== "" ) {
            src += "&key=" + args.apiid;
        } else {
            throw Error('Must specify an API Key');
        }
        src += "&" + config.options;
        if ( args.cnt ) {
            src += "&num_of_days=" + args.cnt;
        }
        if ( args.locale ) {
            src += "&lang=" + args.locale.trim();
        }
        var item;
        if ( args.locations && args.locations!="" && ( !jSignage.isArray( args.locations ) || args.locations.length != 0) ){
            src += "&q=[[i]]";
            item = args.locations;
        } else if ( args.lat && args.lon ){
            src += "&q="+args.lat+"/"+args.lon;
            item = "";            
        } else if ( args.zip ){
            src += "&q=[[i]]";
            item = args.zip;
        } else {
            throw Error('Must specify a location, using either "locations", "lat, lon" or "zip"');
        }
        src += "&format=json";
        if ( !jSignage.isArray( item ) && item.search(";") !== -1 ){
            item = item.split(";");
        }
        if ( jSignage.isArray( item ) ){                
            return getOptionsSrcFromUriArray( dataSource, src, item );
        } else {
            dataSource.src = src.replace( "[[i]]", encodeURIComponent(item.trim())) ;
            return dataSource;
        }
    },        
    current : {            
        parser: [
            {   
                columns: [
                        { name: "name", property: "nearest_area.0.areaName.0.value" },
                        { name: "country", property: "nearest_area.0.country.0.value" },
                        
                        { name: "date", property: "current_condition.0.localObsDateTime" },
                        
                        { name: "id", property: "current_condition.0.weatherCode" },
                        { name: "main", property: "current_condition.0.weatherDesc.0.value" },
                        { name: "description", property: "current_condition.0.weatherDesc.0.value" },
                        
                        { name: "temp_f", property: "current_condition.0.temp_F" },
                        { name: "temp_c", property: "current_condition.0.temp_C" },
                        { name: "pressure", property: "current_condition.0.pressure" },
                        { name: "humidity", property: "current_condition.0.humidity" },
                        { name: "wind_speed", property: "current_condition.0.windspeedKmph" },
                        { name: "wind_deg", property: "current_condition.0.winddirDegree" },
                        { name: "rain", property: "current_condition.0.precipMM" },
                        { name: "clouds", property: "current_condition.0.cloudcover" }
                ],
                rows: "data",
                type: "json"
            },
            { format: "yyyy-MM-dd h:mm a", src: "date", type: "date" }  
        ],
        options : "fx=no&cc=yes"
    },
    daily : {            
        parser: [
            {   
                columns: [
                        { name: "name", property: "nearest_area.0.areaName.0.value" },
                        { name: "country", property: "nearest_area.0.country.0.value" },
                        { name: "id", property: "weather" },
                ],
                rows: "data",
                type: "json"
            },
            {   
                columns: [
                        
                        { name: "id", property: "hourly.0.weatherCode" },
                        { name: "main", property: "hourly.0.weatherDesc.0.value" },
                        { name: "description", property: "hourly.0.weatherDesc.0.value" },
                        
                        { name: "date", property: "date" },
                        
                        { name: "temp_f", property: "hourly.0.tempF" },
                        { name: "temp_c", property: "hourly.0.tempC" },
                        
                        { name: "humidity", property: "hourly.0.humidity" },
                        { name: "wind_speed", property: "hourly.0.windspeedKmph" },
                        { name: "wind_deg", property: "hourly.0.winddirDegree" },
                        { name: "rain", property: "hourly.0.precipMM" },
                        { name: "clouds", property: "hourly.0.cloudcover" }
                ],
                src: "id",
                type: "json"
            },
            { format: "iso8601", src: "date", type: "date" }  
        ],
        options : "fx=yes&cc=no&fx24=yes" // "forecast"
    }
}; 

/* YahooWeather provider implementation */
jSignage.Weather.yahooweather = {
    getOptions: function ( args ) {
        var config = getConfig( args, jSignage.Weather.yahooweather );
        
        // fix for trhe case of rain icon with no rain
        addParser( config, {
            inline: function( id, rowData ) {
                if ( id < 3 ) return {icon: "tornado"};
                if ( id < 5 ) return {icon: "thunderstorm"};
                if ( id == 11 || id == 12 || id == 40 ) return {icon: "cloudy_rain3"};
                if ( id < 19 || id == 41 || id == 42 || id == 43 || id == 46 ) return {icon: "snow"};
                if ( id < 26 ) return {icon: "mist"};
                if ( id == 26 ) return {icon: "cloudy3"};
                if ( id == 27 || id == 28 ) return {icon: "cloudy2"};
                if ( id == 29 || id == 30 || id == 44 ) return {icon: "cloudy1"};
                if ( id == 31 || id == 32 || id == 36 ) return {icon: "sunny"};
                if ( id == 33 || id == 34 ) return {icon: "cloudy_rain1"};
                if ( id == 35 ) return {icon: "rain3"};
                if ( id < 40 || id == 45 || id == 47) return {icon: "thunderstorm"};
                if ( id == 44 ) return {icon: "cloudy1"};
                return {icon: "cloudy_rain2"};
            },
            src: "id",
            type: "custom"
        } );

        var argsProps = ['refresh', 'syncVariable'];            
        var configProps = ['fallbacks', 'parser']; 
        var dataSource = {};
        jSignage.copyProps( args, dataSource, argsProps );
        jSignage.copyProps( config, dataSource, configProps );
        dataSource.type = 'uri';
                
        var base = "https://query.yahooapis.com/v1/public/yql?q=select%20*%20from%20weather.forecast%20where";
        var cityfind = "select%20woeid%20from%20geo.places%281%29%20where%28[[i]]%29";
        var city = "text%3D%22[[i]]%22";
        var woeid = "%20woeid%20in%20%28[[i]]%29";
        var unit = "%20and%20u%3D%22[[i]]%22";
        var option ="&format=json";
        if ( args.units && ( args.units === "imperial" || args.units === "F" || args.units === "f" || args.units === "Fahrenheit" ) ) {
            unit = unit.replace("[[i]]", "f");
        } else {
            unit = unit.replace("[[i]]", "c");
        }
        
        var item;
        if ( args.locations ){            
            item = args.locations;
            if ( args.target !== 'daily' ){
                if ( !jSignage.isArray( item ) && item.search(";") !== -1 ){
                    item = item.split(";");
                }
                if ( jSignage.isArray( item ) ){
                    var str = "";
                    for ( var i=0; i<item.length; i++ ){
                        if ( i!=0) {
                            str += "%20OR%20";
                        }
                        str += city.replace("[[i]]", encodeURIComponent(item[i].trim()));
                    }
                    cityfind = cityfind.replace( "[[i]]", str);                
                } else {
                    cityfind = cityfind.replace( "[[i]]", city.replace("[[i]]", encodeURIComponent(item.trim())));
                }
                woeid = woeid.replace( "[[i]]", cityfind );
                item = "";
            } else {
                woeid = woeid.replace( "[[i]]", cityfind.replace( "[[i]]", city ) );
            }
        } else if ( args.id ){            
            item = args.id;
            if ( args.target !== 'daily' ){
                if ( !jSignage.isArray( item ) && item.search(";") !== -1 ){
                    item = item.split(";");
                }
                if ( jSignage.isArray( item ) ){                    
                    var str = "";
                    for ( var i=0; i<item.length; i++ ){
                        if ( i!=0) {
                            str += ",%20";
                        }
                        str += encodeURIComponent(item[i].trim());
                    }
                    woeid = woeid.replace( "[[i]]", str );                
                } else {
                    woeid = woeid.replace( "[[i]]", item );                
                }
                item = "";
            } else {
                
            }
        } else {
            throw Error('Must specify a location, using either "city", "id"');
        }
        if ( !jSignage.isArray( item ) && item.search(";") !== -1 ){
            item = item.split(";");
        }
        if ( jSignage.isArray( item ) ){                
            return getOptionsSrcFromUriArray( dataSource, base + woeid + unit + option, item );
        } else {
            dataSource.src = base + woeid + unit + option;
            return dataSource;
        }
    },        
    current : {            
        parser: [
            {   
                columns: [
                        { name: "name", property: "location.city" },
                        { name: "country", property: "location.country" },
                        { name: "wind_speed", property: "wind.speed" },
                        { name: "wind_deg", property: "wind.direction" },
                        { name: "humidity", property: "atmosphere.humidity" },
                        { name: "pressure", property: "atmosphere.pressure" },
                        
                        
                        { name: "id", property: "item.condition.code" },
                        //{ name: "date", property: "item.condition.date" },
                        { name: "temp", property: "item.condition.temp" },
                        
                        { name: "main", property: "item.condition.text" },
                        { name: "description", property: "item.condition.text" },
                        
                        { name: "sunrise", property: "astronomy.sunrise" },
                        { name: "sunset", property: "astronomy.sunset" },                                                
                ],
                rows: "query.results.channel",
                type: "json"
            },
            {
                inline: function( id, rowData ) {                    
                    return { date: Date.now() };
                },
                src: "id",
                type: "custom"
            }              
        ]        
    }, 
    daily : {            
        parser: [
            {   
                columns: [
                        { name: "name", property: "location.city" },
                        { name: "country", property: "location.country" },
                        { name: "id", property: "item.forecast" }
                ],
                rows: "query.results.channel",
                type: "json"
            },{   
                columns: [
                        { name: "id", property: "code" },
                        //{ name: "date", property: "item.condition.date" },
                        { name: "temp", property: "high" },
                        { name: "temp_max", property: "high" },
                        { name: "temp_min", property: "low" },
                        
                        { name: "main", property: "text" },
                        { name: "description", property: "text" },
                        { name: "date", property: "date" },                        
                ],
                src: "id",
                type: "json"
            },
            { format: "rfc822", src: "date", type: "date" }             
        ]        
    }  
};

} )();
