<?php

class Validator {
    var $baseAttribute = "";
    var $model = null;
    
    public function __construct( $model, $baseAttribute = 'media' ) {
        $this->model = $model;
        $this->baseAttribute = $baseAttribute;
    }
    public function addError( $attribute, $desc ){
        throw new Exception ( "bad function call" );
    }
    public function addMediaError( $attribute, $desc ){
        if ( $this->model===null ){
            throw new Exception( "no model defined" ) ;
        }
        $this->model->addError( $this->baseAttribute, $attribute.": ".$desc );
    }
    public function validateBoolean( $val, $attribute ){
        if ( $val!==true && $val!==false ){
            $this->addMediaError( $attribute, "should be a boolean" );
            return false;
        }
        return true;
    }
    public function validateColor( $color, $attribute ){
        $colorRegex = "/^#[0-9A-Fa-f]{6}$/";
        if ( !preg_match( $colorRegex, $color) ){
            $this->addMediaError( $attribute, "should be an hex color" );
            return false;
        }
        return true;
    }
    public function validateText( $text, $attribute ){
        if ( strpos($text, "\n")!==false ){
            $this->addMediaError( $attribute, "should be single line" );
            return false;
        }
        return true;
    }
    public function validateTime( $time, $attribute ){
        $timeRegex = "/^(2[0-3]|[0-1]\d)\:[0-5]\d(\:[0-5]\d(\.\d+)?)?Z?$/";
        if ( !preg_match( $timeRegex, $time) ){
            $this->addMediaError( $attribute, "should be iso-8601 time" );
            return false;
        }
        return true;
    }
    public function validateDur( $time, $attribute ){
        $timeRegex = "/^\d+(\.\d)?$/";
        if ( !preg_match( $timeRegex, $time) ){
            $this->addMediaError( $attribute, "should be a number of seconds" );
            return false;
        }
        return true;
    }
    public function validateDate( $date, $attribute ){
        $dateRegex = "/^\d{4}-(0[1-9]|1[0-2])-([0-2]\d|3[0-1])$/";
        if ( !preg_match( $dateRegex, $date) ){
            $this->addMediaError( $attribute, "should be iso-8601 date" );
            return false;
        }
        return true;
    }
    public function validateDateTime( $dateTime, $attribute ){
        $dateRegex = "\d{4}-(0[1-9]|1[0-2])-([0-2]\d|3[0-1])";
        $timeRegex = "(2[0-3]|[0-1]\d)\:[0-5]\d(\:[0-5]\d(\.\d+)?)?Z?";
        $regex = "/^".$dateRegex."T".$timeRegex."$/";
        if ( !preg_match( $regex, $dateTime) ){
            $this->addMediaError( $attribute, "should be iso-8601 date time" );
            return false;
        }
        return true;
    }
    public function validateZone( &$zones, $attribute ){
        if( !is_array($zones) ) {
            $this->addMediaError( $attribute, 
                    "should be an array");
            return false;
        } 
        $i=0;
        foreach ( $zones as &$zone) {
            
            if ( !isset($zone['beginTime']) )
                $zone['beginTime'] = "00:00:00";
            if ( ! $this->validateTime( $zone['beginTime'], 
                                        $attribute."[$i].beginTime" ) )
                    return false;
            if ( !isset($zone['endTime']) )
                $zone['endTime'] = "23:59:59.999";
            if ( ! $this->validateTime( $zone['endTime'], 
                                        $attribute."[$i].endTime" ) )
                    return false;
            
            if ( !isset($zone['beginDay']) )
                $zone['beginDay'] = 0;
            if ( !is_int($zone['beginDay']) || $zone['beginDay']<0 || $zone['beginDay']>6 ){
                $this->addMediaError( $attribute."[$i].beginDay", 
                        "should be between 0 and 6" );
                return false;
            }            
            if ( !isset($zone['endDay']) )
                $zone['endDay'] = 0;
            if ( !is_int($zone['endDay']) || $zone['endDay']<0 || $zone['endDay']>6 ){
                $this->addMediaError( $attribute."[$i].endDay", 
                        "should be between 0 and 6" );
                return false;
            }
                  
            $i++;
        }
        return true;
    }
    public function validateSchedule( &$schedule, $attribute ){        
        if( !is_array($schedule) ) {
            $this->addMediaError( $attribute, 
                    "should be an object");
            return false;
        }  
        if ( isset( $schedule['beginDate'] ) 
                && !$this->validateDateTime( $schedule['beginDate'], 
                                             $attribute.".beginDate") ){
            return false;
        }
        if ( isset( $schedule['endDate'] ) 
                && !$this->validateDateTime( $schedule['endDate'], 
                                             $attribute.".endDate") ){
            return false;
        }
        
        if ( isset( $schedule['validityZones'] ) 
                && !$this->validateZone( $schedule['validityZones'], 
                                        $attribute.".validityZones") )
            return false;
        if ( isset( $schedule['exclusionZones'] ) 
                && !$this->validateZone( $schedule['exclusionZones'], 
                                        $attribute.".exclusionZones" ) )
            return false;
       
        
        return true;
    }
    public function validateQueryValue( $name, &$value, $attribute, $options = null ){
        
        if ( $name === 'apps' ) {
            return true;
        }
        
        if ( !isset( $options[$name] ) ) {
            $this->addMediaError( $attribute, "missing" );
            return false;
        }
        $option = $options[$name];
        if ( $option['type']==='number' || $option['type']==='slider' ) {
            if ( !is_numeric( $value ) ){
                $this->addMediaError( $attribute, "should be a number" );                        
                return false;
            }

            if ( isset($option['min']) && floatval($value)<floatval($option['min']) ) {
                Yii::log("Value $value < ".$option['min'].", set to min", 'info', 'spx.validation');
                $value = floatval($option['min']);
            }

            if ( isset($option['max']) && floatval($value)>floatval($option['max']) ) {
                Yii::log("Value $value > ".$option['max'].", set to max", 'info', 'spx.validation');
                $value = floatval($option['max']);
            }
        }
        if ( $option['type']==='date' ) {
            return $this->validateDate( $value, $attribute);
        }
        if ( $option['type']==='time' ) {
            return  $this->validateTime( $value, $attribute);
        }
        if ( $option['type']==='datetime' ) {
            return $this->validateDateTime( $value, $attribute);
        }
        if ( $option['type']==='string' ) {
            return $this->validateText( $value, $attribute);
        }
        if ( $option['type']==='boolean' ) {
            return $this->validateBoolean( $value, $attribute);
        }
        if ( $option['type']==='color' ) {
            return $this->validateColor( $value, $attribute);
        }
        if ( $option['type']==='choice' ) {
            $found = false;
            if ( !isset( $option['select'] ) || !is_array( $option['select'] ) || count($option['select']) == 0 ) {
                $this->addMediaError(  $attribute.".select", "missing" );
                return false;
            }
            foreach( $option['select'] as $idx => $sel ){
                if ( !isset( $sel['value'] ) ) {
                    $this->addMediaError(  $attribute.".select[$idx].value", "missing" );
                    return false;
                }
                if ( $sel['value'] == $value ){
                    $found = true;
                    break;
                }
                if ( isset( $sel['name'] ) && $sel['name'] == $value ){
                    // backward compatibility
                    $value = $sel['value'];
                    $found = true;
                    break;
                }
            }
            if ( !$found ) {
                $value = $option['select'][0]['value'];                
            }
        }
        return true;
    }
    public function validateQuery( &$query, $attribute, $options = null ){
        if( !is_array($query) ) {
            $this->addMediaError( $attribute, 
                    "should be an object" );
            return false;
        }
        $allOK = true;
        if ( $options ) {
            // query must in options                            
            foreach ( $query as $name=>&$value ){
                $attribute2 = $attribute.".".$name;
                $allOK &= $this->validateQueryValue( $name, $value, $attribute2, $options );
            }
        }
        return $allOK;
    }
    public function validateMedia( &$medias, &$childrens, $attribute, $allowPlaylist=true, $allowPlayout=false ){
        
        if ( !is_array( $medias ) ){//make sure $_media is populated
            $this->addMediaError( $attribute, "should be an array" );
            return false;
        }
        
        $i=0;
        $allOK = true;
        foreach( $medias as &$media ) {
            $allOK &= $this->validateSingleMedia( $media, $childrens, $attribute.'['.$i.']', $allowPlaylist, $allowPlayout ) !== false;                
            $i++;
        }
        return $allOK;
    }
    public function validateSingleMedia( &$media, &$childrens, $attribute, $allowPlaylist=true, $allowPlayout=false ){
        
        if ( !isset($media['id']) ){
            $this->addMediaError( $attribute.".id", "missing" );
            return false;
        }
        if ( !isset($media['type']) ){
            $this->addMediaError( $attribute.".type", "missing" );
            return false;
        }
        $id = $media['id'];
        // id must exist and be a media\
        
        $widgetId = false;
        $widget = null;
        $model = Resource::loadModel( $media['type'], $id);        
        
        if ( !$model ){
            $this->addMediaError( $attribute.".id", 
                    "missing" );
            return false;
        }
        if ( $media['type']==='apps' ){
            if ( !isset($media['widgetId']) ){
                $this->addMediaError( $attribute.".widgetId", "missing" );
                return false;
            }
            $widgetId = $media['widgetId'];
                       
            $widget = $model->getWidget( $widgetId );
            if ( !$widget ){
                $this->addMediaError( $attribute.".widgetId", 
                        "missing" );
                return false;
            }
        } 
        // update the href for the player
        if ( $widget )
            $href = $widget['href'];
        else 
            $href = $model->href;
        $query = "";
        if ( isset($media['query']) ){
            if ( $media['type']==='apps' ) {                
                if ( !$this->validateQuery( $media['query'], $attribute.".query", $widget['options'] ) )
                    return false;
                $sep = "?";
                foreach ($media['query'] as $q=>$v ){
                    if ( $v === true ){
                        $v = 'true';
                    }
                    if ( $v === false ){
                        $v = 'false';
                    }
                    $query .= $sep .  rawurlencode($q) ."=".  rawurlencode($v);
                    $sep = "&";
                }
                if ( empty($media['query']) )
                    unset( $media['query'] );
            } else {
                // we ignore and remove the query in this case
                unset( $media['query'] );
            }
        }
        
        $media['href'] = $href . $query;
        $media['manage'] = $model->manage;
        if ( $media['type']==='media' || $media['type']==='playlist' || $media['type']==='project' ){
            $media['mediaDur'] = floatval($model->dur);
        }
        // Update the dependency
        $childrens[] = $model->resource;
        
        if ( isset($media['dur'])){
            $value = $media['dur'];
            if( !is_numeric($value) ) {
                $this->addMediaError( $attribute.".dur", 
                        "should be a number" );
                return false;
            }              
            if ( floatval($value) == 0 ){
                unset( $media['dur'] );
            }
        }
        
        if ( isset($media['schedule']) ){
            if ( !$this->validateSchedule( $media['schedule'], $attribute.".schedule" ) )
                return false;
            if ( empty($media['schedule']) )
                unset( $media['schedule'] );
        }
        return $model;
    }
    public function validatePlaylist( &$playlist, &$childrens, $attribute, $allowPlaylist=true, $allowPlayout=false ){
        $ret = true;
        if ( isset($playlist['media']) ){
            $ret = $this->validateMedia( $playlist['media'], $childrens, 
                                $attribute.".media", $allowPlaylist, $allowPlayout);            
        } else {
            $playlist['media'] = array();
        }
        $default = array (
            'defaultDur' => 5, 
            'looping' => false,
            'shuffle' => false,
            'transition' => null
        );
        
        if ( isset($playlist['options']) ){
            if ( !is_array($playlist['options']) ){
                $this->addMediaError( $attribute.".option", "wrong format" );
                return false;
            }                        
            $playlist['options'] = array_merge( $default, $playlist['options'] );
            $options = $playlist['options'];
            if ( isset($options['transition']) )
                $ret &= $this->validateTransition( $options['transition'], 
                                    $attribute.".options.transition");

        } else {
            $playlist['options'] = $default;
        }
        
        return $ret;
    }    
    public function validateTransition( $transition, $attribute ){

        if ( $transition=== null )
            return true;
        if ( !isset($transition['ctor']) ){
            $this->addMediaError( $attribute."ctor", "missing");            
            return false;
        }
        if ( !isset($transition['args']) ){
            $this->addMediaError( $attribute."args", "missing" );            
            return false;
        }
        // more validation
        
        return true;
    }
}
