<?php

/**

 */
class Playout extends Resource
{
    protected $playoutFile = "playout.js";
    protected $renderFile = "playout.svg";
    
    protected $_optionsAttr = array( 'dur', 'locked' );
    protected $_options = null;
    protected $_options_ok = true;
    protected $_layers = null;
    protected $_layers_modified = false;
    protected $_layers_fixed = false;
    
    protected $_old_layers = null;
    
    protected $width=1920, $height=1080;
    
    var $sourceId = null;
    var $variablesToDelete = array();
    
	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		$rules = parent::rules();
        $rules[] = array('options', 'validateOptions');
        $rules[] = array('layers', 'validateLayers');
        $rules[] = array('sourceId', 'validateSourceId', 'on'=>'insert');
		return $rules;
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'id',
			'name' => 'name',
			'dur' => 'options.dur',
            'locked' => 'options.locked',
			'layers' => 'layers',
		);
	}
    public function init(){
        parent::init();
        $this->type = "playout"; 
    }
    public function defaultScope(){
        return array(
            'condition'=>"type='playout'",
        );
    }
    public function fixPlayout( $short = false ) {
        if ( $this->_layers_fixed )
            return;
        // fix the layer description for the read only data
        foreach( $this->_layers as $idx=>&$layer ) {
            unset( $layer['ctor'] );       
            if ( $idx === 0 ){
                $this->width = $layer['args']['width'];
                $this->height = $layer['args']['height'];
            }
            if ( $short && isset( $layer['args']['href'] ) ){
                unset($layer['args']['href']);
            }
            if ( $layer['type']==='apps' ) {
                if ( !isset( $layer['data'] ) ){
                    unset( $this->_layers[$idx] );
                    continue;
                }
                $id = $layer['data']['id'];
                $model = Apps::model()->findByPk($id);
                if ( !$model ){
                    // we have a problem the apps is gone, 
                    // remove this from the list
                    unset( $this->_layers[$idx] );
                    continue;
                }   
                if ( !$short ){
                    // add the read only info for the apps
                    $layer['data']['apps'] = $model->getDesc( false );
                }
                if ( !isset( $layer['data']['name'] ) ){
                    $layer['data']['name'] = $model->name;
                }
            } else if ( $layer['type']==='playlist' ) {
                if ( !isset( $layer['data'] ) )
                    $layer['data'] = array();
                if ( !isset( $layer['data']['media'] ) )
                    $layer['data']['media'] = array();                
            } 
            if ( $this->uuid !== null && isset( $layer['data']['name'] ) ) {
                $layer['data']['name'] = I18N::t( $layer['data']['name'] );                    
            } 
        }
        $this->_layers_fixed = true;
    }
    public function loadPlayout( $force=false, $short = false ) {
        $path = $this->getFolder();
        if ( !$path )
            return false;
        if ( !file_exists( $path . $this->playoutFile ))
            return false;
        $playout = file_get_contents( $path . $this->playoutFile );
        if ( empty($playout) )
            return false;
        $items = explode("\n", $playout);
        
        foreach ($items as $item){
            $settings = explode("=", $item, 2);
            if ( count( $settings )!=2 )
                continue;
            $json = substr( trim($settings[1]), 0, -1 );
            if ( trim( $settings[0] )=="options" && ( $force || !$this->_options )){
                $this->_options = json_decode( $json, true );                
            }
            if ( trim( $settings[0] )=="layers" && ( $force || !$this->_layers )){
                $this->_layers = json_decode( $json, true );           
                $this->fixPlayout( $short );
                $this->_old_layers = $this->_layers;
                $this->_layers_modified = false;                
            }
        }        
        
        return true;
    }
    public function savePlayout( $path ) {
        $playout = "";
        if ( !empty(  $this->_options ) )
            $playout .= "options = ".json_encode( $this->_options ). ";\n";
        else
            $playout .= "options = {};\n";
        $saved_layers = array();
        foreach( $this->_layers as $layer ){
            if ( $layer['type']==='apps' ) {
                unset( $layer['data']['apps'] );                
            } else if ( $layer['type']==='playlist' && isset($layer['data']) && isset($layer['data']['media']) ){                                
                foreach( $layer['data']['media'] as &$media ) {
                    if ( $media['type'] === 'apps' ){
                        unset( $media['apps'] );
                    }
                }
            }
            $saved_layers[] = $layer;
        }
        $playout .= "layers = ".json_encode( $saved_layers ). ";\n";
        $filepath = $path . $this->playoutFile;
        if ( !Resource::saveAtomic( $filepath, $playout) ){
            Yii::log("Save Playout failed: cannot write $filepath", 'warning', 'spx.playout');
            $this->addError( "", "save failed" );
            return false;
        }
        
        $renderFilepath = $path . $this->renderFile;
        if ( !@copy( GlobalSettings::$systemMedia.$this->renderFile, $renderFilepath ) ){            
            Yii::log("Save Playout failed: cannot write $renderFilepath", 'warning', 'spx.playout');
            if ( !file_exists( GlobalSettings::$systemMedia.$this->renderFile ) ){
                Yii::log("File not found: ".GlobalSettings::$systemMedia.$this->renderFile, 'warning', 'spx.playout');
            }
            if ( !file_exists( $path ) ){
                Yii::log("Path do not exist: ".$path, 'warning', 'spx.playout');
            }
            if ( file_exists( $renderFilepath ) ){
                Yii::log("Destination file exists: ".$renderFilepath, 'warning', 'spx.playout');
            }
            $this->addError( "", "save failed" );
            return false;
        }
        $svg = file_get_contents( $renderFilepath );
        $svgFixed = preg_replace( '/viewBox="0 0 \d+ \d+"/', 'viewBox="0 0 '.$this->width.' '.$this->height.'"', $svg );
        if ( !Resource::saveAtomic( $renderFilepath, $svgFixed) ){
            Yii::log("Save Playout failed: cannot update $renderFilepath", 'warning', 'spx.playout');
            $this->addError( "", "save failed" );
            return false;
        }
        $this->fixPlayout();
        
        return true;
    }
    
    public function getOptions( $short = false ) {
        if ( $this->_options === null )
            $this->loadPlayout( false, $short );        
        if ( $this->_options === null ){
            $this->setOptions( array() );            
        }
        return $this->_options;
    }
    public function setOptions( $options ) {
        $default = array (
                'dur' => null, 
                'locked' => false, 
            );
        if ( is_array( $options ) ){
            $this->_options = array_merge( $default, $options );  
            $this->_options_ok = true;
        } else {
            $this->_options_ok = false;      
            $this->_options = $default;
        }
    }
    public function __get( $name ) {
        if ( in_array( $name, $this->_optionsAttr ) ){
            $options = $this->getOptions();
            return $options[$name];
        } else
            return parent::__get( $name );            
    }
    public function __set( $name, $value ) {
        if ( in_array( $name, $this->_optionsAttr ) ){
            $options = $this->getOptions();
            $options[$name] = $value;
            
            $this->setOptions( $options );
        } else
            return parent::__set( $name, $value );            
    }
    
    public function getLayers( $short = false ){
        if ( $this->_layers === null ){
            $this->loadPlayout( false, $short );         
        }
        if ( $this->_layers === null ){
            $this->_layers = array();
        }
        return $this->_layers;
    }
    public function setLayers( $layer ){
        $this->_layers = $layer;  
        $this->_layers_modified = true;
        $this->_layers_fixed = false;
    }
    public function addLayerError( $attribute, $desc ){
        $this->addError( "layers", $attribute.": ".$desc );
    }
    public function validateArgs( &$args, $attribute ){
        if( !is_array($args) ) {
            $this->addLayerError( $attribute, 
                    "should be an object");
            return false;
        }
        $mendatory = array( 'top', 'left', 'width', 'height' );
        foreach ($mendatory as $m ){
            if ( !isset($args[$m]) ){
                $this->addLayerError( $attribute.".".$m, "missing" );
                return false;
            }
            if ( !is_numeric ($args[$m]) ){
                $this->addLayerError( $attribute.".".$m, "must be a number" );
                return false;
            }
            $args[$m] = floatval( $args[$m] );
        }
        return true;
    }
    public function validateQuery( $query, $attribute ){
        if( !is_array($query) ) {
            $this->addLayerError( $attribute.".query", 
                    "should be an object" );
            return false;
        }
        return true;
    }
    public function validateOptions( $attribute ){
        if ( $attribute!="options" ){
            $this->addError( $attribute, "should be options" );
            return false;
        }
        // need to have a proper validation for teh options
        if ( !$this->_options_ok ) {            
            $this->addError( $attribute, "wrong format" );
            return false;
        }
    }
    public function validateLayers( $attribute ){
        if ( $attribute!="layers" ){
            $this->addError( $attribute, "should be layers" );
            return false;
        }
        if ( !is_array( $this->getLayers() ) ){//make sure $_layers is populated
            $this->addError( $attribute, "should be an array" );
            return false;
        }
        $childrens = array();
        
        $val = new Validator( $this, "layers" );
        
        foreach( $this->_layers as $idx => &$layer ) {
            $attribute = 'layers['.$idx.']';
            if ( !isset($layer['type']) ){
                $this->addLayerError( $attribute.".type", "missing" );
                return false;
            }
            if ( !isset($layer['args']) ){
                $this->addLayerError( $attribute.".args", "missing" );
                return false;
            }
            if ( !$this->validateArgs( $layer['args'], $attribute.".args" ) )
                return false;  
            if ( !isset($layer['data']) ){
                $this->addLayerError( $attribute.".data", "missing" );
                return false;
            }
            if ( $idx === 0 && $layer['type'] !== 'playlist' ) {
                $this->addLayerError( $attribute.".type", "should be a playlist background" );
                return false;
            }
            if ( $idx === 0 ){
                $this->width = $layer['args']['width'];
                $this->height = $layer['args']['height'];
            }
            if ( $layer['type']==='playlist' ) {
                if ( !isset( $layer['args']['mediaFit']) ) {
                    $layer['args']['mediaFit'] = 'slice';                
                }
                
                $playlist = &$layer['data'];
                if ( $this->type === 'template' && $idx != 0 ){
                    if ( isset($playlist['media']) && count($playlist['media']) ){
                        $this->addLayerError( $attribute.".media", "not allowed" );
                        return false;
                    }
                }
                // additional check on the playlist content
                if ( !$val->validatePlaylist( $playlist, $childrens, $attribute.".data", false ) )
                    return false;
                if ( isset($playlist['media']) ){
                    if ( count( $playlist['media'] )==1 ) {
                        $layer['args']['href'] = $playlist['media'][0]['href'];
                        $layer['ctor'] = "media";
                    }else {                        
                        $layer['ctor'] = "slideshow";
                    }
                }                                
            } else if ( $layer['type']==='apps' ) {
                $layer['ctor'] = "media";
                $apps = &$layer['data'];
                $attribute .= ".data";
                if ( !isset($apps['id']) ){
                    $this->addLayerError( $attribute.".id", "missing" );
                    return false;
                }
                if ( !isset($apps['widgetId']) ){
                    $this->addLayerError( $attribute.".widgetId", "missing" );
                    return false;
                }
                $id = $apps['id'];
                $widgetId = $apps['widgetId'];
                // id must exist and be a widget
                $model = Apps::model()->findByPk($id);
                if ( !$model ){
                    $this->addLayerError( $attribute.".id", "not found" );
                    return false;
                }
                $childrens[] = $model->resource;
                
                $widget = $model->getWidget( $widgetId );
                if ( !$widget ){
                    $this->addLayerError( $attribute.".widgetId", "not found" );
                    return false;
                }
                $query = "";
                $sep = "?";
                if ( isset($apps['query']) ){
                    if ( !$val->validateQuery( $apps['query'], $attribute.".query", $widget['options'] ) )
                        return false;                    
                    foreach ($apps['query'] as $q=>$v ){
                        if ( $v === true ){
                            $v = 'true';
                        }
                        if ( $v === false ){
                            $v = 'false';
                        }
                        $query .= $sep .  rawurlencode($q) ."=".  rawurlencode($v);
                        $sep = "&";
                    }
                    if ( empty($apps['query']) )
                        unset( $apps['query'] );
                }
                $layer['args']['href'] = $widget['href'] . $query;
            } else {
                $this->addLayerError( $attribute.".type", "not valid" );
                return false;
            }
            
        }
        // Update the AR relations
        $this->children = $childrens;
        
        return true;
    }
    public function validateSourceId( $attribute ){
        if ( $attribute!="sourceId" ){
            $this->addError( $attribute, "should be sourceId" );
            return false;
        }
        $id = $this->$attribute;
        if ( empty($id) ){
            return;
        }
        if ( $this->type === 'playout' ){
            $model = Playout::model()->findByPk($id);
            if ( !$model ){
                 $this->addError( $attribute, "not found" );
                 return false;
            }
        } else {
            $model = Template::model()->findByPk($id);
            if ( !$model ){
                 $this->addError( $attribute, "not found" );
                 return false;
            }
        }
        $this->setOptions( $model->getOptions() );
        $this->setLayers( $model->getLayers() );
        $this->validateLayers( 'layers' ); // update the relations
    }
    
    public function removeChild( $childId, $delayPreview = false ) {
        $this->delayPreview = $delayPreview;
        $old_layers = $this->getLayers();
        $new_layers = array();
        foreach( $old_layers as $layer ){
            if ( $layer['type']==='playlist' ){
                $new_media = array();
                foreach( $layer['data']['media'] as $media ){ 
                    if( $media['id']!=$childId )
                        $new_media[] = $media;
                }
                $layer['data']['media'] = $new_media;
            } else {
                // apps
                if ( isset( $layer['data'] ) && isset($layer['data']['id']) && $layer['data']['id'] == $childId ){
                    continue; // remove this layer
                }
            }
            $new_layers[] = $layer;
        }
        $this->setLayers( $new_layers  );
        
        if ( $this->save() ) {
            return true;
        } else {                   
            return false;
        }
    }
    public function rollback() {        
        if ( $this->_layers_modified && $this->_old_layers) {
            $this->setLayers($this->_old_layers);
            if (!$this->savePlayout( $this->getFolder() ) )
                throw new Exception( "rollback failed" );
        }
        parent::rollback( );
    }
    
    public function gethref() {
        return parent::getHref() . $this->renderFile;
    }
    
    public function getListDesc( $preview=true ) {
        $description = array(
            'id' => $this->id,
            'name' => $this->name,
            'type' => $this->type,
            'href' => $this->href,            
            'manage' => $this->manage,
            'modified' => $this->modified,
            'keywords' => $this->keywords
        );        
        if (  $preview ) {
            $previews = $this->previews;
            if ( count( $previews ) )
                $description['previews'] = $previews;
        }
        return $description;
    }
    public function getDesc( $parents = true, $short = false ) {
        $description = $this->getListDesc( !$short ); 
        $description['options']= $this->getOptions( $short );
        $description['layers']= $this->getLayers( $short );
        if ( $parents ) {
            foreach ( $this->parents as $parent ){
                $description['parents'][] = $parent->getListDesc( false );
            }
        }
        return $description;
    }
    public function getPreviews() {
        $where = $this->getFolder() ;
        if ( !file_exists( $where . "previews/" ) ){
            $this->loadPlayout( false, true ); 
            // previews do not existe, we generate them now
            $this->generatePreviews( $where );
        }
        return parent::getPreviews();
    }
    public function generatePreviews( $where ) {
        $path = $where."previews/";
        if ( $this->delayPreview ){
            if (file_exists($path) ){
                GlobalSettings::rrmdir( $path, true );
            }
            return;
        }
        $rootDir = GlobalSettings::$fileBase;
        
        // clear up the preview folder and create it
        
        if (file_exists($path) )
            GlobalSettings::rrmdir( $path, false );            
        else
            mkdir( $path );
        
        
        $exe="cd /srv/raperca && raperca";
        if( PHP_OS == "WINNT" ){
            $exe = "C:\\SpinetiX\\sources\\genesis\\trunk\\user\\hmd\\bin_d\\raperca.exe";
            if ( !file_exists( $exe ) ){
                $exe = "\"C:\\Program Files (x86)\\SpinetiX\\Elementi\\Elementi\\bin\\raperca.exe\"";
            }
        } else if (file_exists('/opt/sysroot/usr/local/lib/liblive555.so.0')) {
            // this is a Bonsai/Sakura system hacked as Ikebana, need special LD_LIBRARY_PATH
            $exe = "cd /srv/raperca && LD_LIBRARY_PATH=/opt/sysroot/lib:/opt/sysroot/usr/lib:/opt/sysroot/usr/local/lib raperca";
        }
        
        $file = $this->renderFile;
              
        $size = array( $this->width/2, $this->height/2 );
        $width = $size[0];
        $height = $size[1];
        $dimension = "{$width}x{$height}";
        
        $cmd = "$exe -preview -s " . escapeshellarg($dimension) . " -pm 1 -d " . escapeshellarg($rootDir) . " " . escapeshellarg($this->gethref()) . " 2>&1";
        Yii::log($cmd, 'info', 'spx.preview');
        $output = array();
        $ret = -1;
        exec( $cmd, $output, $ret );
        if ( $ret !== 0 && PHP_OS == "WINNT" ) {
            $img = imagecreatetruecolor($width/2, $height/2);
            $textcolor = imagecolorallocate($img, 255, 255, 255);
            imagestring( $img, 5, 10, $height/4, $this->name, $textcolor);            
            imagepng( $img, $path."preview_$dimension.png");
            $ret = 0;
        }
        if ( $ret==0 ) {
            // move to the preview folder
            @rename("$where.preview.$file.png", $path."preview_$dimension.png");
            // create smaller images
            while ( $size[0]>100 && $size[1]>50 ){
                $size[0] = round( $size[0]/2 );
                $size[1] = round( $size[1]/2 );
                $this->previewsImage( $size[0]."x".$size[1], $path, $path, "preview_$dimension.png" );            
            }                     
        } 
        
        // default previews
        @copy( GlobalSettings::$systemMedia."playout.png", $path."preview_98x58_0_default.png" );
        
    }

    public function recover( $file = null ){
        $path = $this->getFolder();
        $playoutFilename = $path . $this->playoutFile;
        $recover = parent::recover($playoutFilename);     
        if ( !$recover ){
            $renderFilepath = $path . $this->renderFile;
            if ( !file_exists($renderFilepath) || filesize($renderFilepath) == 0 ){
                $this->loadPlayout( true, true );
                $this->savePlayout( $path );
            }            
        }
        return $recover;
    }
    public function saveFiles(){
        $path = $this->getFolder();
        if ( !$path ){
            Yii::log("Path of Playout not found: ".print_r( $this->desc, true), 'warning', 'spx.playout');
            $this->addError( "", "not found" );
            return false;
        }
        if ( $this->_layers_modified ){
            if ( !$this->savePlayout( $path ) ){
                $this->addError( "", "save failed" );
                return false;
            }
            $this->generatePreviews( $path );
            $this->etag = $this->computeEtag( $path . $this->playoutFile );
        }        
        return true;
    }
    public function beforeSave(){
        if ( !$this->isNewRecord && !$this->saveFiles() ){
            return false;
        }
        return parent::beforeSave();
    }
    public function afterSave(){ 
        parent::afterSave();
        
        if ( $this->isNewRecord && !$this->saveFiles() ){
            throw new Exception( "save failed" );
        }        
    }
    public function beforeDelete() {
        $this->variablesToDelete = array();
        $this->loadPlayout( false, true ); 
        if ( $this->_layers ){
            foreach( $this->_layers as $layer ) {
                if ( isset($layer['type']) && $layer['type']==='apps' ) {
                    $apps = $layer['data'];
                    if ( isset($apps['query']) && isset($apps['query']['variable']) ){                                        
                        $this->variablesToDelete[] = $apps['query']['variable'];
                    }
                }
            }
        }
        // we cannot access the WebStorage API from here, we are not admin
        
        return parent::beforeDelete();
    }
    
    /**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('id',$this->id);
		$criteria->compare('name',$this->name,true);
		$criteria->compare('defaultDur',$this->defaultDur);
		$criteria->compare('transition',$this->transition,true);
		$criteria->compare('looping',$this->looping);
		$criteria->compare('shuffle',$this->shuffle);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return Playout the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}
}

